﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Font
{
    using System;
    using System.Collections.Generic;
    using System.Xml;

    public class CharFilter : HandlerBase
    {
        private const string TagCharacterFilter = "letter-list";
        private const string TagPassCharacters = "letter";
        private const string TagBody = "body";
        private const string TagCode = "code";
        private const string TagSp = "sp";

        private const string AttribVersion = "version";
        private const string XflVersion = "1.0";

        private bool isNoFilter;

        private OutputGlyphList outputGlyphList = new OutputGlyphList();

        private int tagLevel;
        private bool isInCode;
        private bool isInPass;
        private bool isInBody;

        public CharFilter()
        {
            this.tagLevel = 0;
            this.isInCode = false;
            this.isInPass = false;
            this.isInBody = false;
            this.isNoFilter = true;
        }

        public bool IsFiltered(uint c)
        {
            if (this.isNoFilter)
            {
                return false;
            }
            else
            {
                bool value;
                var found = this.outputGlyphList.TryGetValue(c, out value);

                return !found || !value;
            }
        }

        // デバッグ用
        public void DumpFilter()
        {
            foreach (var i in this.outputGlyphList)
            {
                Rpt._RPT2("FILTER: DUMP 0x{0:X4} -> {1}\n", i.Key, i.Value ? "true" : "false");
            }
        }

        // 文字フィルタ定義ファイルを読み込む
        public void Load(string filterFilePath, bool useDtdValidation)
        {
            try
            {
                using (var reader = CreateReader(filterFilePath, useDtdValidation))
                {
                    Parse(reader);
                }
            }
            catch (HandlerBaseException)
            {
                if (Error != null)
                {
                    throw GlCm.ErrMsg(ErrorType.Xml, Strings.IDS_ERR_XERCESC_MSG, Error);
                }
                else
                {
                    throw GlCm.ErrMsg(ErrorType.Xml, Strings.IDS_ERR_INVALID_FILTER_FILE);
                }
            }
            catch (GeneralException ex)
            {
                throw GlCm.ErrMsg(ErrorType.Xml, Strings.IDS_ERR_XERCESC_MSG, ex.GetMsg());
            }
            catch (XmlException ex)
            {
                Rpt._RPT1("XML Error: {0}\n", ex.Message);
                throw GlCm.ErrMsg(ErrorType.Xml, Strings.IDS_ERR_XERCESC_MSG, ex.Message);
            }

            // すべての文字がフィルタされ、空フォントになってしまう。
            if (this.outputGlyphList.Keys.Count <= 0)
            {
                throw GlCm.ErrMsg(ErrorType.Xml, Strings.IDS_ERR_INVALID_FILTER_EMPTY);
            }

            this.RegistPassChar(' ');
            this.isNoFilter = false;
        }

        // order で出力されない文字をフィルタに追加
        public void SetOrder(GlyphOrder order)
        {
            int w = order.GetHNum();
            int h = order.GetVNum();

            if (this.isNoFilter)
            {
                // order で出力される文字を追加
                for (int y = 0; y < h; y++)
                {
                    for (int x = 0; x < w; x++)
                    {
                        var c = order.GetCharCode(x, y);

                        if (c != Runtime.RtConsts.InvalidCharCode)
                        {
                            this.RegistPassChar(c);
                        }
                    }
                }

                // order が空の場合は NoFilter 継続
                if (this.outputGlyphList.Count == 0)
                {
                    return;
                }

                this.isNoFilter = false;
            }
            else if (order.IsValid())
            {
                var newOutput = new OutputGlyphList();

                // order で出力される文字をチェック
                for (int y = 0; y < h; y++)
                {
                    for (int x = 0; x < w; x++)
                    {
                        var c = order.GetCharCode(x, y);

                        if (c != Runtime.RtConsts.InvalidCharCode)
                        {
                            if (this.outputGlyphList.ContainsKey(c))
                            {
                                newOutput[c] = true;
                            }
                        }
                    }
                }

                // 出力するものだけ残す
                this.outputGlyphList = newOutput;
            }
        }

        public void CheckEqual(FontData font)
        {
            GlyphList list = font.GetGlyphList();

            foreach (var pair in this.outputGlyphList)
            {
                if (pair.Value)
                {
                    if (list.GetByCode(pair.Key) == null)
                    {
                        // 「出力する」なのにグリフが無い
                        ProgressControl.Warning(
                            Strings.IDS_WARN_FONT_FILTER_MISMATCH,
                            (char)pair.Key,
                            pair.Key);
                    }
                }
            }
        }

        // ---------------------------------------------------------------------------
        //  PParseHandlers: Overrides of the SAX DocumentHandler interface
        // ---------------------------------------------------------------------------
        protected override void StartElement(string name, AttributeList attributes)
        {
            Rpt._RPT2("XML: {0} ({1})\n", name, this.tagLevel);

            if (name == TagCode)
            {
                if (this.isInPass)
                {
                    this.isInCode = true;
                }
                else
                {
                    throw GlCm.ErrMsg(ErrorType.Xml, Strings.IDS_ERR_TAG_MUST_BE_IN, TagCode, TagPassCharacters);
                }
            }
            else if (name == TagSp)
            {
                if (this.isInPass)
                {
                    this.RegistPassChar(0x0020);
                }
                else
                {
                    throw GlCm.ErrMsg(ErrorType.Xml, Strings.IDS_ERR_TAG_MUST_BE_IN, TagSp, TagPassCharacters);
                }
            }
            else if (name == TagPassCharacters)
            {
                if (this.isInBody)
                {
                    this.isInPass = true;
                }
                else
                {
                    throw GlCm.ErrMsg(ErrorType.Xml, Strings.IDS_ERR_TAG_MUST_BE_IN, TagPassCharacters, TagBody);
                }
            }
            else if (name == TagBody)
            {
                if (this.tagLevel == 1)
                {
                    this.isInBody = true;
                }
                else
                {
                    throw GlCm.ErrMsg(ErrorType.Xml, Strings.IDS_ERR_TAG_MUST_BE_IN, TagBody, TagCharacterFilter);
                }
            }
            else if (name == TagCharacterFilter)
            {
                string version = attributes.GetValue(AttribVersion);

                if (version == null || version != XflVersion)
                {
                    if (Error == null)
                    {
                        string buf;
                        string format = Strings.IDS_ERR_VERSION_MISMATCH;

                        buf = string.Format(
                            format,
                            TagCharacterFilter,
                            version,
                            XflVersion);

                        Error = buf;
                    }
                }
            }

            this.tagLevel++;
        }

        protected override void Characters(string chars, int length)
        {
            if (this.isInPass)
            {
                if (this.isInCode)
                {
                    uint val = 0;

                    uint.TryParse(chars, out val);
                    this.RegistPassChar(val);

                    this.isInCode = false;
                }
                else
                {
                    for (int i = 0; i < length; ++i)
                    {
                        if (!IsWhitespace(chars[i]))
                        {
                            this.RegistPassChar(chars[i]);
                        }
                    }
                }
            }
        }

        protected override void EndElement(string name)
        {
            this.tagLevel--;

            Rpt._RPT2("XML: {0} ({1})\n", name, this.tagLevel);
            if (name == TagCode)
            {
                this.isInCode = false;
            }
            else if (name == TagPassCharacters)
            {
                this.isInPass = false;
            }
            else if (name == TagBody)
            {
                this.isInBody = false;
            }
        }

        private void RegistPassChar(uint c)
        {
            //// _RPT1(_CRT_WARN, "FILTER: regist 0x%04X\n", c);
            this.outputGlyphList[c] = true;
        }

        // 文字コード → 出力する
        private class OutputGlyphList : Dictionary<uint, bool>
        {
        }
    }
}
