﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.Font
{
    using System;
    using System.Collections;
    using System.IO;
    using System.Reflection;
    using System.Runtime.InteropServices;

    public class ByteOrderBinaryWriter : IDisposable
    {
        private BinaryWriter bw;
        private byte[] buf;
        private bool isLittleEndian;

        public ByteOrderBinaryWriter(Stream output, bool isLittleEndian)
        {
            if (output == null)
            {
                throw new ArgumentNullException("output");
            }

            if (!output.CanWrite)
            {
                throw new ArgumentException("output not CanWrite");
            }

            this.bw = new BinaryWriter(output);
            this.buf = new byte[8];
            this.isLittleEndian = isLittleEndian;
        }

        public Stream BaseStream
        {
            get { return this.bw.BaseStream; }
        }

        public void Dispose()
        {
            this.bw.Close();
        }

        public void Flush()
        {
            this.bw.Flush();
        }

        public long Seek(int offset, SeekOrigin origin)
        {
            return this.bw.Seek(offset, origin);
        }

        public void Write(bool value)
        {
            this.bw.Write(value);
        }

        public void Write(byte value)
        {
            this.bw.Write(value);
        }

        public void Write(sbyte value)
        {
            this.bw.Write(value);
        }

        public void Write(byte[] buffer)
        {
            if (buffer == null)
            {
                throw new ArgumentNullException("buffer");
            }

            this.bw.Write(buffer, 0, buffer.Length);
        }

        public void Write(byte[] buffer, int index, int count)
        {
            this.bw.Write(buffer, index, count);
        }

        public void Write(short value)
        {
            this.Write((ushort)value);
        }

        public void Write(ushort value)
        {
            if (this.isLittleEndian)
            {
                this.bw.Write(value);
            }
            else
            {
                this.buf[0] = (byte)(value >> 8);
                this.buf[1] = (byte)value;
                this.bw.Write(this.buf, 0, 2);
            }
        }

        public void Write(int value)
        {
            this.Write((uint)value);
        }

        public void Write(uint value)
        {
            if (this.isLittleEndian)
            {
                this.bw.Write(value);
            }
            else
            {
                this.buf[0] = (byte)(value >> 24);
                this.buf[1] = (byte)(value >> 16);
                this.buf[2] = (byte)(value >> 8);
                this.buf[3] = (byte)value;
                this.bw.Write(this.buf, 0, 4);
            }
        }

        public void Write(long value)
        {
            this.Write((ulong)value);
        }

        public void Write(ulong value)
        {
            if (this.isLittleEndian)
            {
                this.bw.Write(value);
            }
            else
            {
                this.buf[0] = (byte)(value >> 56);
                this.buf[1] = (byte)(value >> 48);
                this.buf[2] = (byte)(value >> 40);
                this.buf[3] = (byte)(value >> 32);
                this.buf[4] = (byte)(value >> 24);
                this.buf[5] = (byte)(value >> 16);
                this.buf[6] = (byte)(value >> 8);
                this.buf[7] = (byte)value;
                this.bw.Write(this.buf, 0, 8);
            }
        }

        public void Write24Bits(uint value)
        {
            if (this.isLittleEndian)
            {
                this.buf[0] = (byte)value;
                this.buf[1] = (byte)(value >> 8);
                this.buf[2] = (byte)(value >> 16);
            }
            else
            {
                this.buf[0] = (byte)(value >> 16);
                this.buf[1] = (byte)(value >> 8);
                this.buf[2] = (byte)value;
            }

            this.bw.Write(this.buf, 0, 3);
        }

        public void Write(object obj)
        {
            this.Write(null, obj);
        }

        private void Write(FieldInfo fieldInfo, object obj)
        {
            var type = obj.GetType();
            if (type.IsPrimitive)
            {
                this.WritePrimitive(type, obj);
            }
            else if (type.IsArray)
            {
                if (null == fieldInfo)
                {
                    throw new InvalidOperationException("fInfo required");
                }

                var marshalAsAttribute = (MarshalAsAttribute)Attribute.GetCustomAttribute(fieldInfo, typeof(MarshalAsAttribute));
                if (null == marshalAsAttribute)
                {
                    throw new InvalidOperationException("MarshalAs required");
                }

                if (UnmanagedType.ByValArray != marshalAsAttribute.Value)
                {
                    throw new InvalidOperationException("ByValArray only!");
                }

                if (((Array)obj).Length != marshalAsAttribute.SizeConst)
                {
                    throw new InvalidOperationException("Length is Invalid!");
                }

                var elementType = type.GetElementType();
                if (elementType.IsPrimitive)
                {
                    this.WritePrimitiveArray(elementType, obj);
                }
                else
                {
                    foreach (var element in (IEnumerable)obj)
                    {
                        this.Write(null, element);
                    }
                }
            }
            else
            {
                foreach (var subFInfo in type.GetFields(BindingFlags.Public | BindingFlags.NonPublic | BindingFlags.Instance))
                {
                    this.Write(subFInfo, subFInfo.GetValue(obj));
                }
            }
        }

        private void WritePrimitive(Type type, object obj)
        {
            if (type == typeof(sbyte))
            {
                this.Write((sbyte)obj);
            }
            else if (type == typeof(byte))
            {
                this.Write((byte)obj);
            }
            else if (type == typeof(short))
            {
                this.Write((short)obj);
            }
            else if (type == typeof(ushort))
            {
                this.Write((ushort)obj);
            }
            else if (type == typeof(int))
            {
                this.Write((int)obj);
            }
            else if (type == typeof(uint))
            {
                this.Write((uint)obj);
            }
            else
            {
                throw new InvalidOperationException(string.Format("Unsupport type. - {0}", type));
            }
        }

        private void WritePrimitiveArray(Type fieldType, object obj)
        {
            if (fieldType == typeof(sbyte))
            {
                Array.ForEach((sbyte[])obj, (value) => this.Write(value));
            }
            else if (fieldType == typeof(byte))
            {
                Array.ForEach((byte[])obj, (value) => this.Write(value));
            }
            else if (fieldType == typeof(short))
            {
                Array.ForEach((short[])obj, (value) => this.Write(value));
            }
            else if (fieldType == typeof(ushort))
            {
                Array.ForEach((ushort[])obj, (value) => this.Write(value));
            }
            else if (fieldType == typeof(int))
            {
                Array.ForEach((int[])obj, (value) => this.Write(value));
            }
            else if (fieldType == typeof(uint))
            {
                Array.ForEach((uint[])obj, (value) => this.Write(value));
            }
            else
            {
                throw new InvalidOperationException(string.Format("Unsupport type. - {0}", fieldType));
            }
        }
    }
}
