﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Font
{
    using System;
    using System.Windows.Input;

    /// <summary>
    /// クローズコマンドを実装するViewModelです。
    /// </summary>
    public abstract class WorkspaceViewModel : ViewModelBase
    {
        private RelayCommand closeCommand;
        private string currentPath;
        private StringObservableCollection pathHistory;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        protected WorkspaceViewModel()
        {
        }

        /// <summary>
        /// クローズ行われたときに発生します。
        /// </summary>
        public event EventHandler RequestClose;

        /// <summary>
        /// クローズコマンドを取得します。
        /// </summary>
        public ICommand CloseCommand
        {
            get
            {
                if (this.closeCommand == null)
                {
                    this.closeCommand = new RelayCommand((param) => this.OnRequestClose());
                }

                return this.closeCommand;
            }
        }

        /// <summary>
        /// 現在のパスを取得・設定します。
        /// </summary>
        public string CurrentPath
        {
            get
            {
                return this.currentPath;
            }

            set
            {
                if (value == this.currentPath)
                {
                    return;
                }

                this.currentPath = value;
                this.OnPropertyChanged("CurrentPath");
            }
        }

        /// <summary>
        /// パスの履歴を取得・設定します。
        /// </summary>
        public StringObservableCollection PathHistory
        {
            get { return this.pathHistory; }
        }

        /// <summary>
        /// 現在のパスを保存します。
        /// </summary>
        public virtual void SavePathToHistory()
        {
            if (null != this.PathHistory)
            {
                this.PathHistory.InsertHistoryFront(this.CurrentPath);
            }
        }

        /// <summary>
        /// パス履歴を設定します。
        /// </summary>
        /// <param name="pathHistory">パス文字列リスト</param>
        protected void SetPathHistory(StringList pathHistory)
        {
            if (null == pathHistory)
            {
                this.pathHistory = new StringObservableCollection();
            }
            else
            {
                this.pathHistory = new StringObservableCollection(pathHistory);
            }

            if (this.pathHistory.Count > 0)
            {
                this.CurrentPath = this.pathHistory[0];
            }
        }

        private void OnRequestClose()
        {
            EventHandler handler = this.RequestClose;
            if (handler != null)
            {
                handler(this, EventArgs.Empty);
            }
        }
    }
}
