﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Font
{
    using System;
    using System.ComponentModel;
    using System.Diagnostics;

    /// <summary>
    /// 全てのViewModelの基本クラスです。
    /// </summary>
    public abstract class ViewModelBase : INotifyPropertyChanged, IDisposable
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        protected ViewModelBase()
        {
        }

#if DEBUG
        /// <summary>
        /// デストラクタ。
        /// ビューモデルオブジェクトがガベージコレクトされたかどうかを確認するのに
        /// 役立ちます。
        /// </summary>
        ~ViewModelBase()
        {
            var msg = string.Format("{0} ({1}) ({2}) Finalized", this.GetType().Name, this.DisplayName, this.GetHashCode());
            Debug.WriteLine(msg);
        }
#endif

        /// <summary>
        /// プロパティ値が変更されたときに発生します。
        /// </summary>
        public event PropertyChangedEventHandler PropertyChanged;

        /// <summary>
        /// 表示名を取得します。
        /// </summary>
        public virtual string DisplayName { get; protected set; }

        /// <summary>
        /// VerifyPropertyNameメソッドを呼び出したときに、
        /// 例外をスローするか、Debug.Fail()を呼び出すかを判別する値を
        /// 取得設定します。
        /// </summary>
        protected virtual bool ThrowOnInvalidPropertyName { get; private set; }

        /// <summary>
        /// 破棄処理を行います。
        /// 内部的には、プロテクトメソッド OnDisposeを呼び出します。
        /// </summary>
        public void Dispose()
        {
            this.OnDispose();
        }

        /// <summary>
        /// プロパティ名の検証を行います。
        /// </summary>
        /// <param name="propertyName">プロパティ名。</param>
        [Conditional("DEBUG")]
        [DebuggerStepThrough]
        public void VerifyPropertyName(string propertyName)
        {
            // プロパティが存在するか確認します。
            if (TypeDescriptor.GetProperties(this)[propertyName] == null)
            {
                string msg = "Invalid property name: " + propertyName;

                if (this.ThrowOnInvalidPropertyName)
                {
                    throw new Exception(msg);
                }
                else
                {
                    Debug.Fail(msg);
                }
            }
        }

        /// <summary>
        /// PropertyChangedイベントを発生させます。
        /// </summary>
        /// <param name="propertyName">値が変更されたプロパティの名前</param>
        protected virtual void OnPropertyChanged(string propertyName)
        {
            this.VerifyPropertyName(propertyName);

            var handler = this.PropertyChanged;
            if (handler != null)
            {
                var e = new PropertyChangedEventArgs(propertyName);
                handler(this, e);
            }
        }

        /// <summary>
        /// 派生クラスにて破棄処理を行う場合にオーバーライドします。
        /// </summary>
        protected virtual void OnDispose()
        {
        }
    }
}
