﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Font
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Windows;
    using System.Windows.Media;
    using NintendoWare.Font.Win32;

    /// <summary>
    /// イメージ出力用ViewModelです。
    /// </summary>
    public class OutImageViewModel : OutSettingViewModel
    {
        // ImageFileFormat と同じ順序でなければならない
        private const string FileFilterStr = "Supported Images (*.bmp, *.tga, *.tif)|*.bmp;*.tga;*.tif"
                                              + "|Bitmap Images (*.bmp)|*.bmp"
                                              + "|Targa Images (*.tga)|*.tga"
                                              + "|TIFF Images (*.tif)|*.tif"
                                              + "|all files (*.*)|*.*";

        private readonly OutImageSettings settings;
        private ImageFileFormatViewModel[] imageFileFormatInfos;
        private List<NnsFileInfo> orderInfos;
        private OrderFileListProxy orderFileListProxy;
        private bool isOrderInfosReady;
        private COLORREF[] customColors;

        public OutImageViewModel(OutImageSettings settings, OrderFileListProxy orderFileListProxy)
        {
            this.DisplayName = "Image";
            this.settings = settings;
            this.orderFileListProxy = orderFileListProxy;

            this.SetPathHistory(this.settings.FilePaths);
        }

        public ImageFileFormatViewModel[] ImageFileFormatInfos
        {
            get
            {
                if (null == this.imageFileFormatInfos)
                {
                    // ImageFileFormat と同じ順序でなければならない
                    this.imageFileFormatInfos = new ImageFileFormatViewModel[]
                    {
                        new ImageFileFormatViewModel(Strings.IDS_OUTIMAGE_FORMAT_EXT, ImageFileFormat.Ext, null),
                        new ImageFileFormatViewModel(Strings.IDS_OUTIMAGE_FORMAT_BMP, ImageFileFormat.Bmp, "bmp"),
                        new ImageFileFormatViewModel(Strings.IDS_OUTIMAGE_FORMAT_TGA, ImageFileFormat.Tga, "tga"),
                        new ImageFileFormatViewModel(Strings.IDS_OUTIMAGE_FORMAT_TIFF, ImageFileFormat.Tif, "tif"),
                    };
                }

                return this.imageFileFormatInfos;
            }
        }

        public string OrderFile
        {
            get
            {
                if (!this.isOrderInfosReady)
                {
                    return null;
                }

                return this.settings.OrderFile;
            }

            set
            {
                if (value == null)
                {
                    Debug.WriteLine("OrderFile property set - null");
                    return;
                }

                if (!this.isOrderInfosReady)
                {
                    return;
                }

                if (value == this.settings.OrderFile)
                {
                    return;
                }

                this.settings.OrderFile = value;
                this.OnPropertyChanged("OrderFile");
            }
        }

        public bool IsSizeOffsetSpec
        {
            get
            {
                return this.settings.IsSizeOffsetSpec;
            }

            set
            {
                if (value == this.settings.IsSizeOffsetSpec)
                {
                    return;
                }

                this.settings.IsSizeOffsetSpec = value;
                this.OnPropertyChanged("IsSizeOffsetSpec");
                this.OnPropertyChanged("MarginSettingVisibility");
                this.OnPropertyChanged("SizeOffsetSettingVisibility");
            }
        }

        public Visibility MarginSettingVisibility
        {
            get
            {
                return !this.IsSizeOffsetSpec ? Visibility.Visible : Visibility.Hidden;
            }
        }

        public Visibility SizeOffsetSettingVisibility
        {
            get
            {
                return this.IsSizeOffsetSpec ? Visibility.Visible : Visibility.Hidden;
            }
        }

        public int CellSizeWidth
        {
            get
            {
                return this.settings.CellSizeWidth;
            }

            set
            {
                if (value == this.settings.CellSizeWidth)
                {
                    return;
                }

                this.settings.CellSizeWidth = value;
                this.OnPropertyChanged("CellSizeWidth");
            }
        }

        public int CellSizeHeight
        {
            get
            {
                return this.settings.CellSizeHeight;
            }

            set
            {
                if (value == this.settings.CellSizeHeight)
                {
                    return;
                }

                this.settings.CellSizeHeight = value;
                this.OnPropertyChanged("CellSizeHeight");
            }
        }

        public int CellSizeLeft
        {
            get
            {
                return this.settings.CellSizeLeft;
            }

            set
            {
                if (value == this.settings.CellSizeLeft)
                {
                    return;
                }

                this.settings.CellSizeLeft = value;
                this.OnPropertyChanged("CellSizeLeft");
            }
        }

        public int CellSizeTop
        {
            get
            {
                return this.settings.CellSizeTop;
            }

            set
            {
                if (value == this.settings.CellSizeTop)
                {
                    return;
                }

                this.settings.CellSizeTop = value;
                this.OnPropertyChanged("CellSizeTop");
            }
        }

        public int CellMarginLeft
        {
            get
            {
                return this.settings.CellMarginLeft;
            }

            set
            {
                if (value == this.settings.CellMarginLeft)
                {
                    return;
                }

                this.settings.CellMarginLeft = value;
                this.OnPropertyChanged("CellMarginLeft");
            }
        }

        public int CellMarginTop
        {
            get
            {
                return this.settings.CellMarginTop;
            }

            set
            {
                if (value == this.settings.CellMarginTop)
                {
                    return;
                }

                this.settings.CellMarginTop = value;
                this.OnPropertyChanged("CellMarginTop");
            }
        }

        public int CellMarginRight
        {
            get
            {
                return this.settings.CellMarginRight;
            }

            set
            {
                if (value == this.settings.CellMarginRight)
                {
                    return;
                }

                this.settings.CellMarginRight = value;
                this.OnPropertyChanged("CellMarginRight");
            }
        }

        public int CellMarginBottom
        {
            get
            {
                return this.settings.CellMarginBottom;
            }

            set
            {
                if (value == this.settings.CellMarginBottom)
                {
                    return;
                }

                this.settings.CellMarginBottom = value;
                this.OnPropertyChanged("CellMarginBottom");
            }
        }

        public bool IsImageCenter
        {
            get
            {
                return this.settings.IsImageCenter;
            }

            set
            {
                if (value == this.settings.IsImageCenter)
                {
                    return;
                }

                this.settings.IsImageCenter = value;
                this.OnPropertyChanged("IsImageCenter");
            }
        }

        public bool IsDrawGrid
        {
            get
            {
                return this.settings.IsDrawGrid;
            }

            set
            {
                if (value == this.settings.IsDrawGrid)
                {
                    return;
                }

                this.settings.IsDrawGrid = value;
                this.OnPropertyChanged("IsDrawGrid");
            }
        }

        public ImageFileFormat OutImageFormat
        {
            get
            {
                return this.settings.OutImageFormat;
            }

            set
            {
                if (value == this.settings.OutImageFormat)
                {
                    return;
                }

                this.settings.OutImageFormat = value;
                this.OnPropertyChanged("OutImageFormat");
                this.OnPropertyChanged("FilterIndex");
            }
        }

        public Color NullBlockColor
        {
            get
            {
                return this.settings.NullBlockColor;
            }

            set
            {
                if (value == this.settings.NullBlockColor)
                {
                    return;
                }

                this.settings.NullBlockColor = value;
                this.OnPropertyChanged("NullBlockColor");
            }
        }

        public Color GridColor
        {
            get
            {
                return this.settings.GridColor;
            }

            set
            {
                if (value == this.settings.GridColor)
                {
                    return;
                }

                this.settings.GridColor = value;
                this.OnPropertyChanged("GridColor");
            }
        }

        public Color WidthBarColor
        {
            get
            {
                return this.settings.WidthBarColor;
            }

            set
            {
                if (value == this.settings.WidthBarColor)
                {
                    return;
                }

                this.settings.WidthBarColor = value;
                this.OnPropertyChanged("WidthBarColor");
            }
        }

        public Color MarginColor
        {
            get
            {
                return this.settings.MarginColor;
            }

            set
            {
                if (value == this.settings.MarginColor)
                {
                    return;
                }

                this.settings.MarginColor = value;
                this.OnPropertyChanged("MarginColor");
            }
        }

        public Color BackgroundColor
        {
            get
            {
                return this.settings.BackgroundColor;
            }

            set
            {
                if (value == this.settings.BackgroundColor)
                {
                    return;
                }

                this.settings.BackgroundColor = value;
                this.OnPropertyChanged("BackgroundColor");
            }
        }

        public List<NnsFileInfo> OrderInfos
        {
            get
            {
                if (null == this.orderInfos)
                {
                    this.orderFileListProxy.Loaded += this.OrderFileListProxy_Loaded;
                    this.orderInfos = this.orderFileListProxy.OrderFiles;
                    this.isOrderInfosReady = this.orderFileListProxy.IsLoaded;
                    this.IsReady = this.isOrderInfosReady;
                }

                return this.orderInfos;
            }

            private set
            {
                this.orderInfos = value;
                Debug.WriteLine("Pre OnPropertyChanged OrderInfos");
                this.OnPropertyChanged("OrderInfos");
                Debug.WriteLine("Aft OnPropertyChanged OrderInfos");
                this.isOrderInfosReady = true;
            }
        }

        /// <summary>
        /// カスタムカラー配列を取得します。
        /// </summary>
        public COLORREF[] CustomColors
        {
            get
            {
                if (null == this.customColors)
                {
                    this.customColors = new COLORREF[16];
                }

                return this.customColors;
            }
        }

        public string FileFilter
        {
            get { return FileFilterStr; }
        }

        public int FilterIndex
        {
            get { return (int)this.OutImageFormat; }
        }

        public override FontWriter GetFontWriter()
        {
            if (!this.IsReady)
            {
                throw GlCm.ErrMsg(ErrorType.Parameter, Strings.IDS_NOW_READING);
            }

            //// CBmpFontWriter::Rotation rot;
            int cellWidth = -1, cellHeight = -1, marginLeft = 0, marginTop = 0;
            int marginRight = 0, marginBottom = 0;

            var file = this.CurrentPath;

            var orderPath = this.settings.OrderFile;
            if (orderPath == null)
            {
                throw GlCm.ErrMsg(ErrorType.Parameter, Strings.IDS_ERR_ORDER_NOT_SELECTED);
            }

            orderPath = Path.Combine(this.orderFileListProxy.OrderFilesDir, orderPath);

            //// rot         = CBmpFontWriter::ROTATE_NONE;  // GetRotate();
            var format = this.settings.OutImageFormat;

            if (this.settings.IsSizeOffsetSpec)
            {
                cellWidth = this.settings.CellSizeWidth;
                cellHeight = this.settings.CellSizeHeight;

                if (!this.settings.IsImageCenter)
                {
                    marginLeft = this.settings.CellSizeLeft;
                    marginTop = this.settings.CellSizeTop;
                }
                else
                {
                    marginLeft = -1;
                    marginTop = -1;
                }
            }
            else
            {
                marginLeft = this.settings.CellMarginLeft;
                marginTop = this.settings.CellMarginTop;

                if (this.settings.IsImageCenter)
                {
                    marginRight = marginLeft;
                    marginBottom = marginTop;
                }
                else
                {
                    marginRight = this.settings.CellMarginRight;
                    marginBottom = this.settings.CellMarginBottom;
                }
            }

            // file の拡張子を調整します。
            if (format != ImageFileFormat.Ext)
            {
                var extFormat = ImageFileFormat.Ext;

                // file の拡張子が対応形式の物かチェックします。
                for (int i = 0; i < this.ImageFileFormatInfos.Length; ++i)
                {
                    if (this.ImageFileFormatInfos[i].FileExtension != null)
                    {
                        var ext = "." + this.ImageFileFormatInfos[i].FileExtension;
                        var testExt = Path.GetExtension(file);

                        if (string.Equals(ext, testExt, StringComparison.OrdinalIgnoreCase))
                        {
                            extFormat = this.ImageFileFormatInfos[i].Format;
                            break;
                        }
                    }
                }

                // file の拡張子が format の拡張子と異なるのであれば拡張子を付加します。
                if (extFormat != format)
                {
                    // file の拡張子が対応形式の物であれば拡張子を取り除きます。
                    if (extFormat != ImageFileFormat.Ext)
                    {
                        file = Path.Combine(Path.GetDirectoryName(file), Path.GetFileNameWithoutExtension(file));
                    }

                    // 拡張子を付加し、表示を更新します。
                    file += "." + this.ImageFileFormatInfos[(int)format].FileExtension;
                    this.CurrentPath = file;
                }
            }

            // ファイルの上書きチェック
            // キャンセルされたら例外が飛ぶ
            FileOverwriteConfirmDialog.Show(file);

            return
                new ImageFontWriter(
                    file,
                    orderPath,
                    format,
                    this.settings.IsDrawGrid,
                    this.settings.IsOutKerningData,
                    cellWidth,
                    cellHeight,
                    marginLeft,
                    marginTop,
                    marginRight,
                    marginBottom,
                    GlCm.ColorToUint(this.settings.GridColor),
                    GlCm.ColorToUint(this.settings.MarginColor),
                    GlCm.ColorToUint(this.settings.WidthBarColor),
                    GlCm.ColorToUint(this.settings.NullBlockColor),
                    GlCm.ColorToUint(this.settings.BackgroundColor));
        }

        public override void SaveState()
        {
            this.settings.FilePaths = new StringList(this.PathHistory);
        }

        private void OrderFileListProxy_Loaded(object sender, EventArgs e)
        {
            this.OrderInfos = this.orderFileListProxy.OrderFiles;

            // 現在の値に一致する項目が無い場合は先頭の内容にする。
            var currentOrderFile = this.settings.OrderFile;
            var found = this.OrderInfos.Find((orderInfo) => orderInfo.FileName == currentOrderFile);
            if (null == found)
            {
                this.settings.OrderFile = this.orderInfos[0].FileName;
            }

            this.OnPropertyChanged("OrderFile");

            this.IsReady = true;
        }
    }
}
