﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Font
{
    using System;
    using System.Globalization;
    using System.Windows.Controls;

    /// <summary>
    /// 整数値の範囲の検証を行うための規則
    /// </summary>
    public class Int32RangeRule : ValidationRule
    {
        /// <summary>
        /// 受けいれる最小値を取得または設定します。
        /// </summary>
        public int Min { get; set; }

        /// <summary>
        /// 受け入れる最大値を取得または設定します。
        /// </summary>
        public int Max { get; set; }

        /// <summary>
        /// 値の検証チェックを行います。
        /// </summary>
        /// <param name="value">チェックするバインディング ターゲットの値。 </param>
        /// <param name="cultureInfo">この規則で使用するカルチャ。 </param>
        /// <returns>ValidationResult オブジェクト。 </returns>
        public override ValidationResult Validate(object value, CultureInfo cultureInfo)
        {
            int intValue = 0;

            try
            {
                var stringValue = value as string;
                if (stringValue == null)
                {
                    return new ValidationResult(false, "Illegal type");
                }

                intValue = Int32.Parse(stringValue);
            }
            catch (FormatException e)
            {
                return CreateExceptionResult(e);
            }
            catch (OverflowException e)
            {
                return CreateExceptionResult(e);
            }

            if (intValue < this.Min || intValue > this.Max)
            {
                var errMsg = string.Format(Strings.ErrorNumberRange, this.Min, this.Max);
                return new ValidationResult(false, errMsg);
            }
            else
            {
                return ValidationResult.ValidResult;
            }
        }

        /// <summary>
        /// 例外が送出された場合のValidationResultを生成します。
        /// </summary>
        /// <param name="e">例外オブジェクト</param>
        /// <returns>ValidationResult オブジェクト。 </returns>
        private static ValidationResult CreateExceptionResult(Exception e)
        {
            return new ValidationResult(false, e.Message);
        }
    }
}
