﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Font
{
    using System.Collections.ObjectModel;
    using System.IO;
    using System.Windows;
    using System.Windows.Controls;

    /// <summary>
    /// FileSlectComboBox.xaml の相互作用ロジック
    /// </summary>
    public partial class FileSelectComboBox : UserControl
    {
        public static readonly DependencyProperty PathHistoryProperty =
            DependencyProperty.Register(
                "PathHistory",
                typeof(ObservableCollection<string>),
                typeof(FileSelectComboBox));

        public static readonly DependencyProperty CurrentPathProperty =
            DependencyProperty.Register(
                "CurrentPath",
                typeof(string),
                typeof(FileSelectComboBox));

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public FileSelectComboBox()
        {
            InitializeComponent();

            this.combBox.DataContext = this;
        }

        /// <summary>
        /// ファイルパスの履歴を取得または設定します。
        /// </summary>
        public ObservableCollection<string> PathHistory
        {
            get
            {
                return (ObservableCollection<string>)this.GetValue(PathHistoryProperty);
            }

            set
            {
                this.SetValue(PathHistoryProperty, value);
            }
        }

        /// <summary>
        /// 現在のファイルパスを取得または設定します。
        /// </summary>
        public string CurrentPath
        {
            get
            {
                return (string)this.GetValue(CurrentPathProperty);
            }

            set
            {
                this.SetValue(CurrentPathProperty, value);
            }
        }

        private void ComboBox_PreviewDragOver(object sender, DragEventArgs e)
        {
            if (this.IsSingleFile(e) != null)
            {
                e.Effects = DragDropEffects.Copy;
            }
            else
            {
                e.Effects = DragDropEffects.None;
            }

            e.Handled = true;
        }

        private void ComboBox_PreviewDrop(object sender, DragEventArgs e)
        {
            e.Handled = true;

            var fileName = this.IsSingleFile(e);
            if (fileName == null)
            {
                return;
            }

            if (sender is ComboBox)
            {
                this.DropFiles(fileName);
            }
        }

        private void DropFiles(string path)
        {
            this.combBox.Text = path;
        }

        /// <summary>
        /// 1つのファイルかどうかを判定し、1つの場合はそのファイル名を返します。
        /// </summary>
        /// <param name="args">DragEventArgs。</param>
        /// <returns>
        /// データが1つのファイルの場合はそのファイル名を返します。
        /// そうでない場合は null を返します。
        /// </returns>
        private string IsSingleFile(DragEventArgs args)
        {
            if (args.Data.GetDataPresent(DataFormats.FileDrop, true))
            {
                string[] fileNames = args.Data.GetData(DataFormats.FileDrop, true) as string[];

                // １つだけか
                if (fileNames.Length == 1)
                {
                    // 存在するか
                    if (File.Exists(fileNames[0]))
                    {
                        return fileNames[0];
                    }
                }
            }

            return null;
        }
    }
}
