﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Font
{
    using System.IO;
    using System.Windows;
    using System.Windows.Controls;
    using Microsoft.Win32;

    /// <summary>
    /// FileDialogButton.xaml の相互作用ロジック
    /// </summary>
    public partial class FileDialogButton : UserControl
    {
        public static readonly DependencyProperty FilePathProperty =
            DependencyProperty.Register(
                "FilePath",
                typeof(string),
                typeof(FileDialogButton));

        public static readonly DependencyProperty FileFilterProperty =
            DependencyProperty.Register(
                "FileFilter",
                typeof(string),
                typeof(FileDialogButton));

        public static readonly DependencyProperty FilterIndexProperty =
            DependencyProperty.Register(
                "FilterIndex",
                typeof(int),
                typeof(FileDialogButton));

        public static readonly DependencyProperty FileExtensionProperty =
            DependencyProperty.Register(
                "FileExtension",
                typeof(string),
                typeof(FileDialogButton));

        public static readonly DependencyProperty IsOutputProperty =
            DependencyProperty.Register(
                "IsOutput",
                typeof(bool),
                typeof(FileDialogButton));

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public FileDialogButton()
        {
            InitializeComponent();

            this.DialogTitle = "Select File";
        }

        /// <summary>
        /// ファイルの完全なパスを含む文字列を取得または設定します。
        /// </summary>
        public string FilePath
        {
            get { return (string)this.GetValue(FilePathProperty); }
            set { this.SetValue(FilePathProperty, value); }
        }

        /// <summary>
        /// ファイルダイアログで表示されるファイルの種類を決定するフィルタ文字列を取得または設定します。
        /// </summary>
        public string FileFilter
        {
            get { return (string)this.GetValue(FileFilterProperty); }
            set { this.SetValue(FileFilterProperty, value); }
        }

        /// <summary>
        /// ファイルダイアログで選択されるフィルタのインデックスを取得または設定します。
        /// 0から始まります。
        /// </summary>
        public int FilterIndex
        {
            get { return (int)this.GetValue(FilterIndexProperty); }
            set { this.SetValue(FilterIndexProperty, value); }
        }

        /// <summary>
        /// 拡張子文字列を取得または設定します。
        /// </summary>
        public string FileExtension
        {
            get { return (string)this.GetValue(FileExtensionProperty); }
            set { this.SetValue(FileExtensionProperty, value); }
        }

        /// <summary>
        /// ファイルダイアログが出力用かどうかを判定する値を取得します。
        /// </summary>
        public bool IsOutput
        {
            get { return (bool)this.GetValue(IsOutputProperty); }
            set { this.SetValue(IsOutputProperty, value); }
        }

        private void Button_Click(object sender, RoutedEventArgs e)
        {
            this.PathSelect();
        }

        /// <summary>
        /// ファイルの完全なパスを含む文字列を取得または設定します。
        /// </summary>
        public string DialogTitle
        {
            get;
            set;
        }

        // コモンダイアログでファイルを指定
        private void PathSelect()
        {
            FileDialog ofn;

            if (!this.IsOutput)
            {
                ofn = new OpenFileDialog();
            }
            else
            {
                var dlg = new SaveFileDialog();
                dlg.OverwritePrompt = true;
                ofn = dlg;
            }

            ofn.Filter = this.FileFilter;
            ofn.FilterIndex = 1 + this.FilterIndex;
            ofn.FileName = File.Exists(this.FilePath) ? this.FilePath : string.Empty;
            ofn.DefaultExt = this.FileExtension;
            ofn.Title = this.DialogTitle;

            // FilePathのディレクトリを初期ディレクトリにする
            // 無効なディレクトリの場合は最後に開かれたディレクトリが初期ディレクトリになる
            try
            {
                ofn.InitialDirectory = Path.GetDirectoryName(this.FilePath);
            }
            catch
            {
                ofn.InitialDirectory = System.String.Empty;
            }

            // カレントディレクトリ保存
            var cwd = Directory.GetCurrentDirectory();

            bool isSelected;
            try
            {
                isSelected = ofn.ShowDialog() ?? false;
            }
            catch
            {
                // 存在しないドライブを指定された場合にShowDialogが例外を投げる
                // その場合は空文字をセットして再度ShowDialogを呼び出すようにする
                ofn.InitialDirectory = System.String.Empty;
                isSelected = ofn.ShowDialog() ?? false;
            }

            // カレントディレクトリ復帰
            Directory.SetCurrentDirectory(cwd);

            if (isSelected)
            {
                this.FilePath = ofn.FileName;
            }
        }
    }
}
