﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <DrawParam.h>


namespace vfxdemo {


//---------------------------------------------------------------------------
//! @brief  基底クラスです。
//---------------------------------------------------------------------------
class Grid
{
public:
    //---------------------------------------------------------------------------
    //! @brief  描画メソッドです。
    //---------------------------------------------------------------------------
    virtual void ProcDraw( const DrawParam& param, float gridSize, int divCount ) NN_NOEXCEPT
    {
        NN_UNUSED( param );
        NN_UNUSED( gridSize );
        NN_UNUSED( divCount );
    }
};


//---------------------------------------------------------------------------
//! @brief  キューブグリッドクラスです。
//---------------------------------------------------------------------------
class CubeGrid : public Grid
{
public:
    //---------------------------------------------------------------------------
    //! @brief  標準背景描画メソッドです。
    //---------------------------------------------------------------------------
    virtual void ProcDraw( const DrawParam& drawParam, float gridSize, int divCount ) NN_NOEXCEPT
    {
        // Cube で divCount には対応しない。
        NN_UNUSED( divCount );

        // Blend
        drawParam.m_CommandBuffer->SetBlendState( drawParam.m_PrimitiveRenderer->GetBlendState( nns::gfx::PrimitiveRenderer::BlendType::BlendType_Normal ) );

        // Depth Enable
        drawParam.m_PrimitiveRenderer->SetDepthStencilState( drawParam.m_CommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthWriteTest );

        drawParam.m_PrimitiveRenderer->SetProjectionMatrix( &drawParam.m_ProjMtx );
        drawParam.m_PrimitiveRenderer->SetViewMatrix( &drawParam.m_ViewMtx );
        nn::util::Matrix4x3fType modelMatrix;
        nn::util::MatrixIdentity(&modelMatrix);
        drawParam.m_PrimitiveRenderer->SetModelMatrix( &modelMatrix );

        int cnt = 0;
        int halfCount = 5;
        for ( int i = -halfCount; i < halfCount; i++ )
        {
            for ( int j = -halfCount; j < halfCount; j++ )
            {
                nn::util::Vector3fType center = { i * gridSize + gridSize / 2.f, 0.f, j * gridSize + gridSize / 2.f };
                nn::util::Vector3fType size   = { gridSize, 0.1f, gridSize };

                nn::util::Uint8x4 color = { { 32, 32, 32, 255 } };
                if ( cnt%2 == 0 )
                {
                    color.v[0] = 64;
                    color.v[1] = 64;
                    color.v[2] = 64;
                }
                drawParam.m_PrimitiveRenderer->SetColor(color);
                drawParam.m_PrimitiveRenderer->DrawCube(drawParam.m_CommandBuffer,
                    nns::gfx::PrimitiveRenderer::Surface::Surface_Solid, center, size);
                cnt++;
            }
            cnt++;
        }
    }
};

//---------------------------------------------------------------------------
//! @brief  標準グリッド描画クラスです。
//---------------------------------------------------------------------------
class DefaultGrid : public Grid
{
public:
    //---------------------------------------------------------------------------
    //! @brief  標準グリッド描画メソッドです。
    //---------------------------------------------------------------------------
    virtual void ProcDraw( const DrawParam& drawParam, float gridSize, int divCount ) NN_NOEXCEPT
    {
        int gridCount = divCount * 10;
        float gridStepSize = gridSize / divCount;

        // Blend
        drawParam.m_CommandBuffer->SetBlendState( drawParam.m_PrimitiveRenderer->GetBlendState( nns::gfx::PrimitiveRenderer::BlendType::BlendType_Normal ) );

        // Depth Enable
        drawParam.m_PrimitiveRenderer->SetDepthStencilState( drawParam.m_CommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthWriteTest );

        drawParam.m_PrimitiveRenderer->SetProjectionMatrix( &drawParam.m_ProjMtx );
        drawParam.m_PrimitiveRenderer->SetViewMatrix( &drawParam.m_ViewMtx );
        nn::util::Matrix4x3fType modelMatrix;
        nn::util::MatrixIdentity(&modelMatrix);

        drawParam.m_PrimitiveRenderer->SetModelMatrix( &modelMatrix );
        {
            nn::util::Uint8x4 color = { { 255, 255, 255, 255 } };
            nn::util::Uint8x4 red = { { 255, 0, 0, 255 } };
            nn::util::Uint8x4 blue = { { 0, 0, 255, 255 } };
            nn::util::Uint8x4 green = { { 0, 255, 0, 255 } };

            drawParam.m_PrimitiveRenderer->SetLineWidth(1.f);
            drawParam.m_PrimitiveRenderer->SetColor(color);

            float interval = -gridStepSize * gridCount/2;
            const float minSize = -gridStepSize * gridCount/2;
            const float maxSize = gridStepSize * gridCount/2;
            for (int i = 0; i < ( gridCount + 1 ); i++)
            {
                {
                    nn::util::Vector3fType from;
                    nn::util::Vector3fType to;
                    nn::util::VectorSet( &from, minSize, 0.f, interval );
                    nn::util::VectorSet( &to, maxSize, 0.f, interval );
                    drawParam.m_PrimitiveRenderer->DrawLine( drawParam.m_CommandBuffer, from, to );
                    nn::util::VectorSet( &from, interval, 0.f, minSize );
                    nn::util::VectorSet( &to, interval, 0.f, maxSize );
                    drawParam.m_PrimitiveRenderer->DrawLine( drawParam.m_CommandBuffer, from, to );
                }
                interval += gridStepSize;
            }

            // AXIS
            nn::util::Vector3fType zeroVector;
            nn::util::VectorSet( &zeroVector, 0.f, 0.f, 0.f );
            nn::util::Vector3fType axisPos;
            nn::util::VectorSet( &axisPos, gridSize, 0.f, 0.f );
            drawParam.m_PrimitiveRenderer->SetLineWidth( 4.f );
            drawParam.m_PrimitiveRenderer->SetColor( red );
            drawParam.m_PrimitiveRenderer->DrawLine( drawParam.m_CommandBuffer, zeroVector, axisPos );
            drawParam.m_PrimitiveRenderer->SetLineWidth( 4.f );
            nn::util::VectorSet( &axisPos, 0.f, 0.f, gridSize );
            drawParam.m_PrimitiveRenderer->SetColor( blue );
            drawParam.m_PrimitiveRenderer->DrawLine( drawParam.m_CommandBuffer, zeroVector, axisPos );
            drawParam.m_PrimitiveRenderer->SetLineWidth( 4.f );
            nn::util::VectorSet( &axisPos, 0.f, gridSize, 0.f );
            drawParam.m_PrimitiveRenderer->SetColor( green );
            drawParam.m_PrimitiveRenderer->DrawLine( drawParam.m_CommandBuffer, zeroVector, axisPos );
        }
    }
};

} // namespace vfxdemo
