﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nns/nac/nac_MemoryAllocator.h>
#include <DrawParam.h>
#include <nn/vfx/vfx_System.h>
#include <nns/gfx/gfx_PrimitiveRenderer.h>


//---------------------------------------------------------------------------
//! @brief        グレアバッファの個数
//---------------------------------------------------------------------------
enum GLAREBUFFERS
{
    GLARE_BUFFER_MAX = 6,
    GLARE_GAUSS_LENGTH = 6
};


//---------------------------------------------------------------------------
//! @brief  バッファ
//---------------------------------------------------------------------------
class ConstantBuffer
{
public:
    //---------------------------------------------------------------------------
    //! @brief  コンストラクタ
    //---------------------------------------------------------------------------
    ConstantBuffer() NN_NOEXCEPT : m_IsInitialized( false ) {}

    //---------------------------------------------------------------------------
    //! @brief  バッファを初期化する
    //!
    //! @param[in] pGfxDevice       fxデバイス
    //! @param[in] heap             ヒープへのポインタ
    //! @param[in] gpuAccess        バッファタイプ
    //! @param[in] bufferCount      バッファ数
    //! @param[in] bufferSize       １バッファのサイズ
    //---------------------------------------------------------------------------
    bool Initialize( nn::gfx::Device* pGfxDevice, nns::nac::MemoryAllocator* allocator,
                        const size_t bufferSize, const uint32_t bufferCount ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  終了処理
    //!
    //! @param[in] pGfxDevice       fxデバイス
    //! @param[in] heap             ヒープへのポインタ
    //---------------------------------------------------------------------------
    void Finalize( nn::gfx::Device* device, nns::nac::MemoryAllocator* allocator ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  指定バッファをMapする
    //!
    //! @param[in] index       Mapするインデックス
    //---------------------------------------------------------------------------
    void* Map( int index ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  Unmapする
    //---------------------------------------------------------------------------
    void Unmap() NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  GpuAddressを取得する
    //!
    //! @param[out] address     GpuAddress
    //! @param[in]  index       GpuAddressを取得するインデックス
    //---------------------------------------------------------------------------
    void GetGpuAddress( nn::gfx::GpuAddress* address, int index ) NN_NOEXCEPT;


protected:
    bool                    m_IsInitialized;              //!< 初期化済みかどうか
    nn::gfx::MemoryPool     m_GfxMemoryPool;              //!< gfx メモリプール
    nn::gfx::Buffer         m_GfxBuffer;                  //!< gfx バッファ
    void*                   m_pBuffer;                    //!< バッファ用に確保した領域の先頭アドレス
    size_t                  m_BufferSize;                 //!< １バッファバッファサイズ
    uint32_t                m_BufferCount;                //!< バッファカウント
    void*                   m_MappedPtr;                  //!< マップして返したアドレス
};


class GlareEffect
{
public:
    GlareEffect():
        m_Width(0), m_Height(0),
        m_ProjMtx(),
        m_ViewMtx()
    {
        nn::util::MatrixIdentity(&m_ProjMtx);
        nn::util::MatrixIdentity(&m_ViewMtx);
    }

    //---------------------------------------------------------------------------
    //! @brief                   グレア関連の初期化を行います.
    //---------------------------------------------------------------------------
    void Initialize( nn::gfx::Device*                          device,
                     nns::nac::MemoryAllocator*                allocator,
                     vfxdemo::TextureDescriptorIndexAllocator* textureDesctAllocator,
                     vfxdemo::SamplerDescriptorIndexAllocator* samplerDesctAllocator,
                     int width, int height ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief                   終了処理を行います。
    //---------------------------------------------------------------------------
    void Finalize( nn::gfx::Device* device, nns::nac::MemoryAllocator* allocator ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief                   グレアの描画を行います。
    //---------------------------------------------------------------------------
    void Draw( const vfxdemo::DrawParam* pDrawParam, int glareNum ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  テクスチャのコピー( TextureView -> ColorTarget )を行います。
    //---------------------------------------------------------------------------
    void CopyTexture( nn::gfx::CommandBuffer*                commandBuffer,
                      nns::gfx::PrimitiveRenderer::Renderer* primitiveRenderer,
                      nn::gfx::ColorTargetView*              pDstColorTarget,
                      nn::gfx::DescriptorSlot                srcTextureDescSlot,
                      bool                                   yFlip = true ) NN_NOEXCEPT;

private:
    int m_Width, m_Height;                                                                      //!< バッファの基準となる大きさ
    nn::util::Matrix4x4fType                        m_ProjMtx;                                      //!< プロジェクションマトリクス
    nn::util::Matrix4x3fType                        m_ViewMtx;                                      //!< ビューマトリクス

    nn::gfx::Shader m_LuminanceSelectionPsShader;                                               //!< 輝度シェーダ
    nn::gfx::Shader m_CopyPsShader;                                                             //!< コピーシェーダ
    nn::gfx::Shader m_GaussXPsShader;                                                           //!< ガウスシェーダX軸
    nn::gfx::Shader m_GaussYPsShader;                                                           //!< ガウスシェーダY軸
    ConstantBuffer                              m_TextureSizeBuffer;                            //!< 縮小バッファサイズのコンスタントバッファ
    int                                         m_SlotGaussXParam;                              //!< X軸ガウスシェーダでのslot
    int                                         m_SlotGaussYParam;                              //!< Y軸ガウスシェーダでのslot

    ConstantBuffer                              m_GaussWeightBuffer;                            //!< 縮小バッファサイズのコンスタントバッファ
    int                                         m_SlotGaussXWeightParam;                        //!< X軸ガウスシェーダでのslot
    int                                         m_SlotGaussYWeightParam;                        //!< Y軸ガウスシェーダでのslot

    nn::gfx::Sampler                            m_Sampler;                                      //!< テクスチャサンプラ
    nn::gfx::DescriptorSlot                     m_SamplerDescSlot;                              //!< テクスチャサンプラ ID
    nn::util::Float2                            m_DownscaledTextureSize[GLARE_BUFFER_MAX];      //!< 縮小バッファのサイズ
    nn::gfx::ViewportScissorState               m_DownscaledViewportScissor[GLARE_BUFFER_MAX];  //!< 縮小バッファのビューポートとシザー
    nn::gfx::BlendState                         m_AddSumBlendState;                             //!< 加算合成を行うブレンドステート
    nn::gfx::ViewportScissorState               m_ViewportScissor;                               //!< 最終出力を行うためのビューポートとシザー

    nn::gfx::MemoryPool                         m_GfxTextureMemoryPool;                         //!< テクスチャ用 メモリプール
    void*                                       m_pTextureMemoryPooBuffer;                      //!< テクスチャ用 メモリプール用に確保した領域の先頭アドレス

    nn::gfx::Texture                            m_DownscaledXTexture[GLARE_BUFFER_MAX];         //!< X軸ガウスシェーダのバッファ
    nn::gfx::TextureView                        m_DownscaledXTextureView[GLARE_BUFFER_MAX];     //!< X軸ガウスシェーダのテクスチャビュー
    nn::gfx::ColorTargetView                    m_DownscaledXColorTargetView[GLARE_BUFFER_MAX]; //!< X軸ガウスシェーダのターゲットビュー
    nn::gfx::DescriptorSlot                     m_DownscaledXTextureDescSlot[GLARE_BUFFER_MAX]; //!< X軸ガウスシェーダのバッファのデスクリプタスロット

    nn::gfx::Texture                            m_DownscaledYTexture[GLARE_BUFFER_MAX];         //!< Y軸ガウスシェーダのバッファ
    nn::gfx::TextureView                        m_DownscaledYTextureView[GLARE_BUFFER_MAX];     //!< Y軸ガウスシェーダのテクスチャビュー
    nn::gfx::ColorTargetView                    m_DownscaledYColorTargetView[GLARE_BUFFER_MAX]; //!< Y軸ガウスシェーダのターゲットビュー
    nn::gfx::DescriptorSlot                     m_DownscaledYTextureDescSlot[GLARE_BUFFER_MAX]; //!< X軸ガウスシェーダのバッファのデスクリプタスロット
};
