﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using EffectMaker.BusinessLogic.IO;

namespace EffectMaker.UIControls.EffectBrowser.Controls.FileListView
{
    using System;
    using System.Collections.Concurrent;
    using System.Collections.Generic;
    using System.ComponentModel;
    using System.IO;
    using System.Linq;
    using System.Threading.Tasks;

    using EffectMaker.UIControls.EffectBrowser.Controls.FileListView.Base;
    using EffectMaker.UIControls.EffectBrowser.Data;
    using EffectMaker.UIControls.EffectBrowser.Data.FileCache;
    using EffectMaker.UIControls.EffectBrowser.Utilities;

    using FileInfo = EffectMaker.UIControls.EffectBrowser.Data.FileInfo;

    /// <summary>
    /// The eb search file list view.
    /// </summary>
    public class EBSearchFileListView : EBFileListView
    {
        #region Fields

        /// <summary>
        /// The directory.
        /// </summary>
        private string directory = string.Empty;

        /// <summary>
        /// The is base search path filtering.
        /// </summary>
        private bool isBaseSearchPathFiltering;

        /// <summary>
        /// The search info.
        /// </summary>
        private SearchFileInfo searchInfo;

        #endregion

        #region Public Events

        /// <summary>
        /// The directory changed.
        /// </summary>
        public event EventHandler DirectoryChanged;

        #endregion

        #region Public Properties

        /// <summary>
        /// Gets or sets the directory.
        /// </summary>
        [Browsable(false)]
        public string Directory
        {
            get
            {
                return this.directory;
            }

            set
            {
                if (this.directory == value)
                {
                    return;
                }

                this.directory = value;

                if (ControlEventSuppressBlock.Enabled == false)
                {
                    if (this.DirectoryChanged != null)
                    {
                        this.DirectoryChanged(this, EventArgs.Empty);
                    }
                }

                this.ClearItems();
            }
        }

        #endregion

        #region Properties

        /// <summary>
        /// リストビューで管理するファイルリストを取得します。
        /// 画面にはこのリストをソートしてフィルタリングした結果が表示されます。
        /// </summary>
        /// <exception cref="NotSupportedException">NotSupportedException</exception>
        protected override IEnumerable<FileInfo> Files
        {
            get
            {
                if (this.searchInfo == null)
                {
                    return new FileInfo[0];
                }

                switch (this.searchInfo.SearchMode)
                {
                    case SearchFileInfo.SearchModeKind.ReferenceFile:
                        return this.ReferenceFiles.Distinct(new FileInfoFileFullPathComparer());

                    case SearchFileInfo.SearchModeKind.UnusedFile:
                        return this.UnusedFiles.Distinct(new FileInfoFileFullPathComparer());

                    case SearchFileInfo.SearchModeKind.EmitterSetUsedFile:
                        return this.EmitterSetUsedFiles.Distinct(new FileInfoFileFullPathComparer());
                }

                throw new NotSupportedException();
            }
        }

        /// <summary>
        /// Gets the children in eset.
        /// </summary>
        private IEnumerable<string> ChildrenInEset
        {
            get
            {
                System.Diagnostics.Debug.Assert(this.searchInfo != null, "this.searchInfo != null");

                return PathUtility.EnumerateFiles(this.Directory, Constants.EsetPattern, SearchOption.AllDirectories)
                    ////.AsParallel()
                    .SelectMany(eset =>
                    {
                        var cache = FileCacheManager.Instance.Get(eset, true);
                        return cache.GetChidlen(this.searchInfo.FileKind);
                    });
            }
        }

        /// <summary>
        /// 「このファイルを参照しているエミッタセット」の検索ロジックに基づいた結果を取得します。
        /// </summary>
        private IEnumerable<FileInfo> EmitterSetUsedFiles
        {
            get
            {
                System.Diagnostics.Debug.Assert(this.searchInfo != null, "this.searchInfo != null");

                return PathUtility.EnumerateFiles(
                    PathUtility.NormalizePathString(this.Directory),
                    Constants.EsetPattern,
                    SearchOption.AllDirectories)
                    ////.AsParallel()
                    .Select(eset =>
                    {
                        var esetCache = FileCacheManager.Instance.Get(eset, true);
                        var baseDirPath = Path.GetDirectoryName(eset);

                        var children = esetCache.GetChidlen(this.searchInfo.FileKind);
                        var dirPaths = PathUtility.MakeChildFileDirPaths(
                            baseDirPath,
                            false,
                            this.searchInfo.FileKind).ToArray();

                        // ReSharper disable LoopCanBeConvertedToQuery
                        foreach (var child in children)
                        {
                            foreach (var dirPath in dirPaths)
                            {
                                var fillPath = Path.Combine(dirPath, child);

                                if (this.searchInfo.ReferenceFilePath == fillPath)
                                {
                                    return new FileInfo(esetCache);
                                }
                            }
                        }

                        // ReSharper restore LoopCanBeConvertedToQuery
                        return null;
                    }).Where(x => x != null);
            }
        }

        /// <summary>
        /// 「参照されているファイル」の検索ロジックに基づいた結果を取得します。
        /// </summary>
        private IEnumerable<FileInfo> ReferenceFiles
        {
            get
            {
                System.Diagnostics.Debug.Assert(this.searchInfo != null, "this.searchInfo != null");

                var childFileDirPaths = PathUtility.MakeReferenceSearchDirPaths(
                    this.Directory,
                    this.isBaseSearchPathFiltering,
                    this.searchInfo.FileKind);

                return this.ChildrenInEset.Select(child =>
                {
                    var childFullPath = PathUtility.MakeExistsReferenceFilePath(child, childFileDirPaths);

                    return childFullPath != null
                                ? new FileInfo(FileCacheManager.Instance.Get(childFullPath, false))
                                : null;
                }).Where(x => x != null);
            }
        }

        /// <summary>
        /// 「未使用ファイル」の検索ロジックに基づいた結果を取得します。
        /// </summary>
        private IEnumerable<FileInfo> UnusedFiles
        {
            get
            {
                System.Diagnostics.Debug.Assert(this.searchInfo != null, "this.searchInfo != null");

                var childFileDirPaths = PathUtility.MakeReferenceSearchDirPaths(
                    this.Directory,
                    this.isBaseSearchPathFiltering,
                    this.searchInfo.FileKind).ToArray();

                var allChildren = new ConcurrentDictionary<string, object>();
                {
                    var childPattern = PathUtility.MakeFileMaskPattern(this.searchInfo.FileKind);

                    foreach (var dir in childFileDirPaths)
                    {
                        foreach (var child in PathUtility.EnumerateFiles(dir, childPattern))
                        {
                            allChildren[child] = null;
                        }
                    }
                }

                Parallel.ForEach(
                    this.ChildrenInEset,
                    child =>
                    {
                        var childFullPath = PathUtility.MakeExistsReferenceFilePath(child, childFileDirPaths);

                        if (childFullPath != null)
                        {
                            object dummy;
                            allChildren.TryRemove(childFullPath, out dummy);
                        }
                    });

                return allChildren.Keys.Select(x => new FileInfo(FileCacheManager.Instance.Get(x, false)));
            }
        }

        #endregion

        #region Public Methods and Operators

        /// <summary>
        /// The request update.
        /// </summary>
        /// <param name="searchInfo">
        /// The search info.
        /// </param>
        /// <param name="isBaseSearchPathFiltering">
        /// The is base search path filtering.
        /// </param>
        public void RequestUpdate(SearchFileInfo searchInfo, bool isBaseSearchPathFiltering)
        {
            this.searchInfo = searchInfo.Clone();
            this.isBaseSearchPathFiltering = isBaseSearchPathFiltering;

            this.RequestUpdate();
        }

        #endregion
    }
}
