﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using EffectMaker.Foundation.Log;

namespace EffectMaker.UIControls.EffectBrowser.Controls.FileListView
{
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;

    using EffectMaker.UIControls.EffectBrowser.Controls.FileListView.Base;
    using EffectMaker.UIControls.EffectBrowser.Data;
    using EffectMaker.UIControls.EffectBrowser.Data.FileCache;

    using FileInfo = EffectMaker.UIControls.EffectBrowser.Data.FileInfo;

    /// <summary>
    /// The eb child file list view.
    /// </summary>
    public class EBChildFileListView : EBFileListView
    {
        #region Fields

        /// <summary>
        /// The directory.
        /// </summary>
        private string directory;

        /// <summary>
        /// The file kind.
        /// </summary>
        private FileKindType fileKind;

        #endregion

        #region Properties

        /// <summary>
        /// リストビューで管理するファイルリストを取得します。
        /// 画面にはこのファイルリストをソートしてフィルタリングした結果が表示されます。
        /// </summary>
        /// <exception cref="NotSupportedException">NotSupportedException</exception>
        protected override IEnumerable<FileInfo> Files
        {
            get
            {
                var maskPattern = Utilities.PathUtility.MakeFileMaskPattern(this.fileKind);
                var filePaths = Utilities.PathUtility.EnumerateFiles(
                    this.directory,
                    maskPattern,
                    SearchOption.AllDirectories).ToList();

                // 上階層のファイルも探索対象にする
                if (!Utilities.PathUtility.IsRootPath(this.directory))
                {
                    // 基点以下は探索済みなので、一個上から探索する
                    var upperOnce = Utilities.PathUtility.NormalizePathString(this.directory);
                    upperOnce = upperOnce.Substring(0, upperOnce.LastIndexOf('\\'));
                    if (Utilities.PathUtility.IsValidFilepathString(upperOnce))
                    {
                        // ディレクトリを列挙し、該当ファイルを探索して追加する
                        var upperDirs = Utilities.PathUtility.MakeChildFileDirPaths(upperOnce, false, this.fileKind);
                        foreach (var upperDir in upperDirs)
                        {
                            // 重複がないように追加する
                            var upperTargets = Utilities.PathUtility.EnumerateFiles(upperDir, maskPattern, SearchOption.TopDirectoryOnly);
                            foreach (var target in upperTargets.Where(target => !filePaths.Contains(target)))
                            {
                                filePaths.Add(target);
                            }
                        }
                    }
                }

                return filePaths.Select(child => new FileInfo(FileCacheManager.Instance.Get(child, false)));
            }
        }

        #endregion

        #region Public Methods and Operators

        /// <summary>
        /// The request update.
        /// </summary>
        /// <param name="directory">
        /// The directory.
        /// </param>
        /// <param name="fileKind">
        /// The file kind.
        /// </param>
        public void RequestUpdate(string directory, FileKindType fileKind)
        {
            this.directory = directory;
            this.fileKind = fileKind;

            this.RequestUpdate();
        }

        #endregion
    }
}
