﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace EffectMaker.UIControls.EffectBrowser.Controls.Basic
{
    using System;
    using System.Runtime.InteropServices;
    using System.Windows.Forms;

    using EffectMaker.Foundation.Disposables;
    using EffectMaker.UIControls.EffectBrowser.Utilities;

    /// <summary>
    /// The eb list view.
    /// </summary>
    public class EBListView : ListView
    {
        #region Constants

        /// <summary>
        /// The hd f_ sortdown.
        /// </summary>
        private const int HdfSortdown = 0x0200;

        /// <summary>
        /// The hd f_ sortup.
        /// </summary>
        private const int HdfSortup = 0x0400;

        /// <summary>
        /// The hd i_ format.
        /// </summary>
        private const int HdiFormat = 0x0004;

        /// <summary>
        /// The hd m_ getitem.
        /// </summary>
        private const int HdmGetitem = 0x1200 + 11;

        /// <summary>
        /// The hd m_ setitem.
        /// </summary>
        private const int HdmSetitem = 0x1200 + 12;

        /// <summary>
        /// The lv m_ getheader.
        /// </summary>
        private const int LvmGetheader = 0x1000 + 31;

        #endregion

        #region Fields

        /// <summary>
        /// The last top item.
        /// </summary>
        private ListViewItem lastTopItem;

        #endregion

        #region Constructors and Destructors

        /// <summary>
        /// Initializes a new instance of the <see cref="EBListView"/> class.
        /// コンストラクタ
        /// </summary>
        public EBListView()
        {
            this.FullRowSelect = true;
            this.HideSelection = false;
            this.DoubleBuffered = true;
        }

        #endregion

        #region Public Properties

        /// <summary>
        /// Gets or sets the h scroll position.
        /// </summary>
        public int HScrollPosition
        {
            get
            {
                return GetScrollPos(this.Handle, Orientation.Horizontal);
            }

            set
            {
                SetScrollPos(this.Handle, Orientation.Horizontal, value, true);
            }
        }

        /// <summary>
        /// Gets or sets the v scroll position.
        /// </summary>
        public int VScrollPosition
        {
            get
            {
                return GetScrollPos(this.Handle, Orientation.Vertical);
            }

            set
            {
                SetScrollPos(this.Handle, Orientation.Vertical, value, true);
            }
        }

        #endregion

        #region Properties

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool CanRaiseEvents
        {
            get
            {
                return !ControlEventSuppressBlock.Enabled && base.CanRaiseEvents;
            }
        }

        /// <summary>
        /// Gets or sets a value indicating whether double buffered.
        /// </summary>
        protected override sealed bool DoubleBuffered
        {
            get
            {
                return base.DoubleBuffered;
            }

            set
            {
                base.DoubleBuffered = value;
            }
        }

        /// <summary>
        /// Gets a value indicating whether is setting column header sort icon.
        /// </summary>
        protected bool IsSettingColumnHeaderSortIcon { get; private set; }

        /// <summary>
        /// Gets the last mouse button.
        /// </summary>
        protected MouseButtons LastMouseButton { get; private set; }

        #endregion

        #region Methods

        /// <summary>
        /// The on h scroll.
        /// </summary>
        /// <param name="e">
        /// イベント引数
        /// </param>
        protected virtual void OnHScroll(EventArgs e)
        {
        }

        /// <summary>
        /// The on mouse down.
        /// </summary>
        /// <param name="e">
        /// イベント引数
        /// </param>
        protected override void OnMouseDown(MouseEventArgs e)
        {
            this.LastMouseButton = e.Button;

            base.OnMouseDown(e);
        }

        /// <summary>
        /// The on top item changed.
        /// </summary>
        /// <param name="e">
        /// イベント引数
        /// </param>
        protected virtual void OnTopItemChanged(EventArgs e)
        {
        }

        /// <summary>
        /// The on v scroll.
        /// </summary>
        /// <param name="e">
        /// イベント引数
        /// </param>
        protected virtual void OnVScroll(EventArgs e)
        {
        }

        /// <summary>
        /// The set column header sort icon.
        /// </summary>
        /// <param name="columnIndex">
        /// The column index.
        /// </param>
        /// <param name="order">
        /// The order.
        /// </param>
        protected void SetColumnHeaderSortIcon(int columnIndex, SortOrder order)
        {
            using (new AnonymousDisposable(() => this.IsSettingColumnHeaderSortIcon = false))
            {
                this.IsSettingColumnHeaderSortIcon = true;

                var header = SendMessage(this.Handle, LvmGetheader, IntPtr.Zero, IntPtr.Zero);
                var item = new Hditem { Mask = HdiFormat };

                SendMessage(header, HdmGetitem, new IntPtr(columnIndex), ref item);

                if (order == SortOrder.Ascending)
                {
                    item.Fmt &= ~HdfSortdown;
                    item.Fmt |= HdfSortup;
                }
                else if (order == SortOrder.Descending)
                {
                    item.Fmt &= ~HdfSortup;
                    item.Fmt |= HdfSortdown;
                }
                else if (order == SortOrder.None)
                {
                    item.Fmt &= ~HdfSortdown & ~HdfSortup;
                }

                SendMessage(header, HdmSetitem, new IntPtr(columnIndex), ref item);
            }
        }

        /// <summary>
        /// The wnd proc.
        /// </summary>
        /// <param name="m">
        /// The m.
        /// </param>
        protected override void WndProc(ref Message m)
        {
            const int LvnEndscroll = 0x204e;
            const int WmHscroll = 0x114;
            const int WmVscroll = 0x115;

            switch (m.Msg)
            {
                    // Trap LVN_ENDSCROLL, delivered with a WM_REFLECT + WM_NOTIFY message
                    // http://stackoverflow.com/questions/8792097/winforms-listview-topitem-changed-event
                case LvnEndscroll:
                    try
                    {
                        var notify = (Nmhdr)Marshal.PtrToStructure(m.LParam, typeof(Nmhdr));
                        if (notify.Code == -181 && this.TopItem != null && !this.TopItem.Equals(this.lastTopItem))
                        {
                            this.OnTopItemChanged(EventArgs.Empty);
                            this.lastTopItem = this.TopItem;
                        }
                    }
                    catch
                    {
                        // TopItemが見つからない旨の例外が出るが、無視して良さそうなので握りつぶす。
                    }

                    break;

                case WmHscroll:
                    this.OnHScroll(EventArgs.Empty);
                    break;

                case WmVscroll:
                    this.OnVScroll(EventArgs.Empty);
                    break;
            }

            base.WndProc(ref m);
        }

        /// <summary>
        /// The get scroll pos.
        /// </summary>
        /// <param name="hwnd">
        /// The hwnd.
        /// </param>
        /// <param name="nbar">
        /// The n bar.
        /// </param>
        /// <returns>
        /// The <see cref="int"/>.
        /// </returns>
        [DllImport("user32.dll")]
        private static extern int GetScrollPos(IntPtr hwnd, Orientation nbar);

        // ReSharper restore InconsistentNaming
        // ReSharper restore FieldCanBeMadeReadOnly.Local
        // ReSharper restore MemberCanBePrivate.Local

        /// <summary>
        /// The send message.
        /// </summary>
        /// <param name="handle">
        /// The handle.
        /// </param>
        /// <param name="msg">
        /// The msg.
        /// </param>
        /// <param name="wparam">
        /// The w param.
        /// </param>
        /// <param name="lparam">
        /// The l param.
        /// </param>
        /// <returns>
        /// The <see cref="IntPtr"/>.
        /// </returns>
        [DllImport("user32.dll")]
        private static extern IntPtr SendMessage(IntPtr handle, uint msg, IntPtr wparam, IntPtr lparam);

        /// <summary>
        /// The send message.
        /// </summary>
        /// <param name="handle">
        /// The handle.
        /// </param>
        /// <param name="msg">
        /// The msg.
        /// </param>
        /// <param name="wparam">
        /// The w param.
        /// </param>
        /// <param name="lparam">
        /// The l param.
        /// </param>
        /// <returns>
        /// The <see cref="IntPtr"/>.
        /// </returns>
        [DllImport("user32.dll")]
        private static extern IntPtr SendMessage(IntPtr handle, int msg, IntPtr wparam, ref Hditem lparam);

        /// <summary>
        /// The set scroll pos.
        /// </summary>
        /// <param name="hwnd">
        /// The h wnd.
        /// </param>
        /// <param name="nbar">
        /// The n bar.
        /// </param>
        /// <param name="npos">
        /// The n pos.
        /// </param>
        /// <param name="bredraw">
        /// The b redraw.
        /// </param>
        /// <returns>
        /// The <see cref="int"/>.
        /// </returns>
        [DllImport("user32.dll")]
        private static extern int SetScrollPos(IntPtr hwnd, Orientation nbar, int npos, bool bredraw);

        #endregion

        /// <summary>
        /// The hditem.
        /// </summary>
        [StructLayout(LayoutKind.Sequential)]
        private struct Hditem
        {
            /// <summary>
            /// The Mask.
            /// </summary>
            public int Mask;

            /// <summary>
            /// The Cxy.
            /// </summary>
            public int Cxy;

            /// <summary>
            /// The psz text.
            /// </summary>
            [MarshalAs(UnmanagedType.LPTStr)]
            public string PszText;

            /// <summary>
            /// The Hbm.
            /// </summary>
            public IntPtr Hbm;

            /// <summary>
            /// The cch text max.
            /// </summary>
            public int CchTextMax;

            /// <summary>
            /// The Fmt.
            /// </summary>
            public int Fmt;

            /// <summary>
            /// The l param.
            /// </summary>
            public int Lparam;

            /// <summary>
            /// The i image.
            /// </summary>
            public int Iimage;

            /// <summary>
            /// The i order.
            /// </summary>
            public int Iorder;
        }

        /// <summary>
        /// The nmhdr.
        /// </summary>
        [StructLayout(LayoutKind.Sequential)]
        private struct Nmhdr
        {
            /// <summary>
            /// The hwnd from.
            /// </summary>
            public readonly IntPtr HwndFrom;

            /// <summary>
            /// The id from.
            /// </summary>
            public readonly IntPtr IdFrom;

            /// <summary>
            /// The Code.
            /// </summary>
            public readonly int Code;
        }
    }
}
