﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace EffectMaker.UIControls.EffectBrowser.Controls.AddressBar.DirBreadcrumbList
{
    using System;
    using System.Drawing;
    using System.Windows.Forms;

    using EffectMaker.UIControls.EffectBrowser.Data;
    using EffectMaker.UIControls.EffectBrowser.Utilities;

    /// <summary>
    /// The eb dir breadcrumb list item.
    /// </summary>
    public class EBDirBreadcrumbListItem : Control
    {
        #region Fields

        /// <summary>
        /// The enabled.
        /// </summary>
        private bool enabled = true;

        /// <summary>
        /// The state.
        /// </summary>
        private StateKind state;

        #endregion

        #region Constructors and Destructors

        /// <summary>
        /// Initializes a new instance of the <see cref="EBDirBreadcrumbListItem"/> class.
        /// </summary>
        public EBDirBreadcrumbListItem()
        {
            this.ResizeRedraw = true;
            this.DoubleBuffered = true;
        }

        #endregion

        #region Enums

        /// <summary>
        /// The state kind.
        /// </summary>
        public enum StateKind
        {
            /// <summary>
            /// The wait.
            /// </summary>
            Wait,

            /// <summary>
            /// The mouse over gray.
            /// </summary>
            MouseOverGray,

            /// <summary>
            /// The mouse over.
            /// </summary>
            MouseOver,

            /// <summary>
            /// The mouse down.
            /// </summary>
            MouseDown
        }

        #endregion

        #region Public Properties

        /// <summary>
        /// Gets or sets a value indicating whether enabled.
        /// </summary>
        public new bool Enabled
        {
            get
            {
                return this.enabled;
            }

            set
            {
                if (value == this.enabled)
                {
                    return;
                }

                this.enabled = value;
                this.Invalidate();
            }
        }

        /// <summary>
        /// Gets or sets the state.
        /// </summary>
        public StateKind State
        {
            get
            {
                return this.state;
            }

            set
            {
                if (this.state == value)
                {
                    return;
                }

                this.state = value;
                this.Invalidate();
            }
        }

        #endregion

        #region Properties

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool CanRaiseEvents
        {
            get
            {
                return !ControlEventSuppressBlock.Enabled && base.CanRaiseEvents;
            }
        }

        /// <summary>
        /// Gets or sets a value indicating whether double buffered.
        /// </summary>
        protected override sealed bool DoubleBuffered
        {
            get
            {
                return base.DoubleBuffered;
            }

            set
            {
                base.DoubleBuffered = value;
            }
        }

        #endregion

        #region Methods

        /// <summary>
        /// The on paint.
        /// </summary>
        /// <param name="e">
        /// イベント引数
        /// </param>
        protected override void OnPaint(PaintEventArgs e)
        {
            base.OnPaint(e);

            if (this.Enabled)
            {
                DrawBackground(e.Graphics, this.State, e.ClipRectangle);
            }
            else
            {
                e.Graphics.Clear(Color.White);
            }
        }

        /// <summary>
        /// The draw background.
        /// </summary>
        /// <param name="g">
        /// The g.
        /// </param>
        /// <param name="state">
        /// The state.
        /// </param>
        /// <param name="rect">
        /// The rect.
        /// </param>
        /// <exception cref="NotSupportedException">
        /// NotSupportedException
        /// </exception>
        private static void DrawBackground(Graphics g, StateKind state, Rectangle rect)
        {
            int srcOffsetX;
            {
                switch (state)
                {
                    case StateKind.Wait:
                        g.Clear(Color.White);
                        return;

                    case StateKind.MouseDown:
                        srcOffsetX = 0;
                        break;

                    case StateKind.MouseOverGray:
                        srcOffsetX = 6;
                        break;

                    case StateKind.MouseOver:
                        srcOffsetX = 12;
                        break;

                    default:
                        throw new NotSupportedException();
                }
            }

            var rectSrc = new Rectangle();
            var rectDst = new Rectangle();

            // Upper-left
            rectSrc.X = srcOffsetX;
            rectSrc.Y = 0;
            rectSrc.Width = 3;
            rectSrc.Height = 2;
            rectDst.X = 0;
            rectDst.Y = 0;
            rectDst.Width = 3;
            rectDst.Height = 2;
            rectDst.Offset(rect.Location);
            g.DrawImage(Constants.BreadcrumbCtrlItemImg, rectDst, rectSrc, GraphicsUnit.Pixel);

            // Left
            rectSrc.X = srcOffsetX;
            rectSrc.Y = 2;
            rectSrc.Width = 3;
            rectSrc.Height = 17;
            rectDst.X = 0;
            rectDst.Y = 2;
            rectDst.Width = 3;
            rectDst.Height = rect.Height - 3;
            rectDst.Offset(rect.Location);
            g.DrawImage(Constants.BreadcrumbCtrlItemImg, rectDst, rectSrc, GraphicsUnit.Pixel);

            // Bottom-left
            rectSrc.X = srcOffsetX;
            rectSrc.Y = 19;
            rectSrc.Width = 3;
            rectSrc.Height = 1;
            rectDst.X = 0;
            rectDst.Y = rect.Height - 1;
            rectDst.Width = 3;
            rectDst.Height = 1;
            rectDst.Offset(rect.Location);
            g.DrawImage(Constants.BreadcrumbCtrlItemImg, rectDst, rectSrc, GraphicsUnit.Pixel);

            // Upper-center
            rectSrc.X = srcOffsetX + 3;
            rectSrc.Y = 0;
            rectSrc.Width = 1;
            rectSrc.Height = 2;
            rectDst.X = 3;
            rectDst.Y = 0;
            rectDst.Width = rect.Width - 4;
            rectDst.Height = 2;
            rectDst.Offset(rect.Location);
            g.DrawImage(Constants.BreadcrumbCtrlItemImg, rectDst, rectSrc, GraphicsUnit.Pixel);

            // Center
            rectSrc.X = srcOffsetX + 3;
            rectSrc.Y = 2;
            rectSrc.Width = 1;
            rectSrc.Height = 17;
            rectDst.X = 3;
            rectDst.Y = 2;
            rectDst.Width = rect.Width - 4;
            rectDst.Height = rect.Height - 3;
            rectDst.Offset(rect.Location);
            g.DrawImage(Constants.BreadcrumbCtrlItemImg, rectDst, rectSrc, GraphicsUnit.Pixel);

            // Bottom-center
            rectSrc.X = srcOffsetX + 3;
            rectSrc.Y = 19;
            rectSrc.Width = 1;
            rectSrc.Height = 1;
            rectDst.X = 3;
            rectDst.Y = rect.Height - 1;
            rectDst.Width = rect.Width - 4;
            rectDst.Height = 1;
            rectDst.Offset(rect.Location);
            g.DrawImage(Constants.BreadcrumbCtrlItemImg, rectDst, rectSrc, GraphicsUnit.Pixel);

            // Upper-right
            rectSrc.X = srcOffsetX + 4;
            rectSrc.Y = 0;
            rectSrc.Width = 2;
            rectSrc.Height = 2;
            rectDst.X = rect.Width - 2;
            rectDst.Y = 0;
            rectDst.Width = 2;
            rectDst.Height = 2;
            rectDst.Offset(rect.Location);
            g.DrawImage(Constants.BreadcrumbCtrlItemImg, rectDst, rectSrc, GraphicsUnit.Pixel);

            // Right
            rectSrc.X = srcOffsetX + 4;
            rectSrc.Y = 2;
            rectSrc.Width = 2;
            rectSrc.Height = 17;
            rectDst.X = rect.Width - 2;
            rectDst.Y = 2;
            rectDst.Width = 2;
            rectDst.Height = rect.Height - 3;
            rectDst.Offset(rect.Location);
            g.DrawImage(Constants.BreadcrumbCtrlItemImg, rectDst, rectSrc, GraphicsUnit.Pixel);

            // Bottom-right
            rectSrc.X = srcOffsetX + 4;
            rectSrc.Y = 19;
            rectSrc.Width = 2;
            rectSrc.Height = 1;
            rectDst.X = rect.Width - 2;
            rectDst.Y = rect.Height - 1;
            rectDst.Width = 2;
            rectDst.Height = 1;
            rectDst.Offset(rect.Location);
            g.DrawImage(Constants.BreadcrumbCtrlItemImg, rectDst, rectSrc, GraphicsUnit.Pixel);
        }

        #endregion
    }
}
