﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace EffectMaker.UIControls.EffectBrowser.Controls.AddressBar.DirBreadcrumbList
{
    using System;
    using System.ComponentModel;
    using System.Diagnostics;
    using System.Drawing;
    using System.Linq;
    using System.Windows.Forms;

    using EffectMaker.Foundation.Extensions;
    using EffectMaker.UIControls.EffectBrowser.Data;
    using EffectMaker.UIControls.EffectBrowser.Utilities;

    /// <summary>
    /// The eb dir breadcrumb list arrow.
    /// </summary>
    public abstract class EBDirBreadcrumbListArrow : EBDirBreadcrumbListItem
    {
        #region Fields

        /// <summary>
        /// The last closed.
        /// </summary>
        private readonly Stopwatch lastClosed = new Stopwatch();

        /// <summary>
        /// The dir list.
        /// </summary>
        private ContextMenuStrip dirList;

        /// <summary>
        /// The directory.
        /// </summary>
        private string directory = string.Empty;

        /// <summary>
        /// The is mouse over.
        /// </summary>
        private bool isMouseOver;

        /// <summary>
        /// The is popupped.
        /// </summary>
        private bool isPopupped;

        #endregion

        #region Constructors and Destructors

        /// <summary>
        /// Initializes a new instance of the <see cref="EBDirBreadcrumbListArrow"/> class.
        /// </summary>
        protected EBDirBreadcrumbListArrow()
        {
            this.Width = 15;
        }

        #endregion

        #region Public Events

        /// <summary>
        /// The popup changed.
        /// </summary>
        public event EventHandler PopupChanged;

        #endregion

        #region Enums

        /// <summary>
        /// The popup type kind.
        /// </summary>
        protected enum PopupTypeKind
        {
            /// <summary>
            /// The normal.
            /// </summary>
            Normal,

            /// <summary>
            /// The wide.
            /// </summary>
            Wide
        }

        #endregion

        #region Public Properties

        /// <summary>
        /// Gets or sets the directory.
        /// </summary>
        [Browsable(false)]
        public string Directory
        {
            get
            {
                return this.directory;
            }

            set
            {
                if (this.directory == value)
                {
                    return;
                }

                this.directory = value;

                this.Visible = PathUtility.EnumerateDirectories(this.directory).Any();
                if (this.Visible == false)
                {
                    this.Width = 0;
                }
            }
        }

        /// <summary>
        /// Gets or sets a value indicating whether is popupped.
        /// </summary>
        [Browsable(false)]
        public bool IsPopupped
        {
            get
            {
                return this.isPopupped;
            }

            set
            {
                if (this.isPopupped == value)
                {
                    return;
                }

                this.isPopupped = value;

                if (this.PopupChanged != null)
                {
                    this.PopupChanged(this, EventArgs.Empty);
                }

                this.Invalidate();
            }
        }

        #endregion

        #region Properties

        /// <summary>
        /// The popup type.
        /// </summary>
        protected PopupTypeKind PopupType { get; set; }

        /// <summary>
        /// The force opend arrow.
        /// </summary>
        protected bool ForceOpendArrow { get; set; }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool CanRaiseEvents
        {
            get
            {
                return !ControlEventSuppressBlock.Enabled && base.CanRaiseEvents;
            }
        }

        #endregion

        #region Methods

        /// <summary>
        /// The create dir list.
        /// </summary>
        /// <param name="param">
        /// The param.
        /// </param>
        /// <returns>
        /// The <see cref="ContextMenuStrip"/>.
        /// </returns>
        protected abstract ContextMenuStrip CreateDirList(object param);

        /// <summary>
        /// The on mouse down.
        /// </summary>
        /// <param name="e">
        /// イベント引数
        /// </param>
        protected override void OnMouseDown(MouseEventArgs e)
        {
            base.OnMouseDown(e);

            if (this.Enabled)
            {
                if (e.Button == MouseButtons.Left)
                {
                    Application.DoEvents();
                    this.PopupMenu();
                }
            }
        }

        /// <summary>
        /// The on mouse enter.
        /// </summary>
        /// <param name="e">
        /// イベント引数
        /// </param>
        protected override void OnMouseEnter(EventArgs e)
        {
            base.OnMouseEnter(e);
            this.isMouseOver = true;
        }

        /// <summary>
        /// The on mouse leave.
        /// </summary>
        /// <param name="e">
        /// イベント引数
        /// </param>
        protected override void OnMouseLeave(EventArgs e)
        {
            base.OnMouseLeave(e);
            this.isMouseOver = false;
        }

        /// <summary>
        /// The on paint.
        /// </summary>
        /// <param name="e">
        /// イベント引数
        /// </param>
        protected override void OnPaint(PaintEventArgs e)
        {
            base.OnPaint(e);

            this.DrawArrow(e.Graphics, this.ForceOpendArrow || this.IsPopupped);
        }

        /// <summary>
        /// The draw arrow.
        /// </summary>
        /// <param name="g">
        /// The g.
        /// </param>
        /// <param name="isOpen">
        /// The is open.
        /// </param>
        private void DrawArrow(Graphics g, bool isOpen)
        {
            var dst = new Rectangle();
            var src = new Rectangle();
            {
                int offsetY;
                if (isOpen)
                {
                    src.X = 6;
                    src.Y = 21;
                    src.Width = 8;
                    src.Height = 7;
                    offsetY = 2;
                }
                else
                {
                    src.X = 0;
                    src.Y = 20;
                    src.Width = 5;
                    src.Height = 8;
                    offsetY = 1;
                }

                dst.X = (this.Width - src.Width) / 2;
                dst.Y = ((this.Height - src.Height) / 2) + offsetY;
                dst.Width = src.Width;
                dst.Height = src.Height;
            }

            if (this.Enabled == false)
            {
                src.X += 14;
            }

            g.DrawImage(Constants.BreadcrumbCtrlItemImg, dst, src, GraphicsUnit.Pixel);
        }

        /// <summary>
        /// The popup menu.
        /// </summary>
        private void PopupMenu()
        {
            if (this.dirList == null)
            {
                this.dirList = this.CreateDirList(this.Directory);
                if (this.dirList == null)
                {
                    return;
                }

                this.dirList.Closed += (ss, ee) =>
                    {
                        this.IsPopupped = false;
                        this.dirList = null;

                        this.lastClosed.Restart();
                    };

                this.dirList.Opening += (ss, ee) =>
                    {
                        // ワイド指定はコンボボックス風に設定
                        if (this.PopupType == PopupTypeKind.Wide)
                        {
                            var height = this.dirList.Height;

                            this.dirList.AutoSize = false;
                            this.dirList.Width = this.Parent.Width;
                            this.dirList.Height = height;

                            this.dirList.Items.OfType<ToolStripItem>().ForEach(
                                x =>
                                    {
                                        x.AutoSize = false;
                                        x.Width = this.dirList.Width;
                                    });
                        }
                    };
            }

            if (this.isMouseOver && this.lastClosed.IsRunning && (this.lastClosed.ElapsedMilliseconds < 20))
            {
                this.dirList.Close();
                this.lastClosed.Stop();
                this.dirList.Dispose();
                this.dirList = null;
                return;
            }

            var baseControl = (this.PopupType == PopupTypeKind.Wide) ? this.Parent : this;
            this.dirList.Show(baseControl, 0, baseControl.Height);

            this.IsPopupped = true;
        }

        #endregion
    }
}
