﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Windows.Forms;
using System.Xml.Serialization;

using EffectMaker.Foundation.Serialization;

using EffectMaker.UIControls.EffectBrowser.Controls.FileListView.Base;

namespace EffectMaker.UIControls.EffectBrowser.Config
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Windows.Forms;
    using System.Xml.Serialization;

    using EffectMaker.UIControls.EffectBrowser.Controls.FileListView.Base;

    /// <summary>
    /// The setting.
    /// </summary>
    [Serializable]
    public class Setting : IXmlDocSerializable
    {
        #region Constructors and Destructors

        /// <summary>
        /// Initializes a new instance of the <see cref="Setting"/> class.
        /// </summary>
        public Setting()
        {
            this.FileTreeView = new FileTreeView();
            this.FileListView = new FileListView();
            this.History = new History();
        }

        #endregion

        #region Public Properties

        /// <summary>
        /// Gets or sets the file list view.
        /// </summary>
        public FileListView FileListView { get; set; }

        /// <summary>
        /// Gets or sets the file tree view.
        /// </summary>
        public FileTreeView FileTreeView { get; set; }

        /// <summary>
        /// Gets or sets the history.
        /// </summary>
        public History History { get; set; }

        #endregion

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public bool ReadXml(XmlDocSerializationContext context)
        {
            this.FileTreeView = this.ReadElement(context, "FileTreeView", this.FileTreeView);
            this.FileListView = this.ReadElement(context, "FileListView", this.FileListView);
            this.History = this.ReadElement(context, "History", this.History);
            return true;
        }

        /// <summary>
        /// Serializes this object to a XML node.
        /// </summary>
        /// <param name="context">The data context needed for the serialization.</param>
        /// <returns>True on success.</returns>
        public bool WriteXml(XmlDocSerializationContext context)
        {
            this.WriteElement(context, "FileTreeView", this.FileTreeView);
            this.WriteElement(context, "FileListView", this.FileListView);
            this.WriteElement(context, "History", this.History);
            return true;
        }
    }

    /// <summary>
    /// The file tree view.
    /// </summary>
    [Serializable]
    public class FileTreeView : IXmlDocSerializable
    {
        #region Constructors and Destructors

        /// <summary>
        /// Initializes a new instance of the <see cref="FileTreeView"/> class.
        /// </summary>
        public FileTreeView()
        {
            this.Width = 280;
            this.DesktopExpanded = false;
            this.DocumentsExpanded = false;
            this.ComputerExpanded = false;
        }

        #endregion

        #region Public Properties

        /// <summary>
        /// Gets or sets a value indicating whether computer expanded.
        /// </summary>
        [XmlAttribute]
        public bool ComputerExpanded { get; set; }

        /// <summary>
        /// Gets or sets a value indicating whether desktop expanded.
        /// </summary>
        [XmlAttribute]
        public bool DesktopExpanded { get; set; }

        /// <summary>
        /// Gets or sets a value indicating whether documents expanded.
        /// </summary>
        [XmlAttribute]
        public bool DocumentsExpanded { get; set; }

        /// <summary>
        /// Gets or sets the width.
        /// </summary>
        [XmlAttribute]
        public int Width { get; set; }

        /// <summary>
        /// Gets or sets the last directory in the computer tab.
        /// </summary>
        [XmlAttribute]
        public string ComputerDirectory { get; set; }

        /// <summary>
        /// Gets or sets the last directory in the favorites tab.
        /// </summary>
        [XmlAttribute]
        public string FavoritesDirectory { get; set; }

        /// <summary>
        /// Gets or sets the last opened tab.
        /// </summary>
        [XmlAttribute]
        public int LastTab { get; set; }

        #endregion

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public bool ReadXml(XmlDocSerializationContext context)
        {
            this.Width = this.ReadAttribute(context, "Width", this.Width);
            this.DesktopExpanded = this.ReadAttribute(context, "DesktopExpanded", this.DesktopExpanded);
            this.DocumentsExpanded = this.ReadAttribute(context, "DocumentsExpanded", this.DocumentsExpanded);
            this.ComputerExpanded = this.ReadAttribute(context, "ComputerExpanded", this.ComputerExpanded);
            this.ComputerDirectory = this.ReadAttribute(context, "ComputerDirectory", this.ComputerDirectory);
            this.FavoritesDirectory = this.ReadAttribute(context, "FavoritesDirectory", this.FavoritesDirectory);
            this.LastTab = this.ReadAttribute(context, "LastTab", this.LastTab);
            return true;
        }

        /// <summary>
        /// Serializes this object to a XML node.
        /// </summary>
        /// <param name="context">The data context needed for the serialization.</param>
        /// <returns>True on success.</returns>
        public bool WriteXml(XmlDocSerializationContext context)
        {
            this.WriteAttribute(context, "Width", this.Width);
            this.WriteAttribute(context, "DesktopExpanded", this.DesktopExpanded);
            this.WriteAttribute(context, "DocumentsExpanded", this.DocumentsExpanded);
            this.WriteAttribute(context, "ComputerExpanded", this.ComputerExpanded);
            this.WriteAttribute(context, "ComputerDirectory", this.ComputerDirectory);
            this.WriteAttribute(context, "FavoritesDirectory", this.FavoritesDirectory);
            this.WriteAttribute(context, "LastTab", this.LastTab);
            return true;
        }
    }

    /// <summary>
    /// The file list view.
    /// </summary>
    [Serializable]
    public class FileListView : IXmlDocSerializable
    {
        #region Constructors and Destructors

        /// <summary>
        /// Initializes a new instance of the <see cref="FileListView"/> class.
        /// </summary>
        public FileListView()
        {
            this.Columns = new List<FileListViewColumn>
                               {
                                   new FileListViewColumn
                                       {
                                           Name =
                                               EBFileListView.ColumnKinds
                                               .Name.ToString(),
                                           DisplayIndex = 0,
                                           Width = 200
                                       },
                                   new FileListViewColumn
                                       {
                                           Name =
                                               EBFileListView.ColumnKinds
                                               .UpdateTimestamp.ToString(),
                                           DisplayIndex = 1,
                                           Width = 120
                                       },
                                   new FileListViewColumn
                                       {
                                           Name =
                                               EBFileListView.ColumnKinds
                                               .CreateTimestamp.ToString(),
                                           DisplayIndex = 2,
                                           Width = 80
                                       },
                                   new FileListViewColumn
                                       {
                                           Name =
                                               EBFileListView.ColumnKinds
                                               .FileKind.ToString(),
                                           DisplayIndex = 3,
                                           Width = 120
                                       },
                                   new FileListViewColumn
                                       {
                                           Name =
                                               EBFileListView.ColumnKinds
                                               .ByteSize.ToString(),
                                           DisplayIndex = 4,
                                           Width = 70
                                       },
                                   new FileListViewColumn
                                       {
                                           Name =
                                               EBFileListView.ColumnKinds
                                               .LabelColor.ToString(),
                                           DisplayIndex = 5,
                                           Width = 90
                                       },
                                   new FileListViewColumn
                                       {
                                           Name =
                                               EBFileListView.ColumnKinds
                                               .Comment.ToString(),
                                           DisplayIndex = 6,
                                           Width = 200
                                       }
                               };

            this.TexturesViewMode = true;
            this.PrimitivesViewMode = true;
            this.CombinersViewMode = false;
            this.OtherViewMode = false;
        }

        #endregion

        #region Public Properties

        /// <summary>
        /// Gets or sets the columns.
        /// </summary>
        public List<FileListViewColumn> Columns { get; set; }

        /// <summary>
        /// Gets or Sets the textures view mode;
        /// </summary>
        public bool TexturesViewMode { get; set; }

        /// <summary>
        /// Gets or Sets the primitives view mode;
        /// </summary>
        public bool PrimitivesViewMode { get; set; }

        /// <summary>
        /// Gets or Sets the combiners view mode;
        /// </summary>
        public bool CombinersViewMode { get; set; }

        /// <summary>
        /// Gets or Sets the other files view mode;
        /// </summary>
        public bool OtherViewMode { get; set; }

        #endregion

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public bool ReadXml(XmlDocSerializationContext context)
        {
            this.Columns.Clear();
            this.Columns = this.ReadListElement(context, "Columns", this.Columns);
            this.TexturesViewMode = this.ReadAttribute(context, "TexturesViewMode", this.TexturesViewMode);
            this.PrimitivesViewMode = this.ReadAttribute(context, "PrimitivesViewMode", this.PrimitivesViewMode);
            this.CombinersViewMode = this.ReadAttribute(context, "CombinersViewMode", this.CombinersViewMode);
            this.OtherViewMode = this.ReadAttribute(context, "OtherViewMode", this.OtherViewMode);
            return true;
        }

        /// <summary>
        /// Serializes this object to a XML node.
        /// </summary>
        /// <param name="context">The data context needed for the serialization.</param>
        /// <returns>True on success.</returns>
        public bool WriteXml(XmlDocSerializationContext context)
        {
            this.WriteEnumerableElement(context, "Columns", this.Columns);
            this.WriteAttribute(context, "TexturesViewMode", this.TexturesViewMode);
            this.WriteAttribute(context, "PrimitivesViewMode", this.PrimitivesViewMode);
            this.WriteAttribute(context, "CombinersViewMode", this.CombinersViewMode);
            this.WriteAttribute(context, "OtherViewMode", this.OtherViewMode);
            return true;
        }
    }

    /// <summary>
    /// The file list view column.
    /// </summary>
    [Serializable]
    public class FileListViewColumn : IXmlDocSerializable
    {
        #region Constructors and Destructors

        /// <summary>
        /// Initializes a new instance of the <see cref="FileListViewColumn"/> class.
        /// </summary>
        public FileListViewColumn()
        {
            this.Visible = true;
            this.SortOrder = SortOrder.Ascending;
        }

        #endregion

        #region Public Properties

        /// <summary>
        /// Gets the column kinds.
        /// </summary>
        [XmlIgnore]
        public EBFileListView.ColumnKinds ColumnKinds
        {
            get
            {
                return
                    Enum.GetValues(typeof(EBFileListView.ColumnKinds))
                        .Cast<EBFileListView.ColumnKinds>()
                        .FirstOrDefault(e => e.ToString() == this.Name);
            }
        }

        /// <summary>
        /// Gets or sets the display index.
        /// </summary>
        [XmlAttribute]
        public int DisplayIndex { get; set; }

        /// <summary>
        /// Gets or sets the name.
        /// </summary>
        [XmlAttribute]
        public string Name { get; set; }

        /// <summary>
        /// Gets or sets the sort order.
        /// </summary>
        [XmlAttribute]
        public SortOrder SortOrder { get; set; }

        /// <summary>
        /// Gets or sets a value indicating whether visible.
        /// </summary>
        [XmlAttribute]
        public bool Visible { get; set; }

        /// <summary>
        /// Gets or sets the width.
        /// </summary>
        [XmlAttribute]
        public int Width { get; set; }

        #endregion

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public bool ReadXml(XmlDocSerializationContext context)
        {
            this.Name = this.ReadAttribute(context, "Name", this.Name);
            this.DisplayIndex = this.ReadAttribute(context, "DisplayIndex", this.DisplayIndex);
            this.Width = this.ReadAttribute(context, "Width", this.Width);
            this.Visible = this.ReadAttribute(context, "Visible", this.Visible);
            this.SortOrder = this.ReadAttribute(context, "SortOrder", this.SortOrder);
            return true;
        }

        /// <summary>
        /// Serializes this object to a XML node.
        /// </summary>
        /// <param name="context">The data context needed for the serialization.</param>
        /// <returns>True on success.</returns>
        public bool WriteXml(XmlDocSerializationContext context)
        {
            this.WriteAttribute(context, "Name", this.Name);
            this.WriteAttribute(context, "DisplayIndex", this.DisplayIndex);
            this.WriteAttribute(context, "Width", this.Width);
            this.WriteAttribute(context, "Visible", this.Visible);
            this.WriteAttribute(context, "SortOrder", this.SortOrder);
            return true;
        }
    }

    /// <summary>
    /// The history.
    /// </summary>
    [Serializable]
    public class History : IXmlDocSerializable
    {
        #region Constructors and Destructors

        /// <summary>
        /// Initializes a new instance of the <see cref="History"/> class.
        /// </summary>
        public History()
        {
            this.Paths = new List<string>();
        }

        #endregion

        #region Public Properties

        /// <summary>
        /// Gets or sets the paths.
        /// </summary>
        [XmlElement(ElementName = "Path", Type = typeof(string))]
        public List<string> Paths { get; set; }

        #endregion

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public bool ReadXml(XmlDocSerializationContext context)
        {
            this.Paths = this.ReadElementsByTagName<string>(context, "Path").ToList();
            return true;
        }

        /// <summary>
        /// Serializes this object to a XML node.
        /// </summary>
        /// <param name="context">The data context needed for the serialization.</param>
        /// <returns>True on success.</returns>
        public bool WriteXml(XmlDocSerializationContext context)
        {
            foreach (string path in this.Paths)
            {
                this.WriteElement(context, "Path", path);
            }

            return true;
        }
    }
}
