﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Runtime.InteropServices;
using System.Text;

namespace EffectMaker.UIDialogs.EmbededViewerDialog
{
    public class HwndPresenterViewModel : INotifyPropertyChanged
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public HwndPresenterViewModel()
        {
        }

        public event PropertyChangedEventHandler PropertyChanged;

        /// <summary>
        /// ViewerProcess プロパティのバッキングフィールドです。
        /// </summary>
        private Process viewerProcess;

        /// <summary>
        /// ビューアのプロセスを取得または設定します。
        /// </summary>
        public Process ViewerProcess
        {
            get
            {
                return this.viewerProcess;
            }

            set
            {
                this.viewerProcess = value;

                if (this.viewerProcess == null)
                {
                    this.WindowHandle = IntPtr.Zero;
                    return;
                }

                // プロセス起動直後はメインウィンドウが作られていなかったり、ログウィンドウが作られて
                // レンダリングウィンドウが作られていない場合があるため、レンダリングウィンドウが取得できるまで待つ
                for (int i = 0; i < 300; i++)
                {
                    IntPtr windowHandle = this.GetRenderingWindowHandle(this.viewerProcess);

                    if (windowHandle != IntPtr.Zero)
                    {
                        this.WindowHandle = this.viewerProcess.MainWindowHandle;
                        return;
                    }

                    System.Threading.Thread.Sleep(10);
                }
            }
        }

        /// <summary>
        /// WindowHandle プロパティのバッキングフィールドです。
        /// </summary>
        private IntPtr windowHandle = IntPtr.Zero;

        /// <summary>
        /// ビューアのウィンドウハンドルを取得または設定します。
        /// </summary>
        public IntPtr WindowHandle
        {
            get
            {
                return this.windowHandle;
            }

            set
            {
                if (this.windowHandle != value)
                {
                    this.windowHandle = value;
                    this.PropertyChanged?.Invoke(this, new PropertyChangedEventArgs(nameof(WindowHandle)));
                }
            }
        }

        /// <summary>
        /// ビューアのプロセスを KILL します。
        /// </summary>
        public void KillViewer()
        {
            if (this.viewerProcess == null)
            {
                return;
            }

            if (this.viewerProcess.HasExited)
            {
                return;
            }

            this.viewerProcess.Kill();

            this.viewerProcess.WaitForExit();
            this.viewerProcess = null;

            this.WindowHandle = IntPtr.Zero;
        }

        #region GetRenderingWindowHandle Method

        /// <summary>
        /// ビューアのレンダリングウィンドウを取得します。
        /// </summary>
        /// <param name="process">ビューアのプロセス</param>
        /// <returns>レンダリングウィンドウのウィンドウハンドルを返します。</returns>
        private IntPtr GetRenderingWindowHandle(Process process)
        {
            List<IntPtr> viewerWindowHandles = new List<IntPtr>();

            // ビューアのウィンドウを列挙するローカル関数
            EnumWindowsDelegate enumFunc = (IntPtr hWnd, IntPtr lParam) =>
            {
                int processId;

                GetWindowThreadProcessId(hWnd, out processId);

                if (processId == process.Id)
                {
                    viewerWindowHandles.Add(hWnd);
                }

                return true;
            };

            // ビューアウィンドウを列挙する
            EnumWindows(enumFunc, IntPtr.Zero);

            IntPtr renderingWindowHandle = IntPtr.Zero;

            // レンダリングウィンドウを検索する
            foreach (IntPtr windowHandle in viewerWindowHandles)
            {
                int windowTextLength = GetWindowTextLength(windowHandle);
                StringBuilder sb = new StringBuilder(windowTextLength + 1);

                // ウィンドウタイトルを取得
                GetWindowText(windowHandle, sb, sb.Capacity);

                // ウィンドウタイトルが "EffectViewer" から始まるものをレンダリングウィンドウとして判定する
                if (sb.ToString().StartsWith("EffectViewer"))
                {
                    renderingWindowHandle = windowHandle;
                    break;
                }
            }

            return renderingWindowHandle;
        }

        /// <summary>
        /// BOOL WNDENUMPROC(HWND hWnd, LPARAM lParam)
        /// </summary>
        /// <param name="hWnd">hWnd</param>
        /// <param name="lParam">lParam</param>
        /// <returns>BOOL</returns>
        private delegate bool EnumWindowsDelegate(IntPtr hWnd, IntPtr lParam);

        /// <summary>
        /// int GetWindowTextLength(HWND hWnd)
        /// </summary>
        /// <param name="hWnd">hWnd</param>
        /// <returns>int</returns>
        [DllImport("user32.dll", CharSet = CharSet.Auto, SetLastError = true)]
        private static extern int GetWindowTextLength(IntPtr hWnd);

        /// <summary>
        /// int GetWindowText(HWND hWnd, LPTSTR lpString, int nMaxCount)
        /// </summary>
        /// <param name="hWnd">hWnd</param>
        /// <param name="lpString">lpString</param>
        /// <param name="nMaxCount">nMaxCount</param>
        /// <returns>int</returns>
        [DllImport("user32.dll", CharSet = CharSet.Auto, SetLastError = true)]
        private static extern int GetWindowText(IntPtr hWnd, System.Text.StringBuilder lpString, int nMaxCount);

        /// <summary>
        /// BOOL EnumWindows(WNDENUMPROC lpEnumFunc, LPARAM lParam)
        /// </summary>
        /// <param name="lpEnumFunc">lpEnumFunc</param>
        /// <param name="lParam">lParam</param>
        /// <returns>BOOL</returns>
        [DllImport("user32.dll", CharSet = CharSet.Auto, SetLastError = true)]
        private static extern bool EnumWindows(EnumWindowsDelegate lpEnumFunc, IntPtr lParam);

        /// <summary>
        /// DWORD GetWindowThreadProcessId(HWND hWnd, LPDWORD lpdwProcessId)
        /// </summary>
        /// <param name="hWnd">hWnd</param>
        /// <param name="lpdwProcessId">lpdwProcessId</param>
        /// <returns>DWORD</returns>
        [DllImport("user32.dll", SetLastError = true)]
        private static extern int GetWindowThreadProcessId(IntPtr hWnd, out int lpdwProcessId);

        #endregion
    }
}
