﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;
using System.Windows.Forms;
using System.Windows.Forms.Layout;
using EffectMaker.UIControls.Extensions;

namespace EffectMaker.UIControls.Layout
{
    /// <summary>
    /// An extended LayoutEngine with wrapping capabilities.
    /// </summary>
    public class WrapLayoutEngine : LayoutEngineBase
    {
        /// <summary>
        /// Perform the layout.
        /// </summary>
        /// <param name="container">Container control.</param>
        /// <param name="containerParentSize">Size of the parent of the current container.</param>
        /// <param name="layoutEventArgs">Layout event argument.</param>
        /// <returns>Returns true if the parent must perform layout too,
        /// false otherwise.</returns>
        protected override bool OnLayout(
            Control container,
            Size containerParentSize,
            LayoutEventArgs layoutEventArgs)
        {
            // Use DisplayRectangle so that parent.Padding is honored
            Rectangle parentDisplayRectangle = container.DisplayRectangle;
            Point nextControlLocation = parentDisplayRectangle.Location;

            var maxHeight = 0;
            int currentTopMargin = 0;

            foreach (var child in container.GetNonCollapsedControls())
            {
                var c = (Control)child;

                // Set the autosized controls to their autosized heights
                if (c.AutoSize)
                {
                    c.Size = c.GetPreferredSize(parentDisplayRectangle.Size);
                }

                if (nextControlLocation.X + c.Margin.Left + c.Width + c.Margin.Right > parentDisplayRectangle.Width)
                {
                    nextControlLocation.X = parentDisplayRectangle.X;
                    nextControlLocation.Y += maxHeight;
                    maxHeight = 0;
                    currentTopMargin = 0;
                }

                // Respect the margin of the control:
                // shift over the left and the top
                nextControlLocation.Offset(c.Margin.Left, c.Margin.Top - currentTopMargin);
                currentTopMargin = c.Margin.Top;

                // Set the location of the control
                c.Location = nextControlLocation;

                maxHeight = Math.Max(maxHeight, c.DisplayRectangle.Height + c.Margin.Bottom);

                nextControlLocation.X += c.Width + c.Margin.Right;
            }

            //// Optional: Return whether or not the container's
            //// parent should perform layout as a result of this
            //// layout. Some layout engines return the value of
            //// the container's AutoSize property.

            return false;
        }
    }
}
