﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.ComponentModel;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Windows.Forms;
using System.Windows.Forms.Layout;
using EffectMaker.UIControls.Extensions;

namespace EffectMaker.UIControls.Layout
{
    /// <summary>
    /// An extended UILayoutPanel with side by side layout capabilities.
    /// </summary>
    public class SideBySidePanesPanel : UILayoutPanel<SideBySidePanesLayoutEngine>
    {
        /// <summary>
        /// Color of the border.
        /// </summary>
        private Color borderColor = Color.FromArgb(255, 200, 200, 200);

        /// <summary>
        /// The constructor.
        /// </summary>
        public SideBySidePanesPanel()
        {
            this.Margin = new Padding(0);
        }

        /// <summary>
        /// 一列に表示するかどうかを取得または設定します.
        /// </summary>
        [DefaultValue(false)]
        public bool IsSingleColumn
        {
            get
            {
                return this.InternalLayoutEngine.IsSingleColumn;
            }

            set
            {
                this.InternalLayoutEngine.IsSingleColumn = value;
            }
        }

        /// <summary>
        /// 左枠の大きさを取得または設定します。
        /// </summary>
        [DefaultValue(-1)]
        public int LeftPaneWidth
        {
            get
            {
                return this.InternalLayoutEngine.LeftPaneWidth;
            }

            set
            {
                this.InternalLayoutEngine.LeftPaneWidth = value;
            }
        }

        /// <summary>
        /// 右枠の大きさを取得または設定します。
        /// </summary>
        [DefaultValue(-1)]
        public int RightPaneWidth
        {
            get
            {
                return this.InternalLayoutEngine.RightPaneWidth;
            }

            set
            {
                this.InternalLayoutEngine.RightPaneWidth = value;
            }
        }

        /// <summary>
        /// Gets or sets the side by side distance of the control.
        /// </summary>
        [DefaultValue(16)]
        public int SideBySideDistance
        {
            get
            {
                return this.InternalLayoutEngine.SideBySideDistance;
            }

            set
            {
                this.InternalLayoutEngine.SideBySideDistance = value;
            }
        }

        /// <summary>
        /// Gets or sets the top and bottom distance of the control.
        /// </summary>
        [DefaultValue(0)]
        public int TopAndBottomDistance
        {
            get
            {
                return this.InternalLayoutEngine.TopAndBottomDistance;
            }

            set
            {
                this.InternalLayoutEngine.TopAndBottomDistance = value;
            }
        }

        /// <summary>
        /// Gets or sets the color of the border.
        /// </summary>
        public Color BorderColor
        {
            get
            {
                return this.borderColor;
            }

            set
            {
                if (value != this.borderColor)
                {
                    this.borderColor = value;
                    this.Invalidate();
                }
            }
        }

        /// <summary>
        /// UIの推薦サイズ(最小サイズ)を取得します.
        /// </summary>
        /// <param name="proposedSize">有効な領域</param>
        /// <returns>UIの推薦サイズを返します.</returns>
        public override Size GetPreferredSize(Size proposedSize)
        {
            return this.InternalLayoutEngine.GetPreferredSize(this, proposedSize);
        }

        /// <summary>
        /// Handle paint event.
        /// </summary>
        /// <param name="e">Event arguments.</param>
        protected override void OnPaint(System.Windows.Forms.PaintEventArgs e)
        {
            base.OnPaint(e);

            if (this.IsSingleColumn)
            {
                return;
            }

            // Get the graphics object for rendering
            Graphics g = e.Graphics;

            // Create the pen and brushes for rendering.
            using (Pen penBorder = new Pen(this.BorderColor, 2.0f))
            {
                g.DrawLine(
                    penBorder,
                    this.InternalLayoutEngine.BorderPosition,
                    this.Padding.Top,
                    this.InternalLayoutEngine.BorderPosition,
                    this.Height - this.Padding.Bottom);
            }
        }

        /// <summary>
        /// Produces a SideBySidePanesLayoutEngine instance.
        /// </summary>
        /// <returns>Returns a SideBySidePanesLayoutEngine instance.</returns>
        protected override SideBySidePanesLayoutEngine ProduceLayoutEngine()
        {
            return new SideBySidePanesLayoutEngine();
        }
    }
}
