﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Text;
using System.Linq;
using System.Windows.Forms;

using EffectMaker.Foundation.Render.ScrollBar;

namespace EffectMaker.UIControls.DebugConsole
{
    /// <summary>
    /// Control class for message views.
    /// </summary>
    public class MessageView : Control
    {
        /// <summary>The separators for lines in a message string.</summary>
        private static string[] newlineSeparators = new string[] { "\n" };

        /// <summary>The list of all the messages.</summary>
        private List<MessageData> messages = new List<MessageData>();

        /// <summary>The list of messages to display.</summary>
        private List<DisplayMessageData> displayMessages = new List<DisplayMessageData>();

        /// <summary>The sum of the height of all the displayed messages.</summary>
        private int totalDisplayHeight = 0;

        /// <summary>The maximum message display width.</summary>
        private int maxMessageWidth = 0;

        /// <summary>Flag indicating whether to scroll to the latest message when added.</summary>
        private bool isScrollToLatestEnabled = true;

        /// <summary>The index to the first displayed message on the scroll position.</summary>
        private int firstDisplayMessageIndex = 0;

        /// <summary>The scroll bars.</summary>
        private AutoHiddenScrollBar scrollBars = null;

        /// <summary>
        /// Constructor.
        /// </summary>
        public MessageView()
        {
            this.DoubleBuffered = true;

            this.scrollBars = new AutoHiddenScrollBar(this);
            this.scrollBars.RenderRequired += (s, e) => { this.Invalidate(); };
            this.scrollBars.Scroll += (s, e) =>
            {
                this.firstDisplayMessageIndex = this.FindLineAt(0);
                this.Invalidate();
            };
        }

        /// <summary>
        /// Get or set the scroll position.
        /// </summary>
        public Point ScrollPosition
        {
            get { return this.scrollBars.ScrollPosition; }
            set { this.SetScrollPosition(value.X, value.Y); }
        }

        /// <summary>Get or set the foreground color of the scroll bars.</summary>
        public Color ScrollBarForeColor
        {
            get { return this.scrollBars.ForeColor; }
            set { this.scrollBars.ForeColor = value; }
        }

        /// <summary>Get or set the background color of the scroll bars.</summary>
        public Color ScrollBarBackColor
        {
            get { return this.scrollBars.BackColor; }
            set { this.scrollBars.BackColor = value; }
        }

        /// <summary>
        /// Get or set the flag indicating whether to scroll to the latest message when added.
        /// </summary>
        public bool IsScrollToLatestEnabled
        {
            get
            {
                return this.isScrollToLatestEnabled;
            }

            set
            {
                this.isScrollToLatestEnabled = value;
                if (value == true)
                {
                    this.SetScrollPosition(0, this.totalDisplayHeight);
                }
            }
        }

        /// <summary>
        /// Set scroll position.
        /// </summary>
        /// <param name="x">The horizontal position.</param>
        /// <param name="y">The vertical position.</param>
        public void SetScrollPosition(int x, int y)
        {
            int maxX = this.maxMessageWidth - this.Width;
            int maxY = this.totalDisplayHeight - this.Height;

            int posX = Math.Max(Math.Min(x, maxX), 0);
            int posY = Math.Max(Math.Min(y, maxY), 0);

            this.scrollBars.ScrollPosition = new Point(posX, posY);

            this.firstDisplayMessageIndex = this.FindLineAt(0);
        }

        /// <summary>
        /// Add a new message.
        /// </summary>
        /// <param name="text">Text for the message.</param>
        public void AddMessage(string text)
        {
            this.AddMessage(text, this.ForeColor, this.BackColor);
        }

        /// <summary>
        /// Add a new message.
        /// </summary>
        /// <param name="text">Text for the message.</param>
        /// <param name="foreColor">Text color for the message.</param>
        public void AddMessage(string text, Color foreColor)
        {
            this.AddMessage(text, foreColor, this.BackColor);
        }

        /// <summary>
        /// Add a new message.
        /// </summary>
        /// <param name="text">Text for the message.</param>
        /// <param name="foreColor">Text color for the message.</param>
        /// <param name="backColor">Background color for the message.</param>
        public void AddMessage(string text, Color foreColor, Color backColor)
        {
            if (string.IsNullOrEmpty(text) == true)
            {
                return;
            }

            bool redraw = false;

            int id = this.messages.Count;

            int width = 0;
            int height = 0;
            using (Graphics g = this.CreateGraphics())
            {
                string[] lines = text.Split(newlineSeparators, StringSplitOptions.None);
                foreach (string line in lines)
                {
                    string myLine = line;

                    // Replace empty line with a space character.
                    if (string.IsNullOrEmpty(myLine) == true)
                    {
                        myLine = " ";
                    }

                    SizeF lineSize = g.MeasureString(myLine, this.Font);
                    width = (int)Math.Ceiling(lineSize.Width);
                    height = (int)Math.Ceiling(lineSize.Height);

                    // Create the message data.
                    var msg = new MessageData()
                    {
                        ID = id,
                        Text = text,
                        ForeColor = foreColor,
                        BackColor = backColor,
                        Width = width,
                        Height = height
                    };

                    // Add to the list of all the messages.
                    this.messages.Add(msg);

                    // If the message passes the filter, add it to the displaying message list too.
                    if (this.FilterMessage(msg) == true)
                    {
                        this.AddDisplayMessage(msg);
                        redraw = true;
                    }
                }
            }

            if (redraw == true)
            {
                this.Invalidate();
            }
        }

        /// <summary>
        /// Clear all the messages.
        /// </summary>
        public void ClearMessages()
        {
            this.messages.Clear();
            this.ClearDisplayMessages();
            this.Invalidate();
        }

        /// <summary>
        /// Filter all the messages with the current conditions.
        /// </summary>
        protected void FilterMessages()
        {
            this.scrollBars.SuspendLayout();

            // Reset first.
            this.totalDisplayHeight = 0;
            this.maxMessageWidth = 0;
            this.displayMessages.Clear();

            // Filter through all the messages.
            foreach (MessageData msg in this.messages)
            {
                if (this.FilterMessage(msg) == true)
                {
                    this.AddDisplayMessage(msg);
                }
            }

            this.scrollBars.ResumeLayout();

            this.Invalidate();
        }

        /// <summary>
        /// Filter the given message with the current conditions.
        /// </summary>
        /// <param name="message">The message to filter.</param>
        /// <returns>True if the message passes the filtering.</returns>
        protected bool FilterMessage(MessageData message)
        {
            // To be implemented.
            return true;
        }

        /// <summary>
        /// Handle Paint event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnPaint(PaintEventArgs e)
        {
            base.OnPaint(e);

            TextRenderingHint originalTextRenderHint = e.Graphics.TextRenderingHint;
            e.Graphics.TextRenderingHint = TextRenderingHint.ClearTypeGridFit;

            int posX = 0;
            int posY = 0;

            // Check if the displayed message index is correct.
            int index = this.firstDisplayMessageIndex;
            if ((index < 0) || (index >= this.displayMessages.Count))
            {
                return;
            }

            Point scrollPos = this.ScrollPosition;

            // Render the messages.
            using (var brush = new SolidBrush(Color.Black))
            {
                var msg = this.displayMessages[index];
                while ((msg != null) &&
                       (msg.Location < this.Height + scrollPos.Y))
                {
                    // Compute text rendering position.
                    posX = -scrollPos.X;
                    posY = msg.Location - scrollPos.Y;

                    // Setup background color.
                    brush.Color = msg.Message.BackColor;

                    // Render background.
                    e.Graphics.FillRectangle(brush, 0, posY, this.Width, msg.Message.Height);

                    // Setup text color.
                    brush.Color = msg.Message.ForeColor;

                    // Render the text.
                    e.Graphics.DrawString(msg.Message.Text, this.Font, brush, posX, posY);

                    // Advance message index.
                    if (++index >= this.displayMessages.Count)
                    {
                        break;
                    }

                    msg = this.displayMessages[index];
                }
            }

            e.Graphics.TextRenderingHint = originalTextRenderHint;

            // Enable alpha blending.
            e.Graphics.CompositingMode =
                System.Drawing.Drawing2D.CompositingMode.SourceOver;

            // Render the scroll bars.
            this.scrollBars.Render(e.Graphics);
        }

        /// <summary>
        /// Handle MouseWheel event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseWheel(MouseEventArgs e)
        {
            base.OnMouseWheel(e);

            int increment = -(e.Delta / 120);
            int numWheelScrolls = Math.Abs(-(e.Delta / 120));

            int index = this.firstDisplayMessageIndex + increment;

            index = Math.Max(Math.Min(index, this.displayMessages.Count), 0);
            if (index < 0 || index >= this.displayMessages.Count)
            {
                return;
            }

            var msg = this.displayMessages[index];
            if (msg == null)
            {
                return;
            }

            this.SetScrollPosition(this.ScrollPosition.X, msg.Location);

            this.Invalidate();
        }

        /// <summary>
        /// Handle MouseDown event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseDown(MouseEventArgs e)
        {
            base.OnMouseDown(e);

            if (e.Button == MouseButtons.Middle)
            {
                this.ClearMessages();
            }

            //// if (e.Button == MouseButtons.Left)
            //// {
            ////     int line = FindLineAt(e.Y);
            ////     if (line >= 0 && line < this.displayMessages.Count)
            ////     {
            ////         DisplayMessageData data = this.displayMessages[line];
            ////         int charIndex = FindCharIndexAt(line, e.X);
            ////         if (charIndex >= 0 && charIndex < data.Message.Text.Length)
            ////         {
            ////         }
            ////     }
            //// }
        }

        /// <summary>
        /// Handle MouseMove event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseMove(MouseEventArgs e)
        {
            base.OnMouseMove(e);

            if ((this.scrollBars.IsMouseOverScrollBars == false) &&
                (this.scrollBars.IsMouseDragging == false))
            {
                Cursor.Current = Cursors.IBeam;
            }
        }

        /// <summary>
        /// Handle SizeChanged event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnSizeChanged(EventArgs e)
        {
            base.OnSizeChanged(e);

            Point scrollPos = this.scrollBars.ScrollPosition;

            this.scrollBars.SetViewSize(this.Size);
            this.scrollBars.SetDisplayRectangle(this.DisplayRectangle);

            // Reset to the same scroll position again.
            this.scrollBars.ScrollPosition = scrollPos;
        }

        /// <summary>
        /// Add the given message to the display list.
        /// </summary>
        /// <param name="message">The message to add.</param>
        private void AddDisplayMessage(MessageData message)
        {
            int originalLineNumberDigitCount =
                (int)Math.Floor(Math.Log10(this.displayMessages.Count) + 1);

            this.displayMessages.Add(new DisplayMessageData(message, this.totalDisplayHeight));

            int currentLineNumberDigitCount =
                (int)Math.Floor(Math.Log10(this.displayMessages.Count) + 1);

            // Have the digit count of the maximum line number changed?
            if (originalLineNumberDigitCount != currentLineNumberDigitCount)
            {
                // Compose a string with all zeros characters,
                // the same characters as the line number digit count.
                // Use this string to measure line number field size.
                string lineNumberString = string.Empty;
                for (int i = 0; i < currentLineNumberDigitCount; ++i)
                {
                    lineNumberString += "0";
                }

                //// Measure the size of the line number field.
                //// this.lineNumberFieldSize = TextRenderer.MeasureText(lineNumberString, this.Font);
            }

            this.maxMessageWidth = Math.Max(message.Width, this.maxMessageWidth);
            this.totalDisplayHeight += message.Height;

            // Update content height to the scroll bars.
            this.scrollBars.SetContentSize(
                new Size(this.maxMessageWidth, this.totalDisplayHeight));

            // Basically all the lines should have the same height,
            // set the height to the scroll bar for the increment.
            this.scrollBars.IncrementV = message.Height;

            // Scroll to the latest message.
            if (this.IsScrollToLatestEnabled == true)
            {
                this.SetScrollPosition(0, this.totalDisplayHeight);
            }
        }

        /// <summary>
        /// Clear all the displayed messages.
        /// </summary>
        private void ClearDisplayMessages()
        {
            this.displayMessages.Clear();

            this.maxMessageWidth = 0;
            this.totalDisplayHeight = 0;

            // Update content height to the scroll bars.
            this.scrollBars.SetContentSize(Size.Empty);

            // Basically all the lines should have the same height,
            // set the height to the scroll bar for the increment.
            this.scrollBars.IncrementV = 0;

            // Scroll to the latest message.
            this.SetScrollPosition(0, 0);
        }

        /// <summary>
        /// Find the line index at the specified Y position in the view.
        /// </summary>
        /// <param name="viewPosY">The Y position in the view.</param>
        /// <returns>The line index.</returns>
        private int FindLineAt(int viewPosY)
        {
            if (this.displayMessages.Count <= 0)
            {
                return 0;
            }

            Point scrollPos = this.ScrollPosition;
            int pos = scrollPos.Y + viewPosY;

            int left = 0;
            int right = this.displayMessages.Count - 1;
            while (left < right)
            {
                int mid = (left + right) >> 1;
                var data = this.displayMessages[mid];
                if (data.Location == pos)
                {
                    return mid;
                }
                else
                {
                    if (data.Location < pos)
                    {
                        left = mid + 1;
                    }
                    else
                    {
                        right = mid - 1;
                    }
                }
            }

            if (this.displayMessages[left].Location > pos)
            {
                return left - 1;
            }
            else
            {
                return left;
            }
        }

        /// <summary>
        /// Find the character index of the specified line at the given X position in the view.
        /// </summary>
        /// <param name="lineIndex">The line index.</param>
        /// <param name="viewPosX">The X position in the view.</param>
        /// <returns>The character index.</returns>
        private int FindCharIndexAt(int lineIndex, int viewPosX)
        {
            if (viewPosX < 0)
            {
                return -1;
            }

            if (lineIndex < 0 || lineIndex >= this.displayMessages.Count)
            {
                return -1;
            }

            DisplayMessageData data = this.displayMessages[lineIndex];
            if (data == null)
            {
                return -1;
            }

            int numChars = data.Text.Length;
            if (numChars <= 0)
            {
                return -1;
            }

            if (viewPosX >= data.Width)
            {
                return numChars - 1;
            }

            float t = (float)viewPosX / (float)data.Width;

            // Roughly compute the character index (treat every character as evenly sized).
            return (int)Math.Round(t * (float)numChars);
        }

        /// <summary>
        /// Class to hold message data.
        /// </summary>
        protected class MessageData
        {
            /// <summary>
            /// Constructor.
            /// </summary>
            public MessageData()
            {
            }

            /// <summary>Get or set the message ID.</summary>
            public int ID { get; set; }

            /// <summary>Get or set the text for the message.</summary>
            public string Text { get; set; }

            /// <summary>Get or set the text color for the message.</summary>
            public Color ForeColor { get; set; }

            /// <summary>Get or set the background color for the message.</summary>
            public Color BackColor { get; set; }

            /// <summary>Get or set the display width of the message.</summary>
            public int Width { get; set; }

            /// <summary>Get or set the display height of the message.</summary>
            public int Height { get; set; }
        }

        /// <summary>
        /// Class for the data of the messages to display.
        /// </summary>
        protected class DisplayMessageData
        {
            /// <summary>
            /// Constructor.
            /// </summary>
            /// <param name="message">The message data.</param>
            /// <param name="location">The vertical display location of the message.</param>
            public DisplayMessageData(MessageData message, int location)
            {
                this.Message = message;
                this.Location = location;
            }

            /// <summary>Get or set the message data.</summary>
            public MessageData Message { get; set; }

            /// <summary>Get the vertical location of the message.</summary>
            public int Location { get; set; }

            /// <summary>Get the message ID.</summary>
            public int ID
            {
                get { return this.Message.ID; }
            }

            /// <summary>Get or set the text for the message.</summary>
            public string Text
            {
                get { return this.Message.Text; }
                set { this.Message.Text = value; }
            }

            /// <summary>Get or set the text color for the message.</summary>
            public Color ForeColor
            {
                get { return this.Message.ForeColor; }
                set { this.Message.ForeColor = value; }
            }

            /// <summary>Get or set the background color for the message.</summary>
            public Color BackColor
            {
                get { return this.Message.BackColor; }
                set { this.Message.BackColor = value; }
            }

            /// <summary>Get or set the display width of the message.</summary>
            public int Width
            {
                get { return this.Message.Width; }
                set { this.Message.Width = value; }
            }

            /// <summary>Get or set the display height of the message.</summary>
            public int Height
            {
                get { return this.Message.Height; }
                set { this.Message.Height = value; }
            }
        }
    }
}
