﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Windows.Forms;
using EffectMaker.Foundation.Log;
using System.Threading;
using EffectMaker.UIControls.Threading;

namespace EffectMaker.UIControls.DebugConsole
{
    /// <summary>
    /// Class for the debug console.
    /// </summary>
    public partial class DebugConsole : Form, ILogHandler
    {
        /// <summary>The default background color for messages.</summary>
        private static Color defaultBackColor = Color.Black;

        /// <summary>The text color for profiling messages.</summary>
        private static Color profileColor = Color.LightSkyBlue;

        /// <summary>The text color for debug messages.</summary>
        private static Color debugColor = Color.Gray;

        /// <summary>The text color for information messages.</summary>
        private static Color informationColor = Color.White;

        /// <summary>The text color for warning messages.</summary>
        private static Color warningColor = Color.Yellow;

        /// <summary>The text color for error messages.</summary>
        private static Color errorColor = Color.Red;

        /// <summary>The text color for fatal messages.</summary>
        private static Color fatalForeColor = Color.Black;

        /// <summary>The background color for fatal messages.</summary>
        private static Color fatalBackColor = Color.Red;

        /// <summary>
        /// Constructor.
        /// </summary>
        public DebugConsole()
        {
            this.InitializeComponent();

            // 同期コンテキストを取得
            this.SynchronizationContext = SynchronizationContext.Current;
        }

        /// <summary>
        /// ログハンドラの名前を取得します。
        /// </summary>
        public string LogHandlerName
        {
            get { return "DebugConsole"; }
        }

        /// <summary>
        /// ログハンドラの同期コンテキストを取得します。
        /// </summary>
        public SynchronizationContext SynchronizationContext { get; private set; }

        /// <summary>
        /// ロガーからメッセージが送られたときの処理を行います。
        /// </summary>
        /// <param name="destination">ログ出力先</param>
        /// <param name="level">ログレベル</param>
        /// <param name="message">ログメッセージ</param>
        /// <param name="callStack">コールスタック</param>
        public void Log(IEnumerable<string> destinations, LogLevels level, string message, StackFrame callStack)
        {
            if (this.IsDisposed) return;

            Color backColor = defaultBackColor;
            Color foreColor;

            switch (level)
            {
                case LogLevels.Profile:
                    foreColor = profileColor;
                    break;

                case LogLevels.Debug:
                    foreColor = debugColor;
                    break;

                case LogLevels.Information:
                    foreColor = informationColor;
                    break;

                case LogLevels.Warning:
                    foreColor = warningColor;
                    break;

                case LogLevels.Error:
                    foreColor = errorColor;
                    break;

                case LogLevels.Fatal:
                    foreColor = fatalForeColor;
                    backColor = fatalBackColor;
                    break;

                default:
                    // We don't handle any other types of messages.
                    return;
            }

            this.messageView.AddMessage(message, foreColor, backColor);
        }
    }
}
