﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using EffectMaker.UIControls.Extensions;

namespace EffectMaker.UIControls.Debug
{
    /// <summary>
    /// A control that creates and display a tree of controls.
    /// </summary>
    public partial class ControlTreeViewerControl : UserControl
    {
        /// <summary>
        /// Stores the root control.
        /// </summary>
        private Control rootControl;

        /// <summary>
        /// Flag shared among the calls of the recursive SelectControl method.
        /// </summary>
        private bool selectedFound;

        /// <summary>
        /// Stores the mouse hovered control.
        /// </summary>
        private Control controlMouseOn;

        /// <summary>
        /// Backing field for Title property.
        /// </summary>
        private string title;

        /// <summary>
        /// Initializes the ControlTreeViewerControl instance.
        /// </summary>
        /// <param name="rootControl">The root control to create a tree from.</param>
        public ControlTreeViewerControl(Control rootControl)
        {
            this.InitializeComponent();

            if (rootControl == null)
            {
                throw new ArgumentNullException("rootControl");
            }

            this.rootControl = rootControl;

            this.propertyGrid.PropertySort = PropertySort.Alphabetical;

            this.trvControlTree.Nodes.Add(new ControlTreeNode(rootControl));
            this.trvControlTree.AfterSelect += this.OnControlTreeAfterSelect;

            this.rootControl.TextChanged += this.OnRootControlTextChanged;

            this.UpdateTitle();

            this.Disposed += this.OnControlTreeViewerDisposed;

            Application.Idle += this.OnApplicationIdle;
        }

        /// <summary>
        /// Raised when the title changes.
        /// </summary>
        public event EventHandler TitleChanged;

        /// <summary>
        /// Gets or sets the control title.
        /// </summary>
        public string Title
        {
            get
            {
                return this.title;
            }

            set
            {
                if (this.title != value)
                {
                    this.title = value;

                    var handler = this.TitleChanged;
                    if (handler != null)
                    {
                        handler(this, EventArgs.Empty);
                    }
                }
            }
        }

        /// <summary>
        /// Called when the control tree viewer form is disposed.
        /// </summary>
        /// <param name="sender">Caller of the event.</param>
        /// <param name="e">Event argument.</param>
        private void OnControlTreeViewerDisposed(object sender, EventArgs e)
        {
            this.rootControl.TextChanged -= this.OnRootControlTextChanged;
            this.DisposeControlTreeNodes(this.trvControlTree.Nodes);
        }

        /// <summary>
        /// Disposes all node that inherit from IDisposable, recursively.
        /// </summary>
        /// <param name="nodes">The nodes to dispose.</param>
        private void DisposeControlTreeNodes(TreeNodeCollection nodes)
        {
            foreach (var node in nodes.OfType<IDisposable>())
            {
                this.DisposeControlTreeNodes(((TreeNode)node).Nodes);
                node.Dispose();
            }
        }

        /// <summary>
        /// Called when the application is idling. This method run several times.
        /// </summary>
        /// <param name="sender">Caller of the event.</param>
        /// <param name="e">Event argument.</param>
        private void OnApplicationIdle(object sender, EventArgs e)
        {
            if (Control.ModifierKeys == Keys.Control)
            {
                var position = this.rootControl.PointToClient(Control.MousePosition);
                var control = this.rootControl.GetBottomMostChildAtPoint(position);

                if (control != null)
                {
                    this.SelectControl(control);
                }

                if (this.controlMouseOn != control)
                {
                    if (this.controlMouseOn != null)
                    {
                        this.LeaveControl(this.controlMouseOn);
                    }

                    this.controlMouseOn = control;

                    if (this.controlMouseOn != null)
                    {
                        this.EnterControl(this.controlMouseOn);
                    }
                }
            }
        }

        /// <summary>
        /// Runs when mouse enters a control.
        /// </summary>
        /// <param name="control">The control on which the mouse entered.</param>
        private void EnterControl(Control control)
        {
            Console.WriteLine("Enter {0} [{1}]", control.Name, control.Text);
        }

        /// <summary>
        /// Runs when mouse leaves a control.
        /// </summary>
        /// <param name="control">The control from which the mouse left.</param>
        private void LeaveControl(Control control)
        {
            Console.WriteLine("Leave {0} [{1}]", control.Name, control.Text);
        }

        /// <summary>
        /// Select a control tree node given an associated control.
        /// </summary>
        /// <param name="control">The control stored by the control tree node to select.</param>
        private void SelectControl(Control control)
        {
            this.selectedFound = false;
            this.SelectControl(this.trvControlTree.Nodes, control);
        }

        /// <summary>
        /// Select a control tree node given an associated control.
        /// </summary>
        /// <param name="nodes">The nodes to check for the control to select.</param>
        /// <param name="control">The control to make selected.</param>
        private void SelectControl(TreeNodeCollection nodes, Control control)
        {
            foreach (var node in nodes.OfType<ControlTreeNode>())
            {
                if (this.selectedFound)
                {
                    break;
                }

                if (node.Control == control)
                {
                    this.trvControlTree.SelectedNode = node;
                    this.selectedFound = true;
                }
                else
                {
                    this.SelectControl(node.Nodes, control);
                }
            }
        }

        /// <summary>
        /// Called when the Text property of the root control changes.
        /// </summary>
        /// <param name="sender">Caller of the event.</param>
        /// <param name="e">Event argument.</param>
        private void OnRootControlTextChanged(object sender, EventArgs e)
        {
            this.UpdateTitle();
        }

        /// <summary>
        /// Updates the title of the control tree viewer window
        /// using the Text property of the root control.
        /// </summary>
        private void UpdateTitle()
        {
            this.Title = this.rootControl.Text;
        }

        /// <summary>
        /// Raised when a node in the control tree is selected.
        /// </summary>
        /// <param name="sender">The caller of the event.</param>
        /// <param name="e">The event argument.</param>
        private void OnControlTreeAfterSelect(object sender, TreeViewEventArgs e)
        {
            var node = e.Node as ControlTreeNode;

            if (node == null)
            {
                this.propertyGrid.SelectedObject = null;
                return;
            }

            this.propertyGrid.SelectedObject = node.Control;
        }
    }
}
