﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using System.Text;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extenders;

namespace EffectMaker.UIControls.Behaviors
{
    /// <summary>
    /// Represent a functionality that can be attached to an object to enrich it.
    /// </summary>
    public abstract class Behavior : IBindable
    {
        /// <summary>
        /// Backing field for the LogicalTreeElementExtender property.
        /// </summary>
        private LogicalTreeElementExtender logicalTreeElementExtender;

        /// <summary>
        /// Backing field for the Children property.
        /// </summary>
        private IIndexableCollection<ILogicalTreeElement> children;

        /// <summary>
        /// Initializes the Behavior instance.
        /// </summary>
        protected Behavior()
        {
            this.Bindings = new BindingContainer(this);

            this.logicalTreeElementExtender = new LogicalTreeElementExtender(this);
            this.children = new NullIndexableCollection<ILogicalTreeElement>();
        }

        /// <summary>
        /// Raised when the value of a property on this control changed.
        /// </summary>
#pragma warning disable 67
        public event PropertyChangedEventHandler PropertyChanged;
#pragma warning restore 67

        /// <summary>
        /// Gets the bindings containter.
        /// </summary>
        public BindingContainer Bindings { get; private set; }

        /// <summary>
        /// Gets the LogicalTreeElementExtender instance.
        /// </summary>
        public LogicalTreeElementExtender LogicalTreeElementExtender
        {
            get { return this.logicalTreeElementExtender; }
        }

        /// <summary>
        /// Gets or sets the data context.
        /// </summary>
        public object DataContext
        {
            get { return this.logicalTreeElementExtender.DataContext; }
            set { this.logicalTreeElementExtender.DataContext = value; }
        }

        /// <summary>
        /// Gets the parent element.
        /// </summary>
        public ILogicalTreeElement Parent
        {
            get
            {
                if (this.AssociatedObject is IControl)
                {
                    return ((IControl)this.AssociatedObject).Behaviors;
                }

                return null;
            }
        }

        /// <summary>
        /// Gets the child elements.
        /// </summary>
        public IIndexableCollection<ILogicalTreeElement> Children
        {
            get { return this.children; }
        }

        /// <summary>
        /// Gets the object to which the functionality is attached.
        /// </summary>
        protected object AssociatedObject { get; private set; }

        /// <summary>
        /// Attach the functionality to an object.
        /// </summary>
        /// <param name="associatedObject">The object to which the
        /// functionality is attached.</param>
        public void Attach(object associatedObject)
        {
            if (associatedObject == null)
            {
                throw new ArgumentException("associatedObject");
            }

            if (this.AssociatedObject != null)
            {
                throw new InvalidOperationException("Cannot be attached several times.");
            }

            this.AssociatedObject = associatedObject;
            this.OnAttached();
        }

        /// <summary>
        /// Detach the functionality from the object.
        /// </summary>
        public void Detach()
        {
            if (this.AssociatedObject == null)
            {
                return;
            }

            this.OnDetaching();
            this.AssociatedObject = null;
        }

        /// <summary>
        /// Clears the data context.
        /// </summary>
        public void ClearDataContext()
        {
            this.logicalTreeElementExtender.ClearDataContext();
        }

        /// <summary>
        /// When overridden, called when the behavior has
        /// been attached to the associated object.
        /// </summary>
        protected abstract void OnAttached();

        /// <summary>
        /// When overridden, called when the behavior is
        /// beeing detached from the associated object.
        /// </summary>
        protected abstract void OnDetaching();
    }

    /// <summary>
    /// Represent a functionality that can be attached to a typed object to enrich it.
    /// </summary>
    /// <typeparam name="T">The type of object to attach to.</typeparam>
    public abstract class Behavior<T> : Behavior
    {
        /// <summary>
        /// Gets the typed object to which the functionality is attached.
        /// </summary>
        protected new T AssociatedObject
        {
            get
            {
                return (T)base.AssociatedObject;
            }
        }
    }
}
