﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Runtime.CompilerServices;
using System.Windows.Forms;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Render.Layout;
using EffectMaker.UIControls.Behaviors;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.EventArguments;
using EffectMaker.UIControls.Extenders;
using EffectMaker.UIControls.Layout;

namespace EffectMaker.UIControls.BaseControls
{
    /// <summary>
    /// An extended UserControl class.
    /// </summary>
    public class UIUserControl : UserControl, IControl, ILayoutElement
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public UIUserControl()
        {
            this.AutoSize = true;

            this.Bindings = new BindingContainer(this);
            this.LogicalTreeElementExtender = new LogicalTreeElementExtender(this);
            this.Behaviors = new BehaviorCollection(this);
            this.Resources = new Dictionary<string, object>();
            this.controlsWrapper = new ControlCollectionWrapper(this);
            this.LayoutElementExtender = new LayoutElementExtender(this);
        }

        #region IBindable members

        /// <summary>
        /// バインダのコレクションを取得します。
        /// </summary>
        public BindingContainer Bindings { get; private set; }

        #endregion

        #region ILogicalTreeElement members

        /// <summary>
        /// LogicalTreeElementExtender を取得します。
        /// </summary>
        public LogicalTreeElementExtender LogicalTreeElementExtender { get; private set; }

        /// <summary>
        /// データコンテキストを取得または設定します。
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public virtual object DataContext
        {
            get { return this.LogicalTreeElementExtender.DataContext; }
            set { this.LogicalTreeElementExtender.DataContext = value; }
        }

        /// <summary>
        /// 親コントロールを取得します。
        /// </summary>
        ILogicalTreeElement ILogicalTreeElement.Parent
        {
            get { return Parent as ILogicalTreeElement; }
        }

        /// <summary>
        /// チャイルドコントロールのコレクションを取得します。
        /// </summary>
        public IIndexableCollection<ILogicalTreeElement> Children
        {
            get { return ((IControl)this).Controls; }
        }

        /// <summary>
        /// データコンテキストをクリアします。
        /// </summary>
        public void ClearDataContext()
        {
            this.LogicalTreeElementExtender.ClearDataContext();
        }

        #endregion

        #region INotifyPropertyChanged members

        /// <summary>
        /// プロパティの値が変わったときに発生するイベントです。
        /// </summary>
        public event PropertyChangedEventHandler PropertyChanged;

        /// <summary>
        /// PropertyChanged イベントを発生させます。
        /// </summary>
        /// <param name="updateType">どのような方法でプロパティが変更されたか</param>
        /// <param name="propertyName">変更されたプロパティの名前</param>
        public void NotifyPropertyChanged(
            BindingUpdateType updateType = BindingUpdateType.PropertyChanged,
            [CallerMemberName]string propertyName = null)
        {
            this.PropertyChanged?.Invoke(this, new PropertyChangedExEventArgs(propertyName, updateType));
        }

        #endregion

        #region IControl members

        /// <summary>
        /// ビヘイビアのコレクションを取得します。
        /// </summary>
        public BehaviorCollection Behaviors { get; private set; }

        /// <summary>
        /// IControls.Controls プロパティのバッキングフィールドです。
        /// </summary>
        private IIndexableCollection<ILogicalTreeElement> controlsWrapper;

        /// <summary>
        /// チャイルドコントロールのコレクションを取得します。
        /// </summary>
        IIndexableCollection<ILogicalTreeElement> IControl.Controls
        {
            get { return this.controlsWrapper; }
        }

        /// <summary>
        /// リソースデータのディクショナリを取得します。
        /// </summary>
        public IDictionary<string, object> Resources { get; private set; }

        #endregion

        #region ILayoutElement members

        /// <summary>
        /// Gets the layout element extender instance of this control.
        /// </summary>
        public LayoutElementExtender LayoutElementExtender { get; private set; }

        /// <summary>
        /// Gets or sets the visibility.
        /// See ControlExtender for more information.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public Visibility Visibility
        {
            get { return this.LayoutElementExtender.Visibility; }
            set { this.LayoutElementExtender.Visibility = value; }
        }

        /// <summary>
        /// Gets or sets the horizontal alignment.
        /// See ControlExtender for more information.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public HAlignment HorizontalAlignment
        {
            get { return this.LayoutElementExtender.HorizontalAlignment; }
            set { this.LayoutElementExtender.HorizontalAlignment = value; }
        }

        /// <summary>
        /// Gets or sets the vertical alignment.
        /// See ControlExtender for more information.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public VAlignment VerticalAlignment
        {
            get { return this.LayoutElementExtender.VerticalAlignment; }
            set { this.LayoutElementExtender.VerticalAlignment = value; }
        }

        /// <summary>
        /// VisibilityがCollapsedになった時に元のサイズを覚えておくためのプロパティ
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public Size OriginalSize { get; set; }

        #endregion

        /// <summary>
        /// Gets or sets the enable menu shortcut.
        /// </summary>
        public static Action EnableMenuShortcut { get; set; }

        /// <summary>
        /// Gets or sets the disable menu shortcut.
        /// </summary>
        public static Action DisableMenuShortcut { get; set; }

        /// <summary>
        /// ダブルバッファリングフラグ。
        /// </summary>
        [DefaultValue(false)]
        public bool DoubleBuffering
        {
            get
            {
                return this.DoubleBuffered;
            }

            set
            {
                this.DoubleBuffered = value;

                // ダブルバッファ時の設定
                this.SetStyle(ControlStyles.ResizeRedraw, value);
            }
        }

        /// <summary>
        /// データコンテキストを切り替え中かどうかを取得します。
        /// </summary>
        public bool IsUpdatingDataContext
        {
            get { return LogicalTreeElementExtender.IsUpdatingDataContext; }
        }
    }
}
