﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.UIControls.Extensions;

namespace EffectMaker.UIControls.BaseControls
{
    /// <summary>
    /// ラジオボタングループです.
    /// </summary>
    public partial class UIRadioButtonGroup : UIUserControl
    {
        /// <summary>
        /// 選択中のインデックスです.
        /// </summary>
        private int selectedIndex = -1;

        /// <summary>
        /// CheckedChangedイベントを無視するためのフラグ.
        /// </summary>
        private bool ignoreCheckedChanged = false;

        /// <summary>
        /// アイテム一覧です.
        /// </summary>
        private IEnumerable<KeyValuePair<string, object>> availableItems;

        /// <summary>
        /// The constructor.
        /// </summary>
        public UIRadioButtonGroup()
        {
            this.InitializeComponent();
        }

        /// <summary>
        /// 選択中のインデックスを設定または取得します.
        /// </summary>
        public int SelectedIndex
        {
            get
            {
                return this.selectedIndex;
            }

            set
            {
                int index = value;

                // インデックスの範囲をチェック
                if (index < 0 || index >= this.stackPanel.Controls.Count)
                {
                    index = -1;
                }

                if (index == this.selectedIndex)
                {
                    return;
                }

                // 新しいインデックスを設定
                this.selectedIndex = value;

                this.ignoreCheckedChanged = true;

                if (index == -1)
                {
                    // インデックスがクリアされた時は全部のボタンのチェックを解除する
                    foreach (var button in this.stackPanel.Controls.OfType<UIRadioButton>())
                    {
                        button.Checked = false;
                    }
                }
                else
                {
                    // 指定されたインデックスのラジオボタンをチェック状態に変更
                    var button = (UIRadioButton)this.stackPanel.Controls.ElementAt(index);
                    button.Checked = true;
                }

                this.ignoreCheckedChanged = false;

                this.LogicalTreeElementExtender.NotifyPropertyChanged(propertyName: "SelectedIndex");
                this.LogicalTreeElementExtender.NotifyPropertyChanged(propertyName: "SelectedItem");
            }
        }

        /// <summary>
        /// 選択中のアイテムを設定または取得します.
        /// </summary>
        public object SelectedItem
        {
            get
            {
                if ((this.SelectedIndex >= 0) &&
                    (this.SelectedIndex < this.stackPanel.Controls.Count))
                {
                    var item = this.availableItems.ElementAt(this.SelectedIndex);
                    return item.Value;
                }
                else
                {
                    return null;
                }
            }

            set
            {
                if (this.availableItems == null)
                {
                    return;
                }

                // uintとulongの比較がobject.Equalsだとできないので、
                // 数値にパースしてから比較する
                ulong tmpA, tmpB;
                int index;
                if (ulong.TryParse(value.ToString(), out tmpA))
                {
                    index = this.availableItems.FindIndex(cbi =>
                    {
                        if (ulong.TryParse(cbi.Value.ToString(), out tmpB))
                        {
                            return tmpA == tmpB;
                        }

                        return object.Equals(cbi.Value, value);
                    });
                }
                else
                {
                    // ulong以外の型は今まで通りobject.Equalsで判定
                    index = this.availableItems
                        .FindIndex(cbi => object.Equals(cbi.Value, value));
                }

                this.SelectedIndex = index;
            }
        }

        /// <summary>
        /// Gets or sets the orientation of the control.
        /// </summary>
        public Layout.Orientation Orientation
        {
            get
            {
                return this.stackPanel.Orientation;
            }

            set
            {
                this.stackPanel.Orientation = value;
            }
        }

        /// <summary>
        /// スタックパネルのコントロールを取得します.
        /// </summary>
        public IIndexableCollection<ILogicalTreeElement> StackControls
        {
            get
            {
                return this.stackPanel.Controls;
            }
        }

        /// <summary>
        /// アイテム一覧を設定または取得します.
        /// </summary>
        public IEnumerable<KeyValuePair<string, object>> AvailableItems
        {
            get
            {
                return this.availableItems;
            }

            set
            {
                if (!this.IsAcceptableItems(value))
                {
                    return;
                }

                IEnumerable oldValue = this.availableItems;

                if (this.LogicalTreeElementExtender.SetValue(ref this.availableItems, value))
                {
                    // 古いアイテムにイベントを登録している場合、解除する。
                    if (oldValue != null && oldValue is INotifyCollectionChanged)
                    {
                        ((INotifyCollectionChanged)oldValue).CollectionChanged -=
                            this.OnCollectionChanged;
                    }

                    // INotifyPropertyChangedを継承しているインスタンスの場合、
                    // イベントを登録する。
                    if (value is INotifyCollectionChanged)
                    {
                        ((INotifyCollectionChanged)value).CollectionChanged +=
                            this.OnCollectionChanged;
                    }

                    // アイテムをクリアする。
                    this.stackPanel.Controls.Clear();

                    // GroupItemを登録する。
                    foreach (KeyValuePair<string, object> item in this.availableItems)
                    {
                        UIRadioButton button = this.CreateRadioButton(item);
                        this.stackPanel.Controls.Add(button);
                    }
                }
            }
        }

        #region Event handlers

        /// <summary>
        /// Called when a parent request the desired size.
        /// </summary>
        /// <param name="proposedSize">The available parent size.</param>
        /// <returns>Returns the desired sife of the control.</returns>
        public override Size GetPreferredSize(Size proposedSize)
        {
            if (this.IsSelfOrParentCollapsed() == true)
            {
                return Size.Empty;
            }

            Size contentpanelSize = this.stackPanel.GetPreferredSize(proposedSize);

            return new Size(
                this.Padding.Horizontal + contentpanelSize.Width,
                this.Padding.Vertical + contentpanelSize.Height);
        }

        /// <summary>
        /// Called when a layout process is running.
        /// </summary>
        /// <param name="e">The event argument.</param>
        protected override void OnLayout(LayoutEventArgs e)
        {
            base.OnLayout(e);

            this.stackPanel.Left = this.Padding.Left + this.stackPanel.Margin.Left;
            this.stackPanel.Top = this.Padding.Top + this.stackPanel.Margin.Top;

            this.stackPanel.Width = this.ClientSize.Width
                - (this.Padding.Horizontal + this.stackPanel.Margin.Horizontal);

            this.stackPanel.Height = this.ClientSize.Height
                - (this.Padding.Vertical + this.stackPanel.Margin.Vertical);
        }

        /// <summary>
        /// Called when the RadioButton get checked or unchecked.
        /// </summary>
        /// <param name="sender">Event sender.</param>
        /// <param name="e">Event argument.</param>
        private void OnCheckedChanged(object sender, EventArgs e)
        {
            if (this.ignoreCheckedChanged)
            {
                return;
            }

            UIRadioButton button = (UIRadioButton)sender;

            if (button.Checked == false)
            {
                return;
            }

            this.SelectedIndex = this.stackPanel.Controls
                .FindIndex(cbi => cbi == button);
        }

        /// <summary>
        /// コレクションが変更された時に呼び出されるイベントです.
        /// </summary>
        /// <param name="sender">sender</param>
        /// <param name="e">event</param>
        private void OnCollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            switch (e.Action)
            {
                case NotifyCollectionChangedAction.Add:
                    {
                        // The items have to be KeyValuePairs.
                        var newItems = e.NewItems.Cast<KeyValuePair<string, object>>();
                        if (newItems == null)
                        {
                            return;
                        }

                        // Add the items.
                        if ((e.NewStartingIndex >= 0) &&
                            (e.NewStartingIndex < this.stackPanel.Controls.Count))
                        {
                            // リストの途中に追加
                            var controls = this.stackPanel.Controls;
                            int index = e.NewStartingIndex;

                            // 追加インデックスより後ろのコントロールを一時的に削除
                            int numRemovedButtons = controls.Count - e.NewStartingIndex;
                            UIRadioButton[] removedButtons = new UIRadioButton[numRemovedButtons];

                            for (int i = 0; i < numRemovedButtons; ++i)
                            {
                                removedButtons[i] = (UIRadioButton)controls[e.NewStartingIndex];
                                controls.Remove(removedButtons[i]);
                            }

                            // 新規ラジオボタンを追加
                            foreach (KeyValuePair<string, object> item in newItems)
                            {
                                UIRadioButton button = this.CreateRadioButton(item);
                                controls.Add(button);
                            }

                            // 削除したコントロールを戻す
                            for (int i = 0; i < numRemovedButtons; ++i)
                            {
                                controls.Add(removedButtons[i]);
                            }
                        }
                        else
                        {
                            // リストの最後尾に追加
                            foreach (KeyValuePair<string, object> item in newItems)
                            {
                                UIRadioButton button = this.CreateRadioButton(item);
                                this.stackPanel.Controls.Add(button);
                            }
                        }

                        break;
                    }

                case NotifyCollectionChangedAction.Remove:
                    {
                        // The items have to be KeyValuePairs.
                        var items = e.OldItems.Cast<KeyValuePair<string, object>>();
                        if (items == null)
                        {
                            return;
                        }

                        // Remove the items.
                        foreach (KeyValuePair<string, object> item in items)
                        {
                            // Find the item from the combo box items and remove it.
                            int numRemovedButtons = 0;
                            int index = e.OldItems.IndexOf(item) - numRemovedButtons;

                            if ((index >= 0) && (index < this.stackPanel.Controls.Count))
                            {
                                var controls = this.stackPanel.Controls;
                                controls.Remove(controls[index]);
                            }
                        }

                        break;
                    }

                case NotifyCollectionChangedAction.Replace:
                    {
                        // The items have to be KeyValuePairs.
                        var oldItems = e.OldItems.Cast<KeyValuePair<string, object>>().ToList();
                        var newItems = e.NewItems.Cast<KeyValuePair<string, object>>().ToList();
                        if (oldItems == null || newItems == null)
                        {
                            return;
                        }

                        // Replace the items.
                        for (int i = 0; (i < oldItems.Count) && (i < newItems.Count); ++i)
                        {
                            var controls = this.stackPanel.Controls;

                            // Find the item from the combo box items and replace it.
                            int index = this.availableItems
                                .FindIndex(cbi => object.Equals(cbi, newItems[i]));

                            if ((index >= 0) && (index < controls.Count))
                            {
                                // The item is found, replace it.
                                var button = (UIRadioButton)controls[index];
                                button.Text = newItems[i].Key;
                            }
                        }

                        break;
                    }

                case NotifyCollectionChangedAction.Move:
                    {
                        var controls = this.stackPanel.Controls;

                        if ((e.OldStartingIndex < 0) ||
                            (e.OldStartingIndex >= controls.Count))
                        {
                            return;
                        }

                        int minimumIndex = (e.OldStartingIndex < e.NewStartingIndex) ?
                            e.OldStartingIndex : e.NewStartingIndex;

                        // Save the item first, it will be removed from the list.
                        var item = controls[e.OldStartingIndex];

                        // インデックスより後ろのコントロールを一時的に削除
                        int numRemovedButtons = controls.Count - minimumIndex;
                        UIRadioButton[] removedButtons = new UIRadioButton[numRemovedButtons];

                        // アイテムを削除する。
                        for (int i = 0; i < numRemovedButtons; ++i)
                        {
                            removedButtons[i] = (UIRadioButton)controls[minimumIndex];
                            controls.Remove(removedButtons[i]);
                        }

                        int index = minimumIndex;

                        // 削除したコントロールを戻す
                        for (int i = 0; i < numRemovedButtons; ++i)
                        {
                            // 移動先にアイテムを追加
                            if (index == e.NewStartingIndex - minimumIndex)
                            {
                                controls.Add(removedButtons[e.OldStartingIndex - minimumIndex]);
                                ++index;
                            }

                            // 移動元のアイテムは無視
                            if (i == e.OldStartingIndex - minimumIndex)
                            {
                                continue;
                            }

                            controls.Add(removedButtons[i]);
                            ++index;
                        }

                        break;
                    }

                case NotifyCollectionChangedAction.Reset:
                    {
                        this.SelectedIndex = -1;
                        this.stackPanel.Controls.Clear();

                        break;
                    }
            }
        }

        #endregion

        /// <summary>
        /// アイテムに対応するラジオボタンを作成します.
        /// </summary>
        /// <param name="item">アイテム.</param>
        /// <returns>作成したラジオボタンを返します.</returns>
        private UIRadioButton CreateRadioButton(KeyValuePair<string, object> item)
        {
            var button = new UIRadioButton();

            button.AutoSize = true;
            button.Text = item.Key;
            button.CheckedChanged += this.OnCheckedChanged;

            return button;
        }

        #region Group item class

        /// <summary>
        /// 新しくセットしようとするアイテムリストが適用できるかどうかをチェックします。
        /// </summary>
        /// <param name="newItems">適用しようとしているアイテムリスト</param>
        /// <returns>完全に今のアイテムリストと等しかったらfalse,それ以外はtrueを返します。</returns>
        private bool IsAcceptableItems(IEnumerable<KeyValuePair<string, object>> newItems)
        {
            // どちらかがnullだったらItemsを受け入れる
            if (this.availableItems == null || newItems == null)
            {
                return true;
            }

            var oldList = this.availableItems.ToList();
            var newList = newItems.ToList();

            // 個数が違ったら受け入れる
            if (oldList.Count != newList.Count)
            {
                return true;
            }

            // KeyとValueを比較して等しくないものがあったら受け入れる
            for (int i = 0; i < oldList.Count; ++i)
            {
                if (oldList[i].Key != newList[i].Key)
                {
                    if (Comparer.Default.Compare(oldList[i].Value, newList[i].Value) != 0)
                    {
                        return true;
                    }
                }
            }

            // 等しいItemsだったら受け入れない
            return false;
        }

        /// <summary>
        /// ラジオボタングループに実際に登録されるアイテム
        /// ラジオボタングループに表示される値は、インスタンスのtoString()の
        /// 値となり、意味のない文字列の可能性もあるため、nameに
        /// セットした値を表示する。
        /// </summary>
        public class GroupItem
        {
            /// <summary>
            /// コンストラクタ
            /// </summary>
            /// <param name="name">ラジオボタングループに表示する名前</param>
            /// <param name="value">The value of the radio button group item.</param>
            public GroupItem(string name, object value)
            {
                this.Name = name;
                this.Value = value;
            }

            /// <summary>
            /// ラジオボタングループに表示する名前
            /// </summary>
            public string Name { get; set; }

            /// <summary>
            /// The value of the radio button group item.
            /// </summary>
            public object Value { get; private set; }

            /// <summary>
            /// ラジオボタングループに表示する名前を返す
            /// </summary>
            /// <returns>ラジオボタングループに表示する名前</returns>
            public override string ToString()
            {
                return this.Name;
            }
        }

        #endregion
    }
}
