﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Windows.Forms;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Render.Layout;
using EffectMaker.UIControls.Behaviors;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.EventArguments;
using EffectMaker.UIControls.Extenders;
using EffectMaker.UIControls.Extensions;
using EffectMaker.UIControls.Focus;
using EffectMaker.UIControls.Input;
using EffectMaker.UIControls.Layout;

namespace EffectMaker.UIControls.BaseControls
{
    /// <summary>
    /// An extended Panel class.
    /// </summary>
    public class UIPanel : Panel, IControl, ILayoutElement, IDocumentLinker
    {
        /// <summary>
        /// フォーカスが当たっているタブ
        /// </summary>
        private Control focusedTab = null;

        /// <summary>
        /// Backing field for the Extender property.
        /// </summary>
        private LogicalTreeElementExtender extender;

        /// <summary>
        /// ILayoutElement extender.
        /// </summary>
        private LayoutElementExtender layoutElementExtender;

        /// <summary>
        /// Backing field for Controls property.
        /// </summary>
        private IIndexableCollection<ILogicalTreeElement> controlsWrapper;

        /// <summary>
        /// Backing field for the Resources property.
        /// </summary>
        private IDictionary<string, object> resources = new Dictionary<string, object>();

        /// <summary>
        /// マウスオーバーによるフォーカスにおいてスクロールを抑制するフラグ
        /// </summary>
        internal static bool ScrollBlocking = false;

        /// <summary>
        /// Constructor.
        /// </summary>
        public UIPanel()
        {
            this.AutoSize = true;
            this.extender = new LogicalTreeElementExtender(this);
            this.layoutElementExtender = new LayoutElementExtender(this);
            this.Bindings = new BindingContainer(this);
            this.Behaviors = new BehaviorCollection(this);
            this.BackColor = System.Drawing.Color.Transparent;
            this.FocusScrollEnable = true;
        }

        /// <summary>
        /// Raised when the value of a property on this control changed.
        /// </summary>
#pragma warning disable 67
        public event PropertyChangedEventHandler PropertyChanged;
#pragma warning restore 67

        /// <summary>
        /// VisibilityがCollapsedになった時に元のサイズを覚えておくためのプロパティ
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public Size OriginalSize { get; set; }

        /// <summary>
        /// Gets the resources.
        /// </summary>
        public IDictionary<string, object> Resources
        {
            get { return this.resources; }
        }

        /// <summary>
        /// gets the parent control.
        /// </summary>
        public new ILogicalTreeElement Parent
        {
            get { return base.Parent as ILogicalTreeElement; }
        }

        /// <summary>
        /// Gets the collection of child controls.
        /// </summary>
        public new IIndexableCollection<ILogicalTreeElement> Controls
        {
            get
            {
                if (this.controlsWrapper == null)
                {
                    this.controlsWrapper = new ControlCollectionWrapper(this);
                }

                return this.controlsWrapper;
            }
        }

        /// <summary>
        /// Gets the collection of logical tree elements.
        /// </summary>
        public IIndexableCollection<ILogicalTreeElement> Children
        {
            get
            {
                return this.Controls;
            }
        }

        /// <summary>
        /// Gets the control extender instance of this control.
        /// </summary>
        public LogicalTreeElementExtender LogicalTreeElementExtender
        {
            get { return this.extender; }
        }

        /// <summary>
        /// Gets the binders collection.
        /// </summary>
        public BindingContainer Bindings { get; private set; }

        /// <summary>
        /// Gets the behaviors collection.
        /// </summary>
        public BehaviorCollection Behaviors { get; private set; }

        /// <summary>
        /// Gets or sets the DataContext.
        /// This property may raise a 'DataContext' change notification.
        /// See ControlExtender for more information.
        /// <see cref="ControlExtender"/>
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public object DataContext
        {
            get { return this.extender.DataContext; }
            set { this.extender.DataContext = value; }
        }

        #region Layout

        /// <summary>
        /// Gets the layout element extender instance of this control.
        /// </summary>
        public LayoutElementExtender LayoutElementExtender
        {
            get { return this.layoutElementExtender; }
        }

        /// <summary>
        /// Gets or sets the visibility.
        /// See ControlExtender for more information.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public Visibility Visibility
        {
            get { return this.layoutElementExtender.Visibility; }
            set { this.layoutElementExtender.Visibility = value; }
        }

        /// <summary>
        /// Gets or sets the horizontal alignment.
        /// See ControlExtender for more information.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public HAlignment HorizontalAlignment
        {
            get { return this.layoutElementExtender.HorizontalAlignment; }
            set { this.layoutElementExtender.HorizontalAlignment = value; }
        }

        /// <summary>
        /// Gets or sets the vertical alignment.
        /// See ControlExtender for more information.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public VAlignment VerticalAlignment
        {
            get { return this.layoutElementExtender.VerticalAlignment; }
            set { this.layoutElementExtender.VerticalAlignment = value; }
        }

        #endregion // Layout

        /// <summary>
        /// フォーカスが当たった際の自動スクロールを有効にするか否か
        /// </summary>
        public bool FocusScrollEnable { get; set; }

        /// <summary>
        /// コンテナーで自動スクロールが実行できる場合はtrue。
        /// それ以外の場合はfalse。
        /// オートスクロールが設定された時にホイールハンドラを登録する。
        /// </summary>
        public override bool AutoScroll
        {
            get
            {
                return base.AutoScroll;
            }

            set
            {
                base.AutoScroll = value;
                if (value)
                {
                    GlobalKeyEventHandler.PreFilterMouseWheel += this.HandleWheelScroll;
                }
                else
                {
                    GlobalKeyEventHandler.PreFilterMouseWheel -= this.HandleWheelScroll;
                }
            }
        }

        /// <summary>
        /// ヘルプとしてリンクするページのIDを取得または設定します。
        /// </summary>
        public string DocumentId { get; set; }

        /// <summary>
        /// Clears the DataContext.
        /// See ControlExtender for more details.
        /// <see cref="ControlExtender"/>
        /// </summary>
        public void ClearDataContext()
        {
            this.extender.ClearDataContext();
        }

        /// <summary>
        /// ホイールのためのフォーカス処理
        /// </summary>
        /// <param name="force">コンボボックスから強制的にフォーカスを外す</param>
        public void FocusForWheel(bool force = false)
        {
            var focused = ControlFocusManager.GlobalTabFocusControl;

            if (ScrollBlocking || focused == null || (!force && focused.ContainsFocus))
            {
                return;
            }

            ScrollBlocking = true;
            this.Select();
            ScrollBlocking = false;

            this.focusedTab = focused;
        }

        /// <summary>
        /// Called when a parent request the desired size.
        /// </summary>
        /// <param name="proposedSize">The available parent size.</param>
        /// <returns>Returns the desired sife of the control.</returns>
        public override Size GetPreferredSize(Size proposedSize)
        {
            if (this.IsSelfOrParentCollapsed() == true)
            {
                return Size.Empty;
            }

            Size preferredSize = new Size(0, 0);

            foreach (ILayoutElement child in ((IControl)this).GetNonCollapsedControls())
            {
                Size childSize = child.GetElementDisplaySize(proposedSize);

                int w = childSize.Width - child.Margin.Left + child.Left;
                int h = childSize.Height - child.Margin.Top + child.Top;

                if (w > preferredSize.Width)
                {
                    preferredSize.Width = w;
                }

                if (h > preferredSize.Height)
                {
                    preferredSize.Height = h;
                }
            }

            return preferredSize;
        }

        /// <summary>
        /// あるコントロールにフォーカスが当たった際のスクロール位置を計算する.
        /// </summary>
        /// <param name="activeControl">フォーカスが当たったコントロール</param>
        /// <returns>スクロール位置</returns>
        protected override Point ScrollToControl(Control activeControl)
        {
            if (ScrollBlocking)
            {
                return this.AutoScrollPosition;
            }

            return this.FocusScrollEnable ? base.ScrollToControl(activeControl) : this.AutoScrollPosition;
        }

        /// <summary>
        /// PreFilterによるマウスオーバーだけでスクロール
        /// </summary>
        /// <param name="sender">sender</param>
        /// <param name="e">event</param>
        private void HandleWheelScroll(object sender, PreFilterMouseEventArgs e)
        {
            if (!this.AutoScroll || !this.ClientRectangle.Contains(this.PointToClient(e.Location)))
            {
                return;
            }

            if (this.ContainsFocus)
            {
                return;
            }

            var scroll = this.VerticalScroll;

            var maximum = 1 + scroll.Maximum - scroll.LargeChange; // ユーザが取り得る最大値
            var delta = -(e.Delta / 6) * scroll.SmallChange;
            var offset = Math.Min(Math.Max(scroll.Value + delta, scroll.Minimum), maximum);

            scroll.Value = offset;
            scroll.Value = offset; // 大事なことなので2回いいました
        }
    }
}
