﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;

namespace EffectMaker.UIControls.Specifics
{
    /// <summary>
    /// Default value flag converter for masked values.
    /// 対応している型: uint, ushort
    /// 対応していない型: int, short, etc...
    /// </summary>
    public class MaskedDefaultValueFlagConverter : IValueConverter
    {
        /// <summary>
        /// Convert a value from a type to another.
        /// </summary>
        /// <param name="value">The value to convert.</param>
        /// <param name="targetType">The type to convert to.</param>
        /// <param name="parameter">A custom parameter.</param>
        /// <returns>Returns the converted value.</returns>
        public object Convert(object value, Type targetType, object parameter)
        {
            var data = value as EffectMaker.UILogic.ViewModels.ModificationFlagViewModel.ModificationData;
            if (data == null)
            {
                return true;
            }

            if (data.CurrentValue == null || data.DefaultValue == null)
            {
                return false;
            }

            // マスクを取得
            bool maskConvertError;
            ulong bitMask = this.GetUnsignedValue(parameter, out maskConvertError);

            if (maskConvertError)
            {
                Logger.Log(LogLevels.Warning, "MaskedDefaultValueFlagConverter: invalid parameter.");
                return true;
            }

            // current, originalの値を取得
            bool currentConvertError;
            ulong currentValue = this.GetUnsignedValue(data.CurrentValue, out currentConvertError);

            bool defaultConvertError;
            ulong defaultlValue = this.GetUnsignedValue(data.DefaultValue, out defaultConvertError);

            if (currentConvertError || defaultConvertError)
            {
                Logger.Log(LogLevels.Warning, "MaskedDefaultValueFlagConverter: invalid value.");
                return true;
            }

            return (currentValue & bitMask) == (defaultlValue & bitMask);
        }

        /// <summary>
        /// Convert a value from a type to another.
        /// </summary>
        /// <param name="value">The value to convert.</param>
        /// <param name="targetType">The type to convert to.</param>
        /// <param name="parameter">A custom parameter.</param>
        /// <returns>Returns the converted value.</returns>
        public object ConvertBack(object value, Type targetType, object parameter)
        {
            return null;
        }

        /// <summary>
        /// object型の値をunsigned型に変換します.
        /// </summary>
        /// <param name="value">object型の値</param>
        /// <param name="error">型変換に失敗したときtrue</param>
        /// <returns>unsigned型の値を返します.</returns>
        private ulong GetUnsignedValue(object value, out bool error)
        {
            error = false;
            ulong result = 0;

            if (value is string)
            {
                result = ulong.Parse((string)value);
            }
            else if (value is ulong)
            {
                result = (ulong)value;
            }
            else if (value is ushort)
            {
                result = (ushort)value;
            }
            else if (value is uint)
            {
                result = (uint)value;
            }

            return result;
        }
    }
}
