﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Linq;
using System.Reflection;
using System.Windows.Forms;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.Extensions;
using EffectMaker.UIControls.Layout;
using EffectMaker.UILogic.ValueConverters;

namespace EffectMaker.UIControls.Specifics
{
    /// <summary>
    /// ラベル付きのUIです。
    /// </summary>
    public partial class UIControlSet : UIUserControl, IDocumentLinker
    {
        #region Member variables

        /// <summary>
        /// Stores a flag telling whether the control as been initialized or not.
        /// </summary>
        private bool initializeDone = false;

        /// <summary>
        /// The placement of the caption label.
        /// </summary>
        private LabelPosition labelPos = LabelPosition.Left;

        /// <summary>
        /// Backing field for the Controls property.
        /// </summary>
        private IIndexableCollection<ILogicalTreeElement> controls;

        /// <summary>
        /// The data context for the modification flags.
        /// This data context will be set to the caption label.
        /// </summary>
        private string modificationDataContext = "ModificationFlagViewModel";

        /// <summary>
        /// The data binding for the modification flags.
        /// This data binding will be set to the caption label.
        /// </summary>
        private string modificationFlagTarget = string.Empty;

        #endregion

        #region Constructor

        /// <summary>
        /// Initializes the UIControlSet instance.
        /// </summary>
        public UIControlSet()
        {
            this.InitializeComponent();

            this.PropertyChanged += this.OnPropertyChanged;

            this.initializeDone = true;
        }

        #endregion

        #region Enum for the label placements

        /// <summary>
        /// ラベルの位置
        /// </summary>
        public enum LabelPosition
        {
            /// <summary>左</summary>
            Left,

            /// <summary>上</summary>
            Top
        }

        #endregion

        #region Properties

        /// <summary>
        /// Gets or sets the orientation of the control.
        /// </summary>
        public Layout.Orientation Orientation
        {
            get
            {
                return this.stackPanel.Orientation;
            }

            set
            {
                this.stackPanel.Orientation = value;
            }
        }

        /// <summary>
        /// ラベルの位置を設定または取得します.
        /// </summary>
        [DefaultValue(typeof(LabelPosition), "Left")]
        public LabelPosition LabelPos
        {
            get
            {
                return this.labelPos;
            }

            set
            {
                if (this.labelPos == value)
                {
                    return;
                }

                this.labelPos = value;

                switch (this.labelPos)
                {
                    case LabelPosition.Left:
                        this.stackPanel.Orientation = EffectMaker.UIControls.Layout.Orientation.Horizontal;
                        this.captionLabel.Margin = new Padding(0);
                        break;

                    case LabelPosition.Top:
                        this.stackPanel.Orientation = EffectMaker.UIControls.Layout.Orientation.Vertical;
                        this.captionLabel.Margin = new Padding(0, 0, 0, 8);
                        break;
                }
            }
        }

        /// <summary>
        /// ラベルのテキストを設定または取得します.
        /// </summary>
        [DefaultValue("CaptionLabel")]
        public override string Text
        {
            get
            {
                return this.captionLabel.Text;
            }

            set
            {
                this.captionLabel.Text = value;
            }
        }

        /// <summary>
        /// ラベルの文字色を設定または取得します.
        /// </summary>
        public Color LabelTextColor
        {
            get
            {
                return this.captionLabel.ForeColor;
            }

            set
            {
                this.captionLabel.ForeColor = value;
            }
        }

        /// <summary>
        /// ラベルの背景色を設定または取得します.
        /// </summary>
        public Color LabelBackColor
        {
            get
            {
                return this.captionLabel.BackColor;
            }

            set
            {
                this.captionLabel.BackColor = value;
            }
        }

        /// <summary>
        /// ラベルの線の色を設定または取得します.
        /// </summary>
        public Color LabelBorderColor
        {
            get
            {
                return this.captionLabel.BorderColor;
            }

            set
            {
                this.captionLabel.BorderColor = value;
            }
        }

        /// <summary>
        /// ヘルプとしてリンクするページのIDを取得または設定します。
        /// </summary>
        public string DocumentId
        {
            get { return this.captionLabel.DocumentId; }
            set { this.captionLabel.DocumentId = value; }
        }

        /// <summary>
        /// Gets the collection of content controls
        /// </summary>
        public new IIndexableCollection<ILogicalTreeElement> Controls
        {
            get
            {
                if (this.initializeDone)
                {
                    return this.stackPanel.Controls;
                }
                else
                {
                    if (this.controls == null)
                    {
                        this.controls = new ControlCollectionWrapper(this);
                    }

                    return this.controls;
                }
            }
        }

        /// <summary>
        /// Gets the collection of content controls
        /// </summary>
        public IIndexableCollection<ILogicalTreeElement> SubControls
        {
            get
            {
                return this.initializeDone ? this.subPanel.Controls : null;
            }
        }

        /// <summary>
        /// Get or set the data context for the modification flags.
        /// (the data context for the caption label.)
        /// </summary>
        public string ModificationDataContext
        {
            get
            {
                return this.modificationDataContext;
            }

            set
            {
                if (this.modificationDataContext == value)
                {
                    return;
                }

                this.modificationDataContext = value;
                this.UpdateModificationDataContext();
            }
        }

        /// <summary>
        /// Get or set the data binding target for the modification flags.
        /// (the data binding target for the caption label.)
        /// </summary>
        public string ModificationFlagTarget
        {
            get
            {
                return this.modificationFlagTarget;
            }

            set
            {
                if (this.modificationFlagTarget == value)
                {
                    return;
                }

                this.modificationFlagTarget = value;
                this.UpdateModificationFlagTarget();
            }
        }

        /// <summary>
        /// Get or set the value changed flag converter.
        /// </summary>
        public IValueConverter ValueChangedFlagConverter { get; set; }

        /// <summary>
        /// Get or set the parameter for the value changed flag converter.
        /// </summary>
        public object ValueChangedFlagConverterParam { get; set; }

        /// <summary>
        /// Get or set the default value flag converter.
        /// </summary>
        public IValueConverter DefaultValueFlagConverter { get; set; }

        /// <summary>
        /// Get or set the parameter for the default value flag converter.
        /// </summary>
        public object DefaultValueFlagConverterParam { get; set; }

        /// <summary>
        /// Get or set the data context of the caption label.
        /// </summary>
        public object CaptionLabelDataContext
        {
            get { return this.captionLabel.DataContext; }
            set { this.captionLabel.DataContext = value; }
        }

        #endregion

        #region Event handlers

        /// <summary>
        /// Called when a parent request the desired size.
        /// </summary>
        /// <param name="proposedSize">The available parent size.</param>
        /// <returns>Returns the desired sife of the control.</returns>
        public override Size GetPreferredSize(Size proposedSize)
        {
            if (this.IsSelfOrParentCollapsed() == true)
            {
                return Size.Empty;
            }

            // ControlSetのPaddingを計算
            Size contentpanelSize = new Size(0, 0);
            Size availableSize = proposedSize;

            contentpanelSize.Width += this.Padding.Horizontal;
            availableSize.Width -= this.Padding.Horizontal;

            contentpanelSize.Height += this.Padding.Vertical;
            availableSize.Height -= this.Padding.Vertical;

            // ラベルのサイズを取得
            Size labelSize = this.captionLabel
                .GetElementDisplaySize(availableSize);

            // パネルのサイズを取得
            Size panelSize = this.stackPanel
                .GetElementDisplaySize(availableSize);

            // サブパネルのサイズを取得
            Size subPanelSize = this.subPanel
                .GetElementDisplaySize(availableSize);

            int leftPaneHeight = labelSize.Height + subPanelSize.Height;

            if (this.LabelPos == LabelPosition.Left)
            {
                contentpanelSize.Width += labelSize.Width + panelSize.Width;
                availableSize.Width -= labelSize.Width + panelSize.Width;

                // サイズの大きいコントロールの高さに合わせる
                contentpanelSize.Height += leftPaneHeight > panelSize.Height
                    ? leftPaneHeight : panelSize.Height;
            }
            else
            {
                contentpanelSize.Height += labelSize.Height + panelSize.Height;
                availableSize.Height -= labelSize.Height + panelSize.Height;

                // サイズの大きいコントロールの幅に合わせる
                contentpanelSize.Width += labelSize.Width > panelSize.Width
                    ? labelSize.Width : panelSize.Width;
            }

            return contentpanelSize;
        }

        /// <summary>
        /// Called when a layout process is running.
        /// </summary>
        /// <param name="e">The event argument.</param>
        protected override void OnLayout(LayoutEventArgs e)
        {
            if (this.IsSelfOrParentCollapsed() == true)
            {
                return;
            }

            base.OnLayout(e);

            Size availableSize = this.ClientSize;
            Point location = new Point(0, 0);

            // ControlSetのPaddingを計算
            availableSize.Width -= this.Padding.Horizontal;
            availableSize.Height -= this.Padding.Vertical;

            location.X += this.Padding.Left;
            location.Y += this.Padding.Top;

            // ラベルの位置を設定
            Point captionLocation = new Point(
                location.X + this.captionLabel.Margin.Left,
                location.Y + this.captionLabel.Margin.Top);

            this.captionLabel.Location = captionLocation;

            // ラベルのサイズを設定
            Size labelSize = this.captionLabel
                .GetElementDisplaySize(availableSize);

            this.captionLabel.SetElementDisplaySize(labelSize);

            if (this.LabelPos == LabelPosition.Left)
            {
                location.X += labelSize.Width + this.stackPanel.Margin.Left;

                availableSize.Width -= labelSize.Width;
            }
            else
            {
                location.Y += labelSize.Height + this.stackPanel.Margin.Top;

                availableSize.Height -= labelSize.Height;
            }

            // スタックパネルの位置を設定
            this.stackPanel.Location = location;

            // スタックパネルのサイズを設定
            Size panelSize = this.stackPanel
                .GetElementDisplaySize(availableSize);

            if (panelSize.Width < availableSize.Width)
            {
                panelSize.Width = availableSize.Width;
            }

            this.stackPanel.SetElementDisplaySize(panelSize);

            if (this.stackPanel.Controls.Any() && this.LabelPos == LabelPosition.Left)
            {
                this.subPanel.Location = new Point(0, captionLocation.Y + labelSize.Height);
            }
            else
            {
                this.subPanel.Visibility = Visibility.Collapsed;
            }
        }

        #endregion

        /// <summary>
        /// Handle PropertyChanged event.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "DataContext")
            {
                this.UpdateModificationDataContext();
            }
        }

        /// <summary>
        /// 変更フラグのDataContextを更新.
        /// </summary>
        private void UpdateModificationDataContext()
        {
            // First remove all the binders that binds to the label's "DataContext" property.
            DataBinding.Binder[] bindersToRemove = this.Bindings.Where(
                item => item.ElementPropertyName == "CaptionLabelDataContext").ToArray();

            // There should be at most one binding for the data context.
            if (bindersToRemove != null && bindersToRemove.Length > 1)
            {
                Logger.Log(
                    LogLevels.Error,
                    "UIControlSet.UpdateModificationDataContext : Data context binds to multiple data sources.");
            }

            foreach (DataBinding.Binder binder in bindersToRemove)
            {
                // If the bound data source is the same, don't replace the binding with the new one.
                if (binder.DataSourcePropertyName == this.modificationDataContext)
                {
                    return;
                }

                this.Bindings.Remove(binder);
            }

            // First update the modification flag target property bindings to the caption label.
            this.UpdateModificationFlagTarget();

            // Add new binder to the data context property.
            var newBinder =
                this.AddBinding("CaptionLabelDataContext", this.modificationDataContext);

            newBinder.UpdateElement();
        }

        /// <summary>
        /// 変更フラグのターゲットをアップデート
        /// </summary>
        private void UpdateModificationFlagTarget()
        {
            // Remove old binders.
            {
                // First remove all the binders that binds to the label's "IsStarMarkVisible" property.
                DataBinding.Binder[] bindersToRemove = this.captionLabel.Bindings.Where(
                    item => item.ElementPropertyName == "IsStarMarkVisible").ToArray();
                foreach (DataBinding.Binder binder in bindersToRemove)
                {
                    this.captionLabel.Bindings.Remove(binder);
                }

                // Remove all the binders that binds to the label's "IsDefaultValue" property.
                bindersToRemove = this.captionLabel.Bindings.Where(
                    item => item.ElementPropertyName == "IsDefaultValue").ToArray();
                foreach (DataBinding.Binder binder in bindersToRemove)
                {
                    this.captionLabel.Bindings.Remove(binder);
                }
            }

            // Add the new binders.
            if (string.IsNullOrEmpty(this.ModificationFlagTarget) == false)
            {
                {
                    IValueConverter converter = this.ValueChangedFlagConverter;
                    if (converter == null)
                    {
                        converter = new ValueChangedModificationFlagConverter();
                    }

                    DataBinding.Binder binder = this.captionLabel.AddBinding(
                        "IsStarMarkVisible",
                        this.modificationFlagTarget,
                        converter,
                        this.ValueChangedFlagConverterParam);

                    binder.Mode = DataBinding.BindingMode.OneWay;
                }

                {
                    IValueConverter converter = this.DefaultValueFlagConverter;
                    if (converter == null)
                    {
                        converter = new DefaultValueModificationFlagConverter();
                    }

                    DataBinding.Binder binder = this.captionLabel.AddBinding(
                        "IsDefaultValue",
                        this.modificationFlagTarget,
                        converter,
                        this.DefaultValueFlagConverterParam);

                    binder.Mode = DataBinding.BindingMode.OneWay;
                }
            }
        }
    }
}
