﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Windows.Forms;
using EffectMaker.DataModel.AnimationTable;
using EffectMaker.Foundation.Editting;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Primitives;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Specifics.TabPages;

namespace EffectMaker.UIControls.Specifics.TableEditor
{
    /// <summary>
    /// The scale anime control set.
    /// </summary>
    public partial class AnimationTableControlSet : UIUserControl
    {
        /// <summary>
        /// The loop mode.
        /// </summary>
        private bool loopMode = false;

        /// <summary>
        /// The inter mode.
        /// </summary>
        private int interMode = 0;

        /// <summary>
        /// The values.
        /// </summary>
        private AnimationTableData values = null;

        /// <summary>
        /// 履歴用の差分データ
        /// </summary>
        private AnimationTableData oldValues = new AnimationTableData();

        /// <summary>
        /// ツールチップ
        /// </summary>
        private ToolTip toolTip = null;

        /// <summary>
        /// ValueChangedExecutable実行中かどうかフラグ
        /// </summary>
        private bool changing = false;

        /// <summary>
        /// サムネイルをクリックされた時だけ立つフラグ
        /// </summary>
        private bool thumnailClicked = false;

        /// <summary>
        /// Initializes a new instance of the <see cref="AnimationTableControlSet"/> class.
        /// </summary>
        public AnimationTableControlSet()
        {
            this.InitializeComponent();

            this.label1.Text = Properties.Resources.TableEditorInterMode;

            this.AutoSizeMode = AutoSizeMode.GrowAndShrink;
            this.button_add.AutoSize = false;
            this.button_copy.AutoSize = false;
            this.button_delete.AutoSize = false;
            this.button_loop.AutoSize = false;
            this.button_reset.AutoSize = false;

            this.text_PrevFocus.GotFocus += (s, e) =>
            {
                if ((Control.ModifierKeys & Keys.Shift) == Keys.Shift)
                {
                    // 逆送り
                    if (this.DataGridView.FocusCellByOrder(false))
                    {
                        // 末端のキーならページ全体のタブオーダーへ移る
                        Control test = this.Parent;
                        while (test != null)
                        {
                            var tabPage = test as PropertyTabPageBase;
                            if (tabPage != null)
                            {
                                tabPage.SelectNextChildControl(false);
                                break;
                            }

                            test = test.Parent;
                        }
                    }
                }
                else
                {
                    // 順送り
                    this.DataGridView.FocusCellLeftTop();
                }
            };
            this.text_PostFocus.GotFocus += (s, e) =>
            {
                if ((Control.ModifierKeys & Keys.Shift) == Keys.Shift)
                {
                    // 逆送り
                    this.DataGridView.FocusCellRightBottom();
                }
                else
                {
                    // 順送り
                    if (this.DataGridView.FocusCellByOrder(true))
                    {
                        // 末端のキーならページ全体のタブオーダーへ移る
                        Control test = this.Parent;
                        while (test != null)
                        {
                            var tabPage = test as PropertyTabPageBase;
                            if (tabPage != null)
                            {
                                tabPage.SelectNextChildControl(true);
                                break;
                            }

                            test = test.Parent;
                        }
                    }
                }
            };

            if (this.toolTip == null)
            {
                this.toolTip = new ToolTip
                {
                    AutoPopDelay = 1200,
                    InitialDelay = 1,
                    ReshowDelay = 1
                };

                this.PutToolTip(
                    this.button_add,
                    Properties.Resources.TableEditorAdd);
                this.PutToolTipForInactive(
                    this.button_add,
                    Properties.Resources.TableEditorAdd);
                this.PutToolTip(
                    this.button_copy,
                    Properties.Resources.TableEditorCopy);
                this.PutToolTipForInactive(
                    this.button_copy,
                    Properties.Resources.TableEditorCopy);
                this.PutToolTip(
                    this.button_delete,
                    Properties.Resources.TableEditorDelete);
                this.PutToolTipForInactive(
                    this.button_delete,
                    Properties.Resources.TableEditorDelete);
                this.PutToolTip(
                    this.button_reset,
                    Properties.Resources.TableEditorClear);
                this.PutToolTipForInactive(
                    this.button_reset,
                    Properties.Resources.TableEditorClear);

                this.button_loop.Click += (s, e) => this.toolTip.Show(
                    this.LoopMode ?
                    Properties.Resources.TableEditorLoop : Properties.Resources.TableEditorOneTime,
                    this.button_loop,
                    20,
                    -25,
                    2000);
                this.button_loop.MouseEnter += (s, e) => this.toolTip.Show(
                    this.LoopMode ?
                    Properties.Resources.TableEditorLoop : Properties.Resources.TableEditorOneTime,
                    this.button_loop,
                    20,
                    -25,
                    2000);
                this.button_loop.MouseLeave += (s, e) => this.toolTip.Hide(this.button_loop);
            }
        }

        /// <summary>
        /// リニア編集モードかどうか取得または設定します。
        /// </summary>
        public bool IsLinearMode
        {
            get { return this.DataGridView.IsLinearMode; }
            set { this.DataGridView.IsLinearMode = value; }
        }

        /// <summary>
        /// Gets the columns.
        /// </summary>
        public DataGridViewColumnCollection Columns
        {
            get { return this.DataGridView.Columns; }
        }

        /// <summary>
        /// 行追加が可能か否かを通知するプロパティ
        /// </summary>
        public bool Addable
        {
            get { return this.DataGridView.CanAdd; }
        }

        /// <summary>
        /// 行削除が可能か否かを通知するプロパティ
        /// </summary>
        public bool Deletable
        {
            get { return this.DataGridView.CanDelete; }
        }

        /// <summary>
        /// 行番号カラムの描画モードを指定するプロパティ
        /// (0:色とは無縁の値, 1:RGBカラー, 2:アルファ)
        /// </summary>
        public int IndexDrawMode
        {
            get
            {
                return this.DataGridView.IndexDrawMode;
            }

            set
            {
                this.DataGridView.IndexDrawMode = value;
                this.LogicalTreeElementExtender.NotifyPropertyChanged(
                    BindingUpdateType.PropertyChanged, propertyName: "EditingColorInfo");
            }
        }

        /// <summary>
        /// カラー編集時にカラーピッカーと接続するためのプロパティ
        /// </summary>
        public EditingColorInfo EditingColorInfo
        {
            get
            {
                if (this.Values.Count == 0)
                {
                    return null;
                }

                ColorRgba tmpColor = this.Values[this.DataGridView.LastCurrentRowIndex].ColorValue;

                if (this.IndexDrawMode == 1)
                {
                    tmpColor.A = 0.0f;
                }
                else if (this.IndexDrawMode == 2)
                {
                    tmpColor.A = tmpColor.R;
                    tmpColor.R = 0.0f;
                    tmpColor.G = 0.0f;
                    tmpColor.B = 0.0f;
                }

                bool tabOrderMode = this.thumnailClicked;
                this.thumnailClicked = false;

                return new EditingColorInfo
                {
                    Control = this.IsUpdatingDataContext ||
                        this.DataGridView.CurrentCell == null ? null : this,
                    Color = tmpColor,
                    RgbEditEnabled = this.IndexDrawMode == 1,
                    AlphaEditEnabled = this.IndexDrawMode == 2,
                    TimeEditEnabled = false,
                    RefreshColorPickerRequest = true,
                    LockingTabOrder = tabOrderMode,
                };
            }

            set
            {
                // カラーピッカー選択時にセルの編集を終了
                this.DataGridView.CommitEdit();

                if (value.RefreshRequest)
                {
                    if (!string.IsNullOrEmpty(value.RefreshTargetName))
                    {
                        return;
                    }
                }

                if (value.Control == this)
                {
                    ColorRgba tmpColor = value.Color;
                    for (int i = 0; i < tmpColor.Count; ++i)
                    {
                        tmpColor[i] = (float)Math.Round(tmpColor[i], 3);
                    }

                    var targetRowIndex = this.DataGridView.CurrentCell == null
                        ? this.DataGridView.LastCurrentRowIndex
                        : this.DataGridView.CurrentCell.RowIndex;

                    if (this.IndexDrawMode == 1)
                    {
                        tmpColor.A = 0.0f;
                        this.Values[targetRowIndex].ColorValue = tmpColor;
                    }
                    else if (this.IndexDrawMode == 2)
                    {
                        this.Values[targetRowIndex].ColorValue
                            = new ColorRgba(tmpColor.A, 0.0f, 0.0f, 0.0f);
                    }

                    this.DataGridView.SyncView();

                    if (!this.IsUpdatingDataContext)
                    {
                        if (value.IsEditing)
                        {
                            this.LogicalTreeElementExtender.NotifyPropertyChanged(
                                BindingUpdateType.PropertyChanged, propertyName: "Values");
                            this.UpdateResetState();
                        }
                        else
                        {
                            this.NotifyValueChanged();
                            this.UpdateResetState();
                        }
                    }
                }
                else
                {
                    this.DataGridView.CurrentCell = null;
                }
            }
        }

        /// <summary>
        /// Gets the values.
        /// </summary>
        public AnimationTableData Values
        {
            get
            {
                return this.DataGridView.Values;
            }

            set
            {
                if (value == null)
                {
                    return;
                }

                this.values = value;
                this.DataGridView.Values = value;
                this.LogicalTreeElementExtender.SetValue(ref this.values, value);

                if (!this.changing)
                {
                    this.oldValues.Set(this.values);
                }

                this.UpdateResetState();
            }
        }

        /// <summary>
        /// Gets or sets the loop mode.
        /// </summary>
        public bool LoopMode
        {
            get
            {
                return this.loopMode;
            }

            set
            {
                this.loopMode = value;
                this.button_loop.Image = this.loopMode == false
                    ? Properties.Resources.Icon_AnimTable_Loop_Off
                    : Properties.Resources.Icon_AnimTable_Loop_On;

                this.LogicalTreeElementExtender.SetValue(ref this.loopMode, value);

                this.UpdateResetState();
            }
        }

        /// <summary>
        /// Gets or sets the loop mode.
        /// </summary>
        public int InterMode
        {
            get
            {
                return this.interMode;
            }

            set
            {
                this.interMode = value;
                this.comboBox_interMode.SelectedIndex = value;
                this.LogicalTreeElementExtender.SetValue(ref this.interMode, value);

                this.UpdateResetState();
            }
        }

        /// <summary>
        /// リセット処理を行うExecutableを設定・取得します.
        /// </summary>
        public IExecutable ResetExecutable { get; set; }

        /// <summary>
        /// リセット処理の対象名を設定・取得します
        /// </summary>
        public string ResetTarget { get; set; }

        /// <summary>
        /// Gets or sets the IExecutable to run when the Value property changes.
        /// </summary>
        public IExecutable ValueChangedExecutable { get; set; }

        /// <summary>
        /// リセット処理の対象名を設定・取得します
        /// </summary>
        public string ValueChangedExecutableParameter { get; set; }

        /// <summary>
        /// The on load.
        /// </summary>
        /// <param name="e">
        /// The e.
        /// </param>
        protected override void OnLoad(EventArgs e)
        {
            this.DataGridViewOnCurrentCellChanged();
            this.DataGridView.AdjustSize();
            this.DataGridView.CurrentCell = null;
            this.DataGridView.OnCurrentCellChangedBlockable += this.DataGridViewOnCurrentCellChanged;
            this.DataGridView.OnHeaderClicked = () =>
            {
                if (this.IndexDrawMode != 0)
                {
                    this.thumnailClicked = true;
                    this.LogicalTreeElementExtender.NotifyPropertyChanged(
                        BindingUpdateType.PropertyChanged, propertyName: "EditingColorInfo");
                }
            };
            this.DataGridView.OnBeginCellEdit = () =>
            {
                this.button_add.Enabled = false;
                this.button_delete.Enabled = false;
                this.button_copy.Enabled = false;

                if (this.IndexDrawMode != 0)
                {
                    this.EditingColorInfo.RefreshColorPickerRequest = true;
                    this.LogicalTreeElementExtender.NotifyPropertyChanged(
                        BindingUpdateType.PropertyChanged, propertyName: "EditingColorInfo");
                }

                this.UpdateResetState();
            };
            this.DataGridView.OnEndCellEdit = () =>
            {
                this.button_add.Enabled = this.Addable;
                this.button_delete.Enabled = this.Deletable;
                this.button_copy.Enabled = this.Addable;

                if (!this.IsUpdatingDataContext)
                {
                    this.NotifyValueChanged();
                }

                if (this.IndexDrawMode != 0)
                {
                    this.EditingColorInfo.RefreshColorPickerRequest = true;
                    this.LogicalTreeElementExtender.NotifyPropertyChanged(
                        BindingUpdateType.PropertyChanged, propertyName: "EditingColorInfo");
                }

                this.UpdateResetState();
            };

            // 初期値をデータモデルに通知
            this.LogicalTreeElementExtender.NotifyPropertyChanged(
                BindingUpdateType.PropertyChanged, propertyName: "Values");

            if (this.IndexDrawMode != 0)
            {
                this.LogicalTreeElementExtender.NotifyPropertyChanged(
                    BindingUpdateType.PropertyChanged, propertyName: "EditingColorInfo");
            }

            base.OnLoad(e);
        }

        /// <summary>
        /// VisibleChangedの際にカラーピッカーとの接続状態を更新
        /// </summary>
        /// <param name="e">event.</param>
        protected override void OnVisibleChanged(EventArgs e)
        {
            if (this.Visible)
            {
                if (this.IndexDrawMode != 0)
                {
                    this.LogicalTreeElementExtender.NotifyPropertyChanged(
                        BindingUpdateType.PropertyChanged, propertyName: "EditingColorInfo");
                }
            }

            base.OnVisibleChanged(e);
        }

        /// <summary>
        /// Called when the Value property changes.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected virtual void OnValueChanged(ValueChangedExEventArgs e)
        {
            IExecutable executable = this.ValueChangedExecutable;
            if (executable != null && executable.CanExecute(e))
            {
                executable.Execute(e);
            }
        }

        /// <summary>
        /// The button_add_ click.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// The e.
        /// </param>
        private void Button_add_Click(object sender, EventArgs e)
        {
            this.DataGridView.AddRow(false);

            this.NotifyValueChanged();

            this.DataGridView.Select();
            if (this.IndexDrawMode != 0)
            {
                this.LogicalTreeElementExtender.NotifyPropertyChanged(
                    BindingUpdateType.PropertyChanged, propertyName: "EditingColorInfo");
            }

            this.UpdateResetState();
        }

        /// <summary>
        /// The button_delete_ click.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// The e.
        /// </param>
        private void Button_delete_Click(object sender, EventArgs e)
        {
            this.DataGridView.DeleteRow();

            this.NotifyValueChanged();

            this.DataGridView.Select();
            if (this.IndexDrawMode != 0)
            {
                this.LogicalTreeElementExtender.NotifyPropertyChanged(
                    BindingUpdateType.PropertyChanged, propertyName: "EditingColorInfo");
            }

            this.UpdateResetState();
        }

        /// <summary>
        /// The button_copy_ click.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// The e.
        /// </param>
        private void Button_copy_Click(object sender, EventArgs e)
        {
            this.DataGridView.AddRow(true);

            this.NotifyValueChanged();

            this.DataGridView.Select();
            if (this.IndexDrawMode != 0)
            {
                this.LogicalTreeElementExtender.NotifyPropertyChanged(
                    BindingUpdateType.PropertyChanged, propertyName: "EditingColorInfo");
            }

            this.UpdateResetState();
        }

        /// <summary>
        /// The button_loop mode_ click.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// The e.
        /// </param>
        private void Button_loopMode_Click(object sender, EventArgs e)
        {
            this.LoopMode = !this.LoopMode;

            this.LogicalTreeElementExtender.NotifyPropertyChanged(
                BindingUpdateType.PropertyChanged, propertyName: "LoopMode");
            this.DataGridView.Select();
            if (this.IndexDrawMode != 0)
            {
                this.LogicalTreeElementExtender.NotifyPropertyChanged(
                    BindingUpdateType.PropertyChanged, propertyName: "EditingColorInfo");
            }
        }

        /// <summary>
        /// The button_reset_ click.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// The e.
        /// </param>
        private void Button_reset_Click(object sender, EventArgs e)
        {
            if (this.ResetExecutable != null)
            {
                this.ResetExecutable.Execute(this.ResetTarget);
            }

            this.DataGridView.Select();

            if (this.IndexDrawMode != 0)
            {
                this.LogicalTreeElementExtender.NotifyPropertyChanged(
                    BindingUpdateType.PropertyChanged, propertyName: "EditingColorInfo");
            }

            this.UpdateResetState();
        }

        /// <summary>
        /// The inter mode_ selected index changed.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// The e.
        /// </param>
        private void InterMode_SelectedIndexChanged(object sender, EventArgs e)
        {
            this.InterMode = this.comboBox_interMode.SelectedIndex;
            this.LogicalTreeElementExtender.NotifyPropertyChanged(
                BindingUpdateType.PropertyChanged, propertyName: "InterMode");
            this.DataGridView.Select();
            if (this.IndexDrawMode != 0)
            {
                this.LogicalTreeElementExtender.NotifyPropertyChanged(
                    BindingUpdateType.PropertyChanged, propertyName: "EditingColorInfo");
            }
        }

        /// <summary>
        /// OnValueChangedを呼び出して、履歴に積む値を確定します。
        /// </summary>
        private void NotifyValueChanged()
        {
            this.changing = true;

            this.LogicalTreeElementExtender.NotifyPropertyChanged(
                BindingUpdateType.PropertyChanged, propertyName: "Values");
            this.OnValueChanged(new ValueChangedExEventArgs(
                this.oldValues,
                this.DataGridView.Values,
                false,
                this.ValueChangedExecutableParameter));
            this.oldValues.Set(this.DataGridView.Values);

            this.changing = false;
        }

        /// <summary>
        /// The data grid view on cell value changed.
        /// </summary>
        private void DataGridViewOnCurrentCellChanged()
        {
            this.button_add.Enabled = this.Addable;
            this.button_delete.Enabled = this.Deletable;
            this.button_copy.Enabled = this.Addable;
            if (this.IndexDrawMode != 0)
            {
                this.LogicalTreeElementExtender.NotifyPropertyChanged(
                    BindingUpdateType.PropertyChanged, propertyName: "EditingColorInfo");
            }
        }

        /// <summary>
        /// コントロールの上にツールチップを乗せます.
        /// </summary>
        /// <param name="ctl">
        /// ツールチップを乗せたいコントロール.
        /// </param>
        /// <param name="str">
        /// 乗せるテキスト.
        /// </param>
        private void PutToolTip(Control ctl, string str)
        {
            ctl.MouseEnter += (s, e) => this.toolTip.Show(str, ctl, 20, -25, 2000);
            ctl.MouseLeave += (s, e) => this.toolTip.Hide(ctl);
        }

        /// <summary>
        /// 非アクティブなコントロールの上にツールチップを乗せます.
        /// </summary>
        /// <param name="ctl">
        /// ツールチップを乗せたいコントロール.
        /// </param>
        /// <param name="str">
        /// 乗せるテキスト.
        /// </param>
        private void PutToolTipForInactive(Control ctl, string str)
        {
            this.MouseMove += (s, e) =>
            {
                if (!ctl.Enabled)
                {
                    if (this.toolTip.Tag != ctl &&
                        ctl.ClientRectangle.Contains(ctl.PointToClient(Cursor.Position)))
                    {
                        this.toolTip.Show(str, ctl, 20, -25, 2000);
                        this.toolTip.Tag = ctl;
                    }
                    else if (this.toolTip.Tag == ctl &&
                        !ctl.ClientRectangle.Contains(ctl.PointToClient(Cursor.Position)))
                    {
                        this.toolTip.Hide(ctl);
                        this.toolTip.Tag = null;
                    }
                }
            };
        }

        /// <summary>
        /// リセットボタンの状態更新
        /// </summary>
        private void UpdateResetState()
        {
            // セルの編集中は無効
            if (this.DataGridView.IsCurrentCellInEditMode)
            {
                this.button_reset.Enabled = false;
                return;
            }

            // テーブルの内容がデフォルト値のときは無効
            if (!this.LoopMode && this.InterMode == 0)
            {
                if (this.Values.Count == 1)
                {
                    var keyValue = this.Values[0].Value;

                    Primitivef prim = new Vector4f(1.0f, 1.0f, 1.0f, 1.0f);
                    if (prim.Equals(keyValue) && this.Values[0].Frame == 0)
                    {
                        this.button_reset.Enabled = false;
                        return;
                    }
                }
            }

            this.button_reset.Enabled = true;
        }
    }
}
