﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Diagnostics;
using System.Diagnostics.CodeAnalysis;
using System.Drawing;
using System.Drawing.Printing;
using System.IO;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.Xml.Schema;
using EffectMaker.Foundation.Coercers;
using EffectMaker.Foundation.Core;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Primitives;
using EffectMaker.Foundation.Render.Layout;
using EffectMaker.Foundation.Utility;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Layout;
using EffectMaker.UIControls.Specifics.Sliders.ValueTransforms;
using EffectMaker.UIControls.Threading;
using Orientation = EffectMaker.UIControls.Layout.Orientation;

namespace EffectMaker.UIControls.Specifics.Sliders
{
    /// <summary>
    /// A custom slider control.
    /// </summary>
    public partial class UISlider : UIUserControl
    {
        /// <summary>
        /// Backing field for the Primitive property.
        /// </summary>
        private IPrimitive primitive;

        /// <summary>
        /// Backing field for the Random Primitive property.
        /// </summary>
        private IPrimitive randomPrimitive;

        /// <summary>
        /// Stores the underlying type of the primitive.
        /// </summary>
        private Type dataType;

        /// <summary>
        /// Stores the flag telling whether data should be updated when UI changes or not.
        /// </summary>
        private bool suspendDataUpdate;

        /// <summary>
        /// Stores the previous value when the track bar value was keeping changing.
        /// </summary>
        private IPrimitive previousThrottledValue;

        /// <summary>
        /// Stores the previous value when the track bar value got validated.
        /// </summary>
        private IPrimitive previousValidationValue;

        /// <summary>
        /// Stores the previous value when the track bar value got validated.
        /// </summary>
        private IPrimitive previousValidationRandomValue;

        /// <summary>
        /// Flag telling whether the value is still changing or is validated.
        /// </summary>
        private bool isChanging;

        /// <summary>
        /// Backing field for the Minimum property.
        /// </summary>
        private float minimum = float.MinValue;

        /// <summary>
        /// Backing field for the Maximum property.
        /// </summary>
        private float maximum = float.MaxValue;

        /// <summary>
        /// Backing field for the Minimum property.
        /// </summary>
        private float randomMinimum = 0.0f;

        /// <summary>
        /// Backing field for the Maximum property.
        /// </summary>
        private float randomMaximum = float.MaxValue;

        /// <summary>
        /// Stores the value transform mode.
        /// </summary>
        private int valueMode;

        /// <summary>
        /// Stores the value transform algorithm.
        /// </summary>
        private IValueTransform<float[]> valueTransform;

        /// <summary>
        /// Flag telling whether to run or skip some events.
        /// </summary>
        private bool eventReentranceLocked;

        /// <summary>
        /// The label list.
        /// </summary>
        private List<UILabel> labelList = null;

        /// <summary>
        /// 比率固定・連動UIの表示状態.
        /// </summary>
        private bool showValueTransformButtons = true;

        /// <summary>
        /// The delta pos.
        /// </summary>
        private int deltaPos = 3;   // 精度1がデフォルト(オフセット適用前)

        /// <summary>
        /// The tool tip.
        /// </summary>
        private ToolTip toolTip = new ToolTip();

        /// <summary>
        /// 変換前の値が飛ぶのをブロックするフラグ
        /// </summary>
        private bool blockingUpdate = false;

        /// <summary>
        /// 整数型限定フラグ
        /// </summary>
        private bool integerOnly = false;

        /// <summary>
        /// コントロールモード
        /// </summary>
        private UIDialSlider.ControlModeType controlMode;

        /////// <summary>
        /////// ラベルの下に移動済みならtrue,そうでなければ
        /////// </summary>
        ////private bool moved = false;

        /// <summary>
        /// ValueChangedExecutableが実行中フラグ.
        /// </summary>
        private bool runnningVCE = false;

        /// <summary>
        /// Initializes the UISlider instance.
        /// </summary>
        public UISlider()
        {
            this.InitializeComponent();

            LayoutEngineBase.SuspendLayout();

            this.AddImageButton(
                "valueTransformIndividual",
                global::EffectMaker.UIControls.Specifics.Properties.Resources.Slider_Separate_Off,
                0);

            this.AddImageButton(
                "valueTransformUniform",
                global::EffectMaker.UIControls.Specifics.Properties.Resources.Slider_Link_On,
                1,
                true);

            this.AddImageButton(
                "valueTransformFixedRatio",
                global::EffectMaker.UIControls.Specifics.Properties.Resources.Slider_Ratio_Off,
                2);

            this.TransformMode = 1;

            this.toolTip.AutoPopDelay = 1200;
            this.toolTip.InitialDelay = 1;
            this.toolTip.ReshowDelay = 1;

            LayoutEngineBase.ResumeLayout();
        }

        /// <summary>
        /// Enum for distinguish modified value types.
        /// </summary>
        private enum ModifiedValueTypes
        {
            /// <summary>The main slider values.</summary>
            PrimaryValues,

            /// <summary>The random values.</summary>
            RandomValues,
        }

        /// <summary>
        /// 比率固定・連動UIの表示状態の設定・変更.
        /// </summary>
        public bool ShowValueTransformButtons
        {
            get
            {
                return this.showValueTransformButtons;
            }

            set
            {
                this.showValueTransformButtons = value;

                // ラジオボタンの表示状態を変更.
                foreach (var item in this.stackPanelOptions.Controls.OfType<UIUserControl>())
                {
                    item.Visibility = this.showValueTransformButtons ?
                        Visibility.Visible : Visibility.Collapsed;
                }
            }
        }

        /// <summary>
        /// Gets or sets the delta level.
        /// UIDialSlider側でオフセットのズレを吸収しているので、こちらでは何もしません。
        /// </summary>
        public int DeltaLevel
        {
            get
            {
                return this.deltaPos;
            }

            set
            {
                this.deltaPos = value;

                for (int i = 0; i < this.rootStackPanel.Controls.Count; ++i)
                {
                    ((UIDialSlider)this.rootStackPanel.Controls[i]).DeltaLevel = this.deltaPos;
                }
            }
        }

        /// <summary>
        /// 整数値のみを取る場合はこの値をtrueにします.
        /// </summary>
        public bool IntegerOnly
        {
            get
            {
                return this.integerOnly;
            }

            set
            {
                this.integerOnly = value;
                foreach (var s in this.rootStackPanel.Controls.OfType<UIDialSlider>())
                {
                    s.IntegerOnly = this.integerOnly;
                }
            }
        }

        /// <summary>
        /// コントロールモードを取得または設定します。
        /// </summary>
        public UIDialSlider.ControlModeType ControlMode
        {
            get
            {
                return this.controlMode;
            }

            set
            {
                this.controlMode = value;
                foreach (var s in this.rootStackPanel.Controls.OfType<UIDialSlider>())
                {
                    s.ControlMode = this.controlMode;
                }
            }
        }

        /// <summary>
        /// Sets the delta.
        /// </summary>
        public float Delta
        {
            set
            {
                for (int i = 0; i < this.rootStackPanel.Controls.Count; ++i)
                {
                    ((UIDialSlider)this.rootStackPanel.Controls[i]).Delta = value;
                    this.deltaPos = ((UIDialSlider)this.rootStackPanel.Controls[i]).DeltaLevel;
                }
            }
        }

        /// <summary>
        /// Gets or sets the minimum allowed value.
        /// </summary>
        public float Minimum
        {
            get
            {
                return this.minimum;
            }

            set
            {
                if (this.minimum == value)
                {
                    return;
                }

                this.minimum = value;

                foreach (var ctrl in this.PrimitiveControls)
                {
                    ctrl.Minimum = this.minimum;
                }
            }
        }

        /// <summary>
        /// Gets or sets the maximum allowed value.
        /// </summary>
        public float Maximum
        {
            get
            {
                return this.maximum;
            }

            set
            {
                if (this.maximum == value)
                {
                    return;
                }

                this.maximum = value;

                foreach (var ctrl in this.PrimitiveControls)
                {
                    ctrl.Maximum = this.maximum;
                }
            }
        }

        /// <summary>
        /// Gets or sets the minimum allowed value.
        /// </summary>
        public float RandomMinimum
        {
            get
            {
                return this.randomMinimum;
            }

            set
            {
                if (this.randomMinimum == value)
                {
                    return;
                }

                this.randomMinimum = value;

                foreach (var ctrl in this.RandomPrimitiveControls)
                {
                    ctrl.Minimum = this.randomMinimum;
                }
            }
        }

        /// <summary>
        /// Gets or sets the maximum allowed value.
        /// </summary>
        public float RandomMaximum
        {
            get
            {
                return this.randomMaximum;
            }

            set
            {
                if (this.randomMaximum == value)
                {
                    return;
                }

                this.randomMaximum = value;

                foreach (var ctrl in this.RandomPrimitiveControls)
                {
                    ctrl.Maximum = this.randomMaximum;
                }
            }
        }

        /// <summary>
        /// Gets or sets the primitive used with the UISlider control.
        /// It must be of type PrimitiveBase(T).
        /// </summary>
        public IPrimitive Primitive
        {
            get
            {
                return this.primitive;
            }

            set
            {
                if (this.primitive == null && value == null)
                {
                    return;
                }

                IPrimitive p = null;

                if (value != null)
                {
                    p = (IPrimitive)value.Clone();
                }

                // アップデートのブロック中はprimitiveの値を更新するだけ
                if (this.blockingUpdate)
                {
                    this.primitive = p;
                }
                else
                {
                    this.LogicalTreeElementExtender.SetValue(ref this.primitive, p);
                }

                this.OnPrimitiveSet();
            }
        }

        /// <summary>
        /// Gets or sets the random primitive used with the UISlider control.
        /// It must be of type PrimitiveBase(T).
        /// </summary>
        public IPrimitive RandomPrimitive
        {
            get
            {
                return this.randomPrimitive;
            }

            set
            {
                if (this.randomPrimitive == null && value == null)
                {
                    return;
                }

                IPrimitive p = null;

                if (value != null)
                {
                    p = (IPrimitive)value.Clone();
                }

                this.LogicalTreeElementExtender.SetValue(ref this.randomPrimitive, p);
                this.OnRandomPrimitiveSet();
            }
        }

        /// <summary>
        /// Gets or sets the IExecutable to run when the Value property changes.
        /// </summary>
        public IExecutable ValueChangedExecutable { get; set; }

        /// <summary>
        /// Gets or sets the custom parameter of the IExecutable
        /// to run when the Value property changes.
        /// </summary>
        public object ValueChangedExecutableParameter { get; set; }

        /// <summary>
        /// Gets or sets the custom parameter of the IExecutable
        /// to run when the Value property changes.
        /// </summary>
        public object RandomValueChangedExecutableParameter { get; set; }

        /// <summary>
        /// Gets the lable list.
        /// </summary>
        public List<UILabel> LabelList
        {
            get
            {
                if (this.labelList == null)
                {
                    this.labelList = new List<UILabel>
                    {
                        new UILabel()
                        {
                            Size = new Size(12, 12),
                            Text = "X",
                            ForeColor = Color.Red,
                            Padding = new Padding(0),
                        },
                        new UILabel()
                        {
                            Size = new Size(12, 12),
                            Text = "Y",
                            ForeColor = Color.FromArgb(0, 192, 0),
                            Padding = new Padding(0),
                        },
                        new UILabel()
                        {
                            Size = new Size(12, 12),
                            Text = "Z",
                            ForeColor = Color.FromArgb(0, 0, 192),
                            Padding = new Padding(0),
                        },
                        new UILabel()
                        {
                            Size = new Size(12, 12),
                            Text = "W",
                            ForeColor = Color.Black,
                            Padding = new Padding(0),
                        },
                    };
                }

                return this.labelList;
            }
        }

        /// <summary>
        /// Gets or sets the transform mode.
        /// </summary>
        public int TransformMode
        {
            get
            {
                return this.valueMode;
            }

            set
            {
                this.valueMode = value;

                // ラジオボタンの表示状態を変更.
                foreach (var buttonBase in this.stackPanelOptions.Controls.OfType<UIUserControl>())
                {
                    var elem = buttonBase.Controls[0] as UIButton;

                    switch ((int)elem.Tag)
                    {
                        case 0:
                            elem.BackgroundImage = 0 == this.valueMode ?
                                global::EffectMaker.UIControls.Specifics.Properties.Resources.Slider_Separate_On :
                                global::EffectMaker.UIControls.Specifics.Properties.Resources.Slider_Separate_Off;
                            break;
                        case 1:
                            elem.BackgroundImage = 1 == this.valueMode ?
                                global::EffectMaker.UIControls.Specifics.Properties.Resources.Slider_Link_On :
                                global::EffectMaker.UIControls.Specifics.Properties.Resources.Slider_Link_Off;
                            break;
                        case 2:
                            elem.BackgroundImage = 2 == this.valueMode ?
                                global::EffectMaker.UIControls.Specifics.Properties.Resources.Slider_Ratio_On :
                                global::EffectMaker.UIControls.Specifics.Properties.Resources.Slider_Ratio_Off;
                            break;
                        default:
                            throw new InvalidDataException();
                    }
                }

                this.valueTransform = null;
            }
        }

        /// <summary>
        /// ラジオボタンが移動しているのでVisibilityを手動で通知する必要がある
        /// </summary>
        public new Visibility Visibility
        {
            get
            {
                return base.Visibility;
            }

            set
            {
                base.Visibility = value;

                if (!this.showValueTransformButtons || this.stackPanelOptions.Parent == this)
                {
                    return;
                }

                // ラジオボタンの表示状態を変更.
                this.stackPanelOptions.Visibility = this.Visibility;
            }
        }

        /// <summary>
        /// Primitive 編集コントロール
        /// </summary>
        private IEnumerable<UIDialSlider> PrimitiveControls
        {
            get
            {
                return this.rootStackPanel.Controls.OfType<UIDialSlider>().Where(x => (x is UIRandomRangeDialSlider) == false);
            }
        }

        /// <summary>
        /// RandomPrimitive 編集コントロール
        /// </summary>
        private IEnumerable<UIRandomRangeDialSlider> RandomPrimitiveControls
        {
            get
            {
                return this.rootStackPanel.Controls.OfType<UIRandomRangeDialSlider>();
            }
        }

        /// <summary>
        /// The on visible changed.
        /// </summary>
        /// <param name="e">
        /// The e.
        /// </param>
        protected override void OnVisibleChanged(EventArgs e)
        {
            this.ResetTransformMode();
            base.OnVisibleChanged(e);
        }

        /// <summary>
        /// The reset transform mode.
        /// </summary>
        private void ResetTransformMode()
        {
            if (this.primitive == null ||
                this.primitive.Count == 0 ||
                this.runnningVCE)
            {
                return;
            }

            if (!this.ShowValueTransformButtons)
            {
                this.TransformMode = 0;
                return;
            }

            bool isSame = true;

            if (this.primitive.ValueType == ValueTypes.Int)
            {
                int elemValue = (int)this.primitive[0];
                for (int i = 1; i < this.primitive.Count; ++i)
                {
                    if (Math.Abs((int)this.primitive[i] - elemValue) > 0)
                    {
                        isSame = false;
                        break;
                    }
                }
            }
            else if (this.primitive.ValueType == ValueTypes.Float)
            {
                float elemValue = (float)this.primitive[0];
                for (int i = 1; i < this.primitive.Count; ++i)
                {
                    if (Math.Abs((float)this.primitive[i] - elemValue) > 0.000001f)
                    {
                        isSame = false;
                        break;
                    }
                }
            }
            else
            {
                throw new Exception("想定していないPrimitive");
            }

            this.TransformMode = isSame ? 1 : 0;
        }

        /// <summary>
        /// Called when the Primitive property is set with a new value.
        /// It checks validity of the primitive, and then updates the UI controls.
        /// </summary>
        private void OnPrimitiveSet()
        {
            if (this.primitive == null)
            {
                this.rootStackPanel.Controls.Clear();
                return;
            }

            this.previousThrottledValue = (IPrimitive)this.primitive.Clone();

            if (this.isChanging == false)
            {
                this.previousValidationValue = (IPrimitive)this.primitive.Clone();
            }

            if (this.primitive.Count < 0)
            {
                PrimitiveExtensions.ThrowInvalidPrimitiveTypeException(
                    this.primitive,
                    "Primitive");
            }

            if (this.primitive.IsPrimitiveBase(out this.dataType) == false)
            {
                PrimitiveExtensions.ThrowInvalidPrimitiveTypeException(
                    this.primitive,
                    "Primitive");
            }

            this.UpdateControls(false);
            this.ResetTransformMode();
        }

        /// <summary>
        /// Called when the RandomPrimitive property is set with a new value.
        /// It checks validity of the primitive, and then updates the UI controls.
        /// </summary>
        private void OnRandomPrimitiveSet()
        {
            this.previousThrottledValue = (IPrimitive)this.randomPrimitive.Clone();

            if (this.isChanging == false)
            {
                this.previousValidationRandomValue = (IPrimitive)this.randomPrimitive.Clone();
            }

            if (this.randomPrimitive.Count < 0)
            {
                PrimitiveExtensions.ThrowInvalidPrimitiveTypeException(
                    this.randomPrimitive,
                    "RandomPrimitive");
            }

            if (this.randomPrimitive.IsPrimitiveBase(out this.dataType) == false)
            {
                PrimitiveExtensions.ThrowInvalidPrimitiveTypeException(
                    this.randomPrimitive,
                    "RandomPrimitive");
            }

            this.UpdateControls(true);
        }

        /// <summary>
        /// Updates the track bars, create new one(s) or delete existing one(s) if needed.
        /// </summary>
        /// <param name="isRandom">ランダム幅関係か？</param>
        private void UpdateControls(bool isRandom)
        {
            LayoutEngineBase.SuspendLayout();

            var primitiveControlCount = this.PrimitiveControls.Count();
            var randomPrimitiveControlCount = this.RandomPrimitiveControls.Count();

            var isControlUpdated = false;

            if (isRandom == false)
            {
                // Primitive
                if (this.primitive != null)
                {
                    // スライダの生成と追加
                    {
                        var diff = this.primitive.Count - primitiveControlCount;
                        if (diff > 0)
                        {
                            for (var i = 0; i != diff; ++i)
                            {
                                var slider = this.CreateChildSliderControl(false, i);
                                slider.AutoSize = false;
                                slider.Width += 12;
                                foreach (var ctl in slider.Controls.OfType<Control>())
                                {
                                    ctl.Location = new Point(ctl.Left + 12, ctl.Top);
                                }

                                slider.Controls.Add(this.LabelList[i]);
                                this.LabelList[i].Location = new Point(0, 4);

                                this.rootStackPanel.Controls.Add(slider);
                            }
                        }
#if false
                        else if (diff < 0)
                        {
                            System.Diagnostics.Debug.Assert(false);     // 未実装
                        }
#endif
                        isControlUpdated = diff != 0;
                    }
                }
            }

            if (isRandom)
            {
                // RandomPrimitive
                if (this.randomPrimitive != null)
                {
                    // スライダの生成と追加
                    {
                        var diff = this.randomPrimitive.Count - randomPrimitiveControlCount;
                        if (diff > 0)
                        {
                            for (var i = 0; i != diff; ++i)
                            {
                                this.rootStackPanel.Controls.Add(this.CreateChildSliderControl(true, i));
                            }
                        }
#if false
                        else if (diff < 0)
                        {
                            System.Diagnostics.Debug.Assert(false);     // 未実装
                        }
#endif
                        isControlUpdated = isControlUpdated || (diff != 0);
                    }
                }
            }

            // UIControlSetのサブパネルへのお引越し
/*            if (!this.moved)
            {
                var parent = this.Parent;
                while (parent != null)
                {
                    if (parent is UIControlSet)
                    {
                        var uiControlSet = parent as UIControlSet;
                        if (uiControlSet.LabelPos == UIControlSet.LabelPosition.Left)
                        {
                            this.Controls.Remove(this.stackPanelOptions);
                            uiControlSet.SubControls.Add(this.stackPanelOptions);
                        }

                        this.moved = true;
                        break;
                    }

                    parent = parent.Parent;
                }
            }*/

            if (isControlUpdated)
            {
                this.UpdateControlsMargin();
            }

            LayoutEngineBase.ResumeLayout();

            this.UpdateViewFromData(isRandom);
        }

        /// <summary>
        /// スライダーのマージンを再計算します.
        /// </summary>
        private void UpdateControlsMargin()
        {
            // コントロール列間インターバル。ランダムを考慮するためのスケール値
            var intervalScale = this.RandomPrimitiveControls.Any() ? 3 : 7;

            // 比率固定・連動UIの表示がされていない場合は、インターバルを0にする
            if (this.ShowValueTransformButtons == false)
            {
                intervalScale = 0;
            }

            // アイコン列のマージンをここで調整
            this.stackPanelOptions.Margin = new Padding(58, 0, 0, intervalScale);

            // ソート
            {
                var sortedControls = this.PrimitiveControls.Concat(this.RandomPrimitiveControls).OrderBy(x => (int)x.Tag).ToArray();

                this.rootStackPanel.Controls.Clear();
                foreach (var c in sortedControls)
                {
                    this.rootStackPanel.Controls.Add(c);
                    c.TabIndex = (int)c.Tag;
                }
            }

            var controls = this.rootStackPanel.Controls;
            for (int i = 0; i < controls.Count; ++i)
            {
                // スライダーのマージンをここで調整
                ((UIDialSlider)controls[i]).Margin =
                    new Padding(
                        controls[i] is UIRandomRangeDialSlider ? 12 : 0,
                        0,
                        0,
                        i == controls.Count - 1 ? 0 : 3);
            }
        }

        /// <summary>
        /// Creates a UIDialSlider instance.
        /// </summary>
        /// <param name="isRandom">ランダム幅関係か？</param>
        /// <param name="elementIndex">要素インデックス</param>
        /// <returns>Returns a UIDialSlider instance.</returns>
        private UIDialSlider CreateChildSliderControl(bool isRandom, int elementIndex)
        {
            var slider = isRandom ? new UIRandomRangeDialSlider() : new UIDialSlider();

            slider.Minimum = isRandom ? this.RandomMinimum : this.Minimum;
            slider.Maximum = isRandom ? this.RandomMaximum : this.Maximum;
            slider.DeltaLevel = this.DeltaLevel;
            slider.Tag = (elementIndex * 2) + (isRandom ? 1 : 0);
            slider.IntegerOnly = this.IntegerOnly;
            slider.ControlMode = this.ControlMode;

            if (isRandom)
            {
                slider.ValueChanged += this.OnChildSliderValueChangedRandomPrimitive;
            }
            else
            {
                slider.ValueChanged += this.OnChildSliderValueChangedPrimitive;
            }

            return slider;
        }

        /// <summary>
        /// Cleanup a previously created UIDialSlider instance.
        /// </summary>
        /// <param name="slider">The UIDialSlider instance to cleanup.</param>
        private void DisposeChildSliderControl(UIDialSlider slider)
        {
            slider.ValueChanged -= this.OnChildSliderValueChangedPrimitive;
            slider.ValueChanged -= this.OnChildSliderValueChangedRandomPrimitive;
        }

        /// <summary>
        /// Updates the view values using the primitive data values.
        /// </summary>
        private void UpdateViewFromData()
        {
            this.UpdateViewFromData(false);
            this.UpdateViewFromData(true);
        }

        /// <summary>
        /// Updates the view values using the primitive data values.
        /// </summary>
        /// <param name="isRandom">ランダム幅関係か？</param>
        private void UpdateViewFromData(bool isRandom)
        {
            this.suspendDataUpdate = true;

            try
            {
                if (isRandom == false)
                {
                    if (this.primitive != null)
                    {
                        var primitiveControls = this.PrimitiveControls.ToArray();
                        for (var i = 0; i != primitiveControls.Count(); ++i)
                        {
                            // get the data value from the primitive
                            object primitiveValue = this.primitive[i];

                            // get the slider
                            var childSlider = primitiveControls[i];

                            // set the trackbar with the view value
                            if (this.primitive is Primitivef)
                            {
                                childSlider.Value = (float)primitiveValue;
                            }
                            else if (this.primitive is Primitived)
                            {
                                childSlider.Value = (float)(double)primitiveValue;
                            }
                            else if (this.primitive is Primitivei)
                            {
                                childSlider.Value = (int)primitiveValue;
                            }
                            else if (this.primitive is Primitiveu)
                            {
                                childSlider.Value = (uint)primitiveValue;
                            }
                        }
                    }
                }

                if (isRandom)
                {
                    if (this.randomPrimitive != null)
                    {
                        var randomPrimitiveControls = this.RandomPrimitiveControls.ToArray();
                        for (var i = 0; i != this.randomPrimitive.Count; ++i)
                        {
                            // get the data value from the primitive
                            object primitiveValue = this.randomPrimitive[i];

                            // get the slider
                            var childSlider = randomPrimitiveControls[i];

                            // set the trackbar with the view value
                            if (this.primitive is Primitivef)
                            {
                                childSlider.Value = (float)primitiveValue;
                            }
                            else if (this.primitive is Primitived)
                            {
                                childSlider.Value = (float)(double)primitiveValue;
                            }
                            else if (this.primitive is Primitivei)
                            {
                                childSlider.Value = (int)primitiveValue;
                            }
                            else if (this.primitive is Primitiveu)
                            {
                                childSlider.Value = (uint)primitiveValue;
                            }
                        }
                    }
                }
            }
            finally
            {
                this.suspendDataUpdate = false;
            }
        }

        /// <summary>
        /// Updates the primitive data values using the view values.
        /// </summary>
        private void UpdateDataFromView()
        {
            if (this.suspendDataUpdate)
            {
                return;
            }

            // primitive
            if (this.primitive != null)
            {
                var primitiveControls = this.PrimitiveControls.ToArray();
                for (var i = 0; i != this.primitive.Count; ++i)
                {
                    // get the slider
                    var childSlider = primitiveControls[i];

                    // get the trackbar value
                    float viewValue = childSlider.Value;

                    // set the primitive with data value
                    if (this.primitive is Primitivef)
                    {
                        this.primitive[i] = viewValue;
                    }
                    else if (this.primitive is Primitived)
                    {
                        this.primitive[i] = (double)viewValue;
                    }
                    else if (this.primitive is Primitivei)
                    {
                        this.primitive[i] = (int)viewValue;
                    }
                    else if (this.primitive is Primitiveu)
                    {
                        this.primitive[i] = (uint)viewValue;
                    }
                }
            }

            // randomPrimitive
            if (this.randomPrimitive != null)
            {
                var randomPrimitiveControls = this.RandomPrimitiveControls.ToArray();
                for (var i = 0; i != this.randomPrimitive.Count; ++i)
                {
                    // get the slider
                    var childSlider = randomPrimitiveControls[i];

                    // get the trackbar value
                    float viewValue = childSlider.Value;

                    // set the primitive with data value
                    if (this.randomPrimitive is Primitivef)
                    {
                        this.randomPrimitive[i] = viewValue;
                    }
                    else if (this.randomPrimitive is Primitived)
                    {
                        this.randomPrimitive[i] = (double)viewValue;
                    }
                    else if (this.randomPrimitive is Primitivei)
                    {
                        this.randomPrimitive[i] = (int)viewValue;
                    }
                    else if (this.randomPrimitive is Primitiveu)
                    {
                        this.randomPrimitive[i] = (uint)viewValue;
                    }
                }
            }
        }

        /// <summary>
        /// Trigger the value changed executable event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        private void TriggerValueChangedExecutable(ValueChangedExEventArgs e)
        {
            this.runnningVCE = true;

            IExecutable executable = this.ValueChangedExecutable;

            if (executable != null && executable.CanExecute(e))
            {
                executable.Execute(e);
            }

            this.runnningVCE = false;
        }

        /// <summary>
        /// 子スライダコントロール値変更イベント（Primitive用）
        /// </summary>
        /// <param name="sender">呼び出し元</param>
        /// <param name="e">イベント引数</param>
        private void OnChildSliderValueChangedPrimitive(object sender, EventArgs e)
        {
            this.OnChildSliderValueChangedInternal(sender, e, ModifiedValueTypes.PrimaryValues);
        }

        /// <summary>
        /// 子スライダコントロール値変更イベント（RandomPrimitive用）
        /// </summary>
        /// <param name="sender">呼び出し元</param>
        /// <param name="e">イベント引数</param>
        private void OnChildSliderValueChangedRandomPrimitive(object sender, EventArgs e)
        {
            this.OnChildSliderValueChangedInternal(sender, e, ModifiedValueTypes.RandomValues);
        }

        /// <summary>
        /// 子スライダコントロール値変更イベント実装用
        /// </summary>
        /// <param name="sender">呼び出し元</param>
        /// <param name="e">イベント引数</param>
        /// <param name="modifiedValueType">The modified value type.</param>
        private void OnChildSliderValueChangedInternal(
            object sender,
            EventArgs e,
            ModifiedValueTypes modifiedValueType)
        {
            if (this.eventReentranceLocked)
            {
                return;
            }

            IPrimitive origValue;
            IPrimitive currValue;
            object executableParam;

            if (modifiedValueType == ModifiedValueTypes.PrimaryValues)
            {
                origValue = this.previousValidationValue;
                currValue = this.primitive;
                executableParam = this.ValueChangedExecutableParameter;
            }
            else if (modifiedValueType == ModifiedValueTypes.RandomValues)
            {
                origValue = this.previousValidationRandomValue;
                currValue = this.randomPrimitive;
                executableParam = this.RandomValueChangedExecutableParameter;
            }
            else
            {
                // This shouldn't happen, unless there are newly added modified value types
                // and they haven't been added to this if statement.
                return;
            }

            var ee = e as ValueChangedExEventArgs;

            if (ee != null)
            {
                // this is a ValueChangedExEventArgs event.
                if (ee.IsChanging == false)
                {
                    // previousValidationValueがUpdateDataFromViewで更新されるので先にCloneしておく
                    var savedOrigValue = origValue.Clone();

                    //// value has been validated

                    this.blockingUpdate = true;
                    this.UpdateDataFromView();
                    this.blockingUpdate = false;

                    ////if (this.valueTransform == null)
                    {
                        this.SetupValueTransform(sender, currValue);
                    }

                    if (this.suspendDataUpdate == false)
                    {
                        this.RunValueTransform();
                    }

                    // Trigger the executable for the modified value.
                    this.TriggerValueChangedExecutable(new ValueChangedExEventArgs(
                        savedOrigValue,
                        currValue.Clone(),
                        false,
                        executableParam));

                    // リアルタイム編集中の値通知はこのタイミングで行う
                    this.LogicalTreeElementExtender.NotifyPropertyChanged(
                        BindingUpdateType.PropertyChanged,
                        "Primitive");

                    this.previousValidationValue.Set(currValue);

                    this.isChanging = false;
                    this.valueTransform = null;
                }
                else
                {
                    if (this.isChanging == false)
                    {
                        origValue.Set(currValue);
                        this.SetupValueTransform(sender, currValue);
                        this.isChanging = true;
                    }

                    if (this.suspendDataUpdate == false)
                    {
                        this.RunValueTransform();
                    }

                    this.TriggerValueChangedExecutable(new ValueChangedExEventArgs(
                        this.previousThrottledValue.Clone(),
                        currValue.Clone(),
                        true,
                        executableParam));

                    // リアルタイム編集中の値通知はこのタイミングで行う
                    this.LogicalTreeElementExtender.NotifyPropertyChanged(
                        BindingUpdateType.PropertyChanged,
                        "Primitive");

                    this.previousThrottledValue.Set(currValue);
                }
            }
            else
            {
                // previousValidationValueがUpdateDataFromViewで更新されるので先にCloneしておく
                var savedOrigValue = origValue.Clone();

                //// this is NOT a ValueChangedExEventArgs event

                this.UpdateDataFromView();

                ////if (this.valueTransform == null)
                {
                    this.SetupValueTransform(sender, currValue);
                }

                if (!this.suspendDataUpdate)
                {
                    this.RunValueTransform();
                }

                this.TriggerValueChangedExecutable(new ValueChangedExEventArgs(
                    savedOrigValue,
                    currValue.Clone(),
                    false,
                    executableParam));

                this.previousValidationValue.Set(currValue);
            }

            //// TODO: update the viewer ?
        }

        /// <summary>
        /// Setup the value transform instance based on the value mode and
        /// the control that request it.
        /// </summary>
        /// <param name="valueTransformRequestingControl">
        /// The control that request the value transform.
        /// </param>
        /// <param name="primitive">
        /// The primitive.
        /// </param>
        private void SetupValueTransform(object valueTransformRequestingControl, IPrimitive primitive)
        {
            if (this.valueMode == 0)
            {
                this.valueTransform = new PassThroughValueTransform();
            }
            else if (this.valueMode == 1)
            {
                this.valueTransform = new UniformValueTransform(
                    this.rootStackPanel.Controls.FindIndex(valueTransformRequestingControl));
            }
            else if (this.valueMode == 2)
            {
                var ratio = CastToFloatArray((IPrimitive)this.previousValidationValue.Clone());
                this.valueTransform = new FixedRatioValueTransform(
                    this.rootStackPanel.Controls.FindIndex(valueTransformRequestingControl), ratio);
            }
            else
            {
                throw new ArgumentException("valueMode");
            }

            this.valueTransform.Initialize(CastToFloatArray(primitive));
        }

        /// <summary>
        /// Run the value transormation process.
        /// It consists of updating the UI sliders according to the value transform mode.
        /// </summary>
        private void RunValueTransform()
        {
            if (this.valueTransform != null)
            {
                // UpdataDataFromViewでPrimitiveへのsetが発生するので、値の変更通知をブロックしておく
                this.blockingUpdate = true;
                this.UpdateDataFromView();
                this.blockingUpdate = false;

                // primitive
                if (this.primitive != null)
                {
                    // transform values from the primitive and set new values back directly.
                    var floats = this.valueTransform.TransformValue(CastToFloatArray(this.Primitive));

                    for (int i = 0; i < floats.Length; ++i)
                    {
                        floats[i] = MathUtility.Clamp(floats[i], this.Minimum, this.Maximum);
                    }

                    if (this.Primitive is Primitivef)
                    {
                        this.Primitive.FromEnumerable(floats);
                    }
                    else if (this.Primitive is Primitived)
                    {
                        this.Primitive.FromEnumerable(CastFromFloatArray<double>(floats));
                    }
                    else if (this.Primitive is Primitivei)
                    {
                        this.Primitive.FromEnumerable(CastFromFloatArray<int>(floats));
                    }
                    else if (this.Primitive is Primitiveu)
                    {
                        this.Primitive.FromEnumerable(CastFromFloatArray<uint>(floats));
                    }
                }

                // randomPrimitive
                if (this.randomPrimitive != null)
                {
                    // transform values from the primitive and set new values back directly.
                    var floats = this.valueTransform.TransformValue(CastToFloatArray(this.RandomPrimitive));

                    for (int i = 0; i < floats.Length; ++i)
                    {
                        floats[i] = MathUtility.Clamp(floats[i], this.RandomMinimum, this.RandomMaximum);
                    }

                    if (this.RandomPrimitive is Primitivef)
                    {
                        this.RandomPrimitive.FromEnumerable(floats);
                    }
                    else if (this.RandomPrimitive is Primitived)
                    {
                        this.RandomPrimitive.FromEnumerable(CastFromFloatArray<double>(floats));
                    }
                    else if (this.RandomPrimitive is Primitivei)
                    {
                        this.RandomPrimitive.FromEnumerable(CastFromFloatArray<int>(floats));
                    }
                    else if (this.RandomPrimitive is Primitiveu)
                    {
                        this.RandomPrimitive.FromEnumerable(CastFromFloatArray<uint>(floats));
                    }
                }

                this.eventReentranceLocked = true;

                try
                {
                    this.UpdateViewFromData();
                }
                finally
                {
                    this.eventReentranceLocked = false;
                }
            }
        }

        /// <summary>
        /// The add image button.
        /// </summary>
        /// <param name="name">
        /// The name.
        /// </param>
        /// <param name="image">
        /// The image.
        /// </param>
        /// <param name="tag">
        /// The tag.
        /// </param>
        /// <param name="isChecked">
        /// The is checked.
        /// </param>
        private void AddImageButton(string name, Image image, object tag, bool isChecked = false)
        {
            var button = new UIButton()
            {
                Name = name,
                Size = new Size(16, 16),
                BackgroundImage = image,
                Tag = tag,
                Margin = new Padding(0, 0, 10, 0),
                FlatStyle = FlatStyle.Flat,
                TabStop = false,
            };

            button.FlatAppearance.BorderSize = 0;

            button.Click += this.OnOptionButtonClick;
            button.MouseDown += this.OnOptionButtonMouseDown;
            button.MouseEnter += this.OnOptionButtonEnter;
            button.MouseLeave += this.OnOptionButtonLeave;
            button.Paint += this.OnOptionButtonPaint;

            var buttonBase = new UIUserControl()
            {
                Size = new Size(26, 16),
                Margin = new Padding(0, 0, 0, 0),
            };
            buttonBase.Controls.Add(button);

            this.stackPanelOptions.Controls.Add(buttonBase);

            button.Visibility = this.showValueTransformButtons ? Visibility.Visible : Visibility.Collapsed;
        }

        /// <summary>
        /// The on option button mouse down.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// The e.
        /// </param>
        private void OnOptionButtonMouseDown(object sender, MouseEventArgs e)
        {
            if (e.Button != MouseButtons.Left)
            {
                return;
            }

            this.OnOptionButtonClick(sender, null);
        }

        /// <summary>
        /// The on option button paint.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// The e.
        /// </param>
        private void OnOptionButtonPaint(object sender, PaintEventArgs e)
        {
            var btn = (Button)sender;
            e.Graphics.DrawImage(btn.BackgroundImage, btn.ClientRectangle);
        }

        /// <summary>
        /// The on option button leave.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// The e.
        /// </param>
        private void OnOptionButtonLeave(object sender, EventArgs e)
        {
            var ctrl = sender as UIButton;

            if (ctrl != null)
            {
                this.toolTip.Hide(ctrl);
            }
        }

        /// <summary>
        /// The on option button enter.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// The e.
        /// </param>
        private void OnOptionButtonEnter(object sender, EventArgs e)
        {
            var ctrl = sender as UIButton;

            if (ctrl != null)
            {
                string text = string.Empty;
                switch ((int)ctrl.Tag)
                {
                    case 0:
                    text =
                        Properties.Resources.SliderValueTransformIndividual;
                        break;
                    case 1:
                    text =
                        Properties.Resources.SliderValueTransformUniform;
                        break;
                    case 2:
                    text =
                        Properties.Resources.SliderValueTransformFixedRatio;
                        break;
                }

                this.toolTip.Show(text, ctrl, 20, -25, 2000);
            }
        }

        /// <summary>
        /// The on option button on click.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="eventArgs">
        /// The event args.
        /// </param>
        private void OnOptionButtonClick(object sender, EventArgs eventArgs)
        {
            var ctrl = sender as UIButton;

            if (ctrl != null)
            {
                this.TransformMode = (int)ctrl.Tag;
            }
        }

        private static float[] CastToFloatArray(IPrimitive primitive)
        {
            if (primitive == null)
            {
                return null;
            }

            if (primitive is Primitivef)
            {
                return primitive.ToEnumerable<float>().ToArray();
            }

            float[] result = new float[primitive.Count];
            if (primitive is Primitived)
            {
                for (int i = 0; i < result.Length; i++)
                {
                    result[i] = (float)(double)primitive[i];
                }
            }
            else if (primitive is Primitivei)
            {
                for (int i = 0; i < result.Length; i++)
                {
                    result[i] = (int)primitive[i];
                }
            }
            else if (primitive is Primitiveu)
            {
                for (int i = 0; i < result.Length; i++)
                {
                    result[i] = (uint)primitive[i];
                }
            }

            return result;
        }

        private static T[] CastFromFloatArray<T>(float[] argArray) where T : struct
        {
            if (typeof(T) == typeof(float))
            {
                return (T[])(object)argArray;
            }

            T[] result = new T[argArray.Length];
            if (typeof(T) == typeof(double))
            {
                for (int i = 0; i < result.Length; i++)
                {
                    result[i] = (T)(object)(double)argArray[i];
                }
            }
            else if (typeof(T) == typeof(int))
            {
                for (int i = 0; i < result.Length; i++)
                {
                    result[i] = (T)(object)(int)argArray[i];
                }
            }
            else if (typeof(T) == typeof(uint))
            {
                for (int i = 0; i < result.Length; i++)
                {
                    result[i] = (T)(object)(uint)argArray[i];
                }
            }

            return result;
        }
    }
}
