﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.CodeDom.Compiler;
using System.ComponentModel;
using System.Drawing;
using System.Globalization;
using System.Linq;
using System.Reflection;
using System.Windows.Forms;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.Foundation.Coercers;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Utility;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extenders;
using EffectMaker.UIControls.Extensions;
using EffectMaker.UIControls.Specifics.GradationEditor;
using EffectMaker.UIControls.Specifics.Properties;

namespace EffectMaker.UIControls.Specifics.Sliders
{
    /// <summary>
    /// ダイアルスライダクラス.
    /// </summary>
    public partial class UIDialSlider : UIUserControl
    {
        /// <summary>
        /// 縦移動の閾値. ユーザーの意見を反映させた感覚的な値
        /// </summary>
        private const int ChangeLengthOfDelta = 36;

        /// <summary>
        /// 横移動の感度. ユーザーの意見を反映させた感覚的な値
        /// </summary>
        private const float SliderInteractRatio = 0.2f / 1.3f;

        /// <summary>
        /// 切り捨ての値.
        /// </summary>
        private const float CutOffValue = 0.0001f;

        /// <summary>
        /// The font name.
        /// </summary>
        private static string fontName = "Arial";

        /// <summary>
        /// The font point.
        /// </summary>
        private static float fontPoint = 6.0F;

        /// <summary>
        /// 値の変更単位.
        /// </summary>
        private static readonly float[] deltaArray = { 0.0001f, 0.001f, 0.01f, 0.1f, 1.0f, 10.0f, 100.0f, 1000.0f, 10000.0f };

        /// <summary>
        /// マウス.
        /// </summary>
        private readonly UtilMouse mouse = new UtilMouse();

        /// <summary>
        /// ツールチップ.
        /// </summary>
        private readonly ToolTip toolTip = new ToolTip();

        /// <summary>
        /// 値.
        /// </summary>
        private float value = 0.0f;

        /// <summary>
        /// ドラッグ前の値.
        /// </summary>
        private float originalValue = 0.0f;

        /// <summary>
        /// 値変更のベースとなる値(ドラッグ開始時と精度変更時に更新)
        /// </summary>
        private float baseValue = 0.0f;

        /// <summary>
        /// 値の変更単位の初期値（配列の位置）.
        /// </summary>
        private int deltaPos = 4;   // 精度1がデフォルト(配列の実インデックス)

        /// <summary>
        /// ドラッグの種類.
        /// </summary>
        private DraggingType draggingType = DraggingType.None;

        /// <summary>
        /// Flag locking the ValueChanged event to prevent multiple iterations.
        /// </summary>
        private bool valueChangedLock;

        /// <summary>
        /// Backing field for the Minimum property.
        /// </summary>
        private float minimum;

        /// <summary>
        /// Backing field for the Minimum property.
        /// </summary>
        private float originalMinimum;

        /// <summary>
        /// Backing field for the Minimum property.
        /// </summary>
        private bool enabledMinusValue;

        /// <summary>
        /// Backing field for the Maximum property.
        /// </summary>
        private float maximum;

        /// <summary>
        /// テキストフォーマット
        /// </summary>
        private string textFormat = string.Empty;

        /// <summary>
        /// Flag indicating whether is setting the "Value" property.
        /// </summary>
        private bool isSettingValueProperty = false;

        /// <summary>
        /// Backing field for the Coercer property.
        /// </summary>
        private CoercerBase<float> coercer;

        /// <summary>
        /// コントロールモード
        /// </summary>
        private ControlModeType controlMode;

        /// <summary>
        /// 小数点以下の丸める桁数です。
        /// </summary>
        private int roundValue = 4;

        /// <summary>
        /// 初期化を行います。
        /// </summary>
        public UIDialSlider()
        {
            this.InitializeComponent();
            this.InitializeInternal();
        }

        /// <summary>
        /// 値変更時のイベントハンドラ.
        /// </summary>
        public event EventHandler ValueChanged;

        /// <summary>
        /// ドラッグの種類です。
        /// </summary>
        public enum DraggingType
        {
            /// <summary>なし.</summary>
            None,

            /// <summary>縦.</summary>
            Vertical,

            /// <summary>横.</summary>
            Horizontal,
        }

        /// <summary>
        /// コントロールモード
        /// </summary>
        public enum ControlModeType
        {
            /// <summary>
            /// 普通
            /// </summary>
            Normal,

            /// <summary>
            /// スピンボタンあり
            /// </summary>
            Spin,

            /// <summary>
            /// スピンボタンと、上下ボタンあり
            /// </summary>
            SpinAdjust,

            /// <summary>
            /// テキストのみ
            /// </summary>
            TextOnly
        }

        /// <summary>
        /// Gets or sets coercion availability.
        /// </summary>
        public bool IsCoercerActive { get; set; }

        /// <summary>
        /// Gets or sets the value coercer.
        /// </summary>
        public CoercerBase<float> Coercer
        {
            get
            {
                return this.coercer;
            }

            set
            {
                if (this.coercer != value)
                {
                    this.coercer = value;
                    //// TODO: Coerce!
                }
            }
        }

        /// <summary>
        /// スライダの値です。
        /// </summary>
        public float Value
        {
            get
            {
                return this.value;
            }

            set
            {
                this.isSettingValueProperty = true;

                float oldValue = this.Value;
                if (this.LogicalTreeElementExtender.SetValue(ref this.value, MathUtility.RoundToMinimumDigit(value, this.RoundValue)))
                {
                    if (this.IsUpdatingDataContext == false)
                    {
                        this.OnValueChangedInternal(oldValue, false);
                    }

                    this.UpdateTextBox();
                }

                this.isSettingValueProperty = false;
            }
        }

        /// <summary>
        /// Gets or sets the minimum allowed value.
        /// </summary>
        public float Minimum
        {
            get
            {
                return this.minimum;
            }

            set
            {
                this.minimum = value;
            }
        }

        /// <summary>
        /// Gets or sets the minimum allowed value.
        /// </summary>
        public float OriginalMinimum
        {
            get
            {
                return this.originalMinimum;
            }

            set
            {
                this.originalMinimum = value;
            }
        }

        /// <summary>
        /// Gets or sets the minimum allowed value.
        /// </summary>
        public bool EnabledMinusValue
        {
            get
            {
                return this.enabledMinusValue;
            }

            set
            {
                this.enabledMinusValue = value;

                // Minimum を作る
                {
                    if (this.enabledMinusValue)
                    {
                        if (this.OriginalMinimum < 0.0f)
                        {
                            this.Minimum = this.OriginalMinimum;
                        }
                    }
                    else
                    {
                        if (this.Minimum < 0.0f)
                        {
                            this.Minimum = 0.0f;
                        }
                    }
                }
            }
        }

        /// <summary>
        /// テキストフォーマット
        /// </summary>
        public string TextFormat
        {
            get
            {
                return this.txb_Value.TextFormat;
            }

            set
            {
                this.txb_Value.TextFormat = value;
            }
        }

        /// <summary>
        /// フォーマットを外す処理
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public Func<string, string> Deformat
        {
            get
            {
                return this.txb_Value.Deformat;
            }

            set
            {
                this.txb_Value.Deformat = value;
            }
        }

        /// <summary>
        /// Gets or sets the maximum allowed value.
        /// </summary>
        public float Maximum
        {
            get
            {
                return this.maximum;
            }

            set
            {
                this.maximum = value;
                if (this.Value > this.maximum)
                {
                    this.SetValueInternal(this.maximum, false);
                }
            }
        }

        /// <summary>
        /// Gets or sets the round value.
        /// </summary>
        public int RoundValue
        {
            get
            {
                return this.roundValue;
            }

            set
            {
                this.roundValue = value;
            }
        }

        /// <summary>
        /// Gets or sets the delta level.
        /// </summary>
        public int DeltaLevel
        {
            get
            {
                return this.deltaPos - 1;
            }

            set
            {
                if (value + 1 < 0 || value + 1 >= deltaArray.Length)
                {
                    return;
                }

                this.deltaPos = value + 1;
                if (this.IntegerOnly && this.deltaPos < 4)
                {
                    this.deltaPos = 4;
                }

                if (this.deltaPos < 4 - this.roundValue)
                {
                    // 丸め値以下を指定できないようにする
                    this.deltaPos = 4 - this.roundValue;
                }

                this.btn_Drag.Text = this.Delta.ToString();
            }
        }

        /// <summary>
        /// 整数値のみを取る場合はこの値をtrueにします.
        /// </summary>
        public bool IntegerOnly { get; set; }

        /// <summary>
        /// Gets or sets the IExecutable to run when the Value property changes.
        /// </summary>
        public IExecutable ValueChangedExecutable { get; set; }

        /// <summary>
        /// Gets or sets the custom parameter of the IExecutable
        /// to run when the Value property changes.
        /// </summary>
        public object ValueChangedExecutableParameter { get; set; }

        /// <summary>
        /// コントロールモード
        /// </summary>
        public ControlModeType ControlMode
        {
            get
            {
                return this.controlMode;
            }

            set
            {
                if (this.ControlMode == value)
                {
                    return;
                }

                this.controlMode = value;

                var isNormal = this.ControlMode == ControlModeType.Normal;
                var isSpin = this.ControlMode == ControlModeType.Spin;
                var isSpinAdjust = this.ControlMode == ControlModeType.SpinAdjust;

                this.txb_Value.Dock = isNormal || isSpinAdjust ? DockStyle.None : DockStyle.Fill;
                this.pnl_Spin.Dock = isSpin ? DockStyle.Right : DockStyle.None;

                this.btn_Drag.Visible = isNormal;
                this.btn_Up.Visible = isNormal || isSpin || isSpinAdjust;
                this.btn_Down.Visible = isNormal || isSpin || isSpinAdjust;
                this.btn_Adjust.Visible = isNormal || isSpinAdjust;
            }
        }

        /// <summary>
        /// Get or set the horizontal text alignment of the text box.
        /// </summary>
        public HorizontalAlignment TextBoxTextAlign
        {
            get { return this.txb_Value.TextAlign; }
            set { this.txb_Value.TextAlign = value; }
        }

        /// <summary>
        /// 数値の増減値を返します。
        /// </summary>
        public float Delta
        {
            get
            {
                return deltaArray[this.deltaPos];
            }

            set
            {
                for (int i = 0; i < deltaArray.Length; ++i)
                {
                    if (value <= deltaArray[i])
                    {
                        this.DeltaLevel = i - 1;
                        return;
                    }
                }

                this.DeltaLevel = deltaArray.Length;
            }
        }

        /// <summary>
        /// ダイアル上でのマウスダウンか否かを保持するプロパティ
        /// </summary>
        private bool IsMouseDownOnDial { get; set; }

        /// <summary>
        /// Gets the size the control want or need to have, according to a given available space.
        /// </summary>
        /// <param name="proposedSize">The available space.</param>
        /// <returns>Returns the desired size.</returns>
        public override Size GetPreferredSize(Size proposedSize)
        {
            if (this.IsSelfOrParentCollapsed() == true)
            {
                return Size.Empty;
            }

            switch (this.controlMode)
            {
                case ControlModeType.Normal:
                    return new Size(146, 20);

                case ControlModeType.Spin:
                    return new Size(80, 20);

                case ControlModeType.SpinAdjust:
                    return new Size(80, 20);

                // 使ったことがないので最適なサイズは不明(66 x 20？)
                case ControlModeType.TextOnly:
                    return new Size(80, 20);
            }

            return new Size(0, 0);
        }

        /// <summary>
        /// Called when the Value property changes.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected virtual void OnValueChanged(ValueChangedExEventArgs e)
        {
            var handler = this.ValueChanged;

            if (handler != null)
            {
                handler(this, e);
            }

            if (this.isSettingValueProperty == false)
            {
                IExecutable executable = this.ValueChangedExecutable;

                if (executable != null && executable.CanExecute(e))
                {
                    executable.Execute(e);
                }
            }
        }

        /// <summary>
        /// 内部的な初期化を行います。
        /// </summary>
        private void InitializeInternal()
        {
            this.btn_Drag.Disposed += this.On_btn_Drag_Disposed;

            this.btn_Up.Click += this.On_btn_Up_Click;
            this.btn_Down.Click += this.On_btn_Down_Click;
            this.btn_Adjust.Click += this.On_btn_Adjust_Click;

            this.txb_Value.ValueChanged += this.On_txb_Value_ValueChanged;
            this.txb_Value.Click += (s, e) =>
            {
                if (OptionStore.RootOptions.Interface.ClickToSelectAllInTextBox)
                {
                    this.txb_Value.SelectAll();
                }
            };
            this.txb_Value.GotFocus += (s, e) =>
            {
                if (UIUserControl.DisableMenuShortcut != null)
                {
                    UIUserControl.DisableMenuShortcut();
                }
            };
            this.txb_Value.LostFocus += (s, e) =>
            {
                if (UIUserControl.EnableMenuShortcut != null)
                {
                    UIUserControl.EnableMenuShortcut();
                }
            };

            this.Minimum = float.MinValue;
            this.Maximum = float.MaxValue;

            this.UpdateTextBox();

            this.toolTip.AutoPopDelay = 1200;
            this.toolTip.InitialDelay = 1;
            this.toolTip.ReshowDelay = 1;

            // ダイヤル上のフォントサイズ指定
            this.btn_Drag.Font = new Font(UIDialSlider.fontName, UIDialSlider.fontPoint);
            this.btn_Drag.TextAlign = ContentAlignment.MiddleCenter;
            this.btn_Drag.Text = this.Delta.ToString();
            this.btn_Drag.Cursor = Cursors.NoMove2D;

            this.btn_Up.Image = Properties.Resources.TextBoxButton_Top_Normal;
            this.btn_Down.Image = Properties.Resources.TextBoxButton_Bottom_Normal;

            this.btn_Down.MouseDown += (s, e) => btn_Down.Image = Properties.Resources.TextBoxButton_Bottom_Down;
            this.btn_Down.MouseHover += (s, e) => btn_Down.Image = Properties.Resources.TextBoxButton_Bottom_Over;
            this.btn_Down.MouseLeave += (s, e) => btn_Down.Image = Properties.Resources.TextBoxButton_Bottom_Normal;
            this.btn_Down.MouseUp += (s, e) => btn_Down.Image = Properties.Resources.TextBoxButton_Bottom_Over;
            this.btn_Up.MouseDown += (s, e) => btn_Up.Image = Properties.Resources.TextBoxButton_Top_Down;
            this.btn_Up.MouseHover += (s, e) => btn_Up.Image = Properties.Resources.TextBoxButton_Top_Over;
            this.btn_Up.MouseLeave += (s, e) => btn_Up.Image = Properties.Resources.TextBoxButton_Top_Normal;
            this.btn_Up.MouseUp += (s, e) => btn_Up.Image = Properties.Resources.TextBoxButton_Top_Over;
        }

        /// <summary>
        /// 値の変更を通知します。
        /// </summary>
        private void NotifyValueValidated()
        {
            this.LogicalTreeElementExtender.NotifyPropertyChanged(
                DataBinding.BindingUpdateType.Validation,
                "Value");
        }

        /// <summary>
        /// Simplify the raising of the OnValueChanged event.
        /// </summary>
        /// <param name="oldValue">The value before the Value property got set.</param>
        /// <param name="isChanging">Flag telling whether the value is
        /// still changing or is validated.</param>
        private void OnValueChangedInternal(float oldValue, bool isChanging)
        {
            if (this.valueChangedLock)
            {
                return;
            }

            this.valueChangedLock = true;

            try
            {
                this.OnValueChanged(
                    new ValueChangedExEventArgs(
                        oldValue,
                        this.Value,
                        isChanging,
                        this.ValueChangedExecutableParameter));
            }
            finally
            {
                this.valueChangedLock = false;
            }
        }

        //-----------------------------------------------------------------
        #region Event
        //-----------------------------------------------------------------

        /// <summary>
        /// Called when the btn_Drag Button is disposed.
        /// </summary>
        /// <param name="sender">The sender object.</param>
        /// <param name="e">Event arguments.</param>
        private void On_btn_Drag_Disposed(object sender, EventArgs e)
        {
            // make sure the tooltip attach to a control is hidden
            // before the attached control get disposed.
            this.toolTip.Hide(this.btn_Drag);
        }

        /// <summary>
        /// ダイアル上でキーが離されたとき。
        /// </summary>
        /// <param name="sender">The sender object</param>
        /// <param name="e">event arguments</param>
        void On_btn_Drag_KeyUp(object sender, KeyEventArgs e)
        {
            if (e.KeyCode == Keys.A && this.mouse.IsLeftDown)
            {
                this.AdjustValue();
            }

            if (e.KeyCode == Keys.Z && this.mouse.IsLeftDown)
            {
                this.DownDelta();
                this.mouse.ResetMoveX();
                this.mouse.ResetMoveY();
                this.ShowToolTipBig();
            }

            if (e.KeyCode == Keys.X && this.mouse.IsLeftDown)
            {
                this.UpDelta();
                this.mouse.ResetMoveX();
                this.mouse.ResetMoveY();
                this.ShowToolTipBig();
            }
        }

        /// <summary>
        /// 数値が変更されたとき。
        /// </summary>
        /// <param name="sender">The sender object</param>
        /// <param name="e">event arguments</param>
        private void On_txb_Value_ValueChanged(object sender, EventArgs e)
        {
            this.txb_Value.ValueChanged -= this.On_txb_Value_ValueChanged;

            this.txb_Value.Value =
                MathUtility.Clamp(this.txb_Value.Value, this.Minimum, this.Maximum);

            this.txb_Value.ValueChanged += this.On_txb_Value_ValueChanged;

            this.SetValueInternal(this.txb_Value.Value, false);
        }

        /// <summary>
        /// ▼が押されたとき。
        /// </summary>
        /// <param name="sender">The sender object</param>
        /// <param name="e">event arguments</param>
        private void On_btn_Down_Click(object sender, EventArgs e)
        {
            this.SetValueInternal(MathUtility.RoundToMinimumDigit(this.Value - this.Delta, this.RoundValue), false);
        }

        /// <summary>
        /// ▲が押されたとき。
        /// </summary>
        /// <param name="sender">The sender object</param>
        /// <param name="e">event arguments</param>
        private void On_btn_Up_Click(object sender, EventArgs e)
        {
            this.SetValueInternal(MathUtility.RoundToMinimumDigit(this.value + this.Delta, this.RoundValue), false);
        }

        /// <summary>
        /// ダイヤル上でマウスボタンが離されたとき。
        /// </summary>
        /// <param name="sender">The sender object</param>
        /// <param name="e">event arguments</param>
        private void On_btn_Drag_MouseUp(object sender, MouseEventArgs e)
        {
            this.NotifyValueValidated();
            this.OnValueChangedInternal(this.originalValue, false);

            this.toolTip.Hide(this.btn_Drag);
            this.draggingType = DraggingType.None;
            this.btn_Drag.Cursor = Cursors.NoMove2D;

            if (e.Button == MouseButtons.Left)
            {
                this.btn_Drag.Image = Properties.Resources.Icon_Dial_Default;
                this.IsMouseDownOnDial = false;
            }
        }

        /// <summary>
        /// ダイヤル上でマウスボタンを押下したとき。
        /// </summary>
        /// <param name="sender">The sender object</param>
        /// <param name="e">event arguments</param>
        private void On_btn_Drag_MouseDown(object sender, MouseEventArgs e)
        {
            this.mouse.OnMouseDown(e);

            this.originalValue = this.baseValue = this.value;
            this.ShowToolTipBig();

            if (e.Button == MouseButtons.Left)
            {
                this.btn_Drag.Image = Properties.Resources.Icon_Dial_Push;
                this.IsMouseDownOnDial = true;
            }
        }

        /// <summary>
        /// 値を丸めます。
        /// </summary>
        private void AdjustValue()
        {
            this.SetValueInternal(UtilDialSlider.Round(this.value), false);
        }

        /// <summary>
        /// マウスホーバーしたとき。
        /// </summary>
        /// <param name="sender">The sender object</param>
        /// <param name="e">event arguments</param>
        private void On_btn_Drag_MouseHover(object sender, EventArgs e)
        {
            this.ShowToolTipSmall();
        }

        /// <summary>
        /// ダイアルマウス入イベント
        /// </summary>
        /// <param name="sender">呼び出し元</param>
        /// <param name="e">イベント引数</param>
        private void On_btn_Drag_MouseEnter(object sender, EventArgs e)
        {
            this.btn_Drag.Image = Properties.Resources.Icon_Dial_Over;
        }

        /// <summary>
        /// ダイアルマウス出イベント
        /// </summary>
        /// <param name="sender">呼び出し元</param>
        /// <param name="e">イベント引数</param>
        private void On_btn_Drag_MouseLeave(object sender, EventArgs e)
        {
            this.btn_Drag.Image = Properties.Resources.Icon_Dial_Default;
        }

        /// <summary>
        /// 横移動の処理をします。
        /// </summary>
        private void DoHorizon()
        {
            this.draggingType = DraggingType.Horizontal;
            this.btn_Drag.Cursor = Cursors.NoMoveHoriz;
        }

        /// <summary>
        /// 縦移動の処理をします。
        /// </summary>
        private void DoVertical()
        {
            this.draggingType = DraggingType.Vertical;
            this.btn_Drag.Cursor = Cursors.NoMoveVert;
        }

        /// <summary>
        /// 縦移動か、横移動かを切り替えます。
        /// </summary>
        private void ChangeHorizonVertical()
        {
            float ax = Math.Abs(this.mouse.AverageX);
            float ay = Math.Abs(this.mouse.AverageY);
            float dx = Math.Abs(this.mouse.Delta.X);
            float dy = Math.Abs(this.mouse.Delta.Y);

            if (ax > ay * 2)
            {
                this.DoHorizon();
                return;
            }

            if (ay > ax * 2)
            {
                this.DoVertical();
                return;
            }

            if (dx - dy > 25)
            {
                this.DoHorizon();
                return;
            }

            if (dy - dx > 25)
            {
                this.DoVertical();
                return;
            }
        }

        /// <summary>
        /// ダイヤル上でマウスを動かしたとき。
        /// </summary>
        /// <param name="sender">The sender object</param>
        /// <param name="e">event arguments</param>
        private void On_btn_Drag_MouseMove(object sender, MouseEventArgs e)
        {
            this.mouse.OnMouseMove(e);
            if (this.mouse.IsLeftDown && this.IsMouseDownOnDial)
            {
                this.ChangeHorizonVertical();

                if (this.draggingType == DraggingType.Horizontal)
                {
                    float v = this.baseValue + ((int)(this.mouse.TotalDelta.X * SliderInteractRatio) * this.Delta);
                    this.SetValueInternal(MathUtility.RoundToMinimumDigit(v, this.RoundValue), true);
                }

                if (this.draggingType == DraggingType.Vertical)
                {
                    if (this.mouse.TotalDelta.Y > ChangeLengthOfDelta)
                    {
                        this.DownDelta();
                        this.mouse.ResetMoveX();
                        this.mouse.ResetMoveY();
                        this.baseValue = this.Value;
                        this.ShowToolTipBig();
                    }
                    else if (this.mouse.TotalDelta.Y < -ChangeLengthOfDelta)
                    {
                        this.UpDelta();
                        this.mouse.ResetMoveX();
                        this.mouse.ResetMoveY();
                        this.baseValue = this.Value;
                        this.ShowToolTipBig();
                    }
                }
            }
        }

        /// <summary>
        /// フォーム上でキーアップしたとき。
        /// </summary>
        /// <param name="sender">The sender object</param>
        /// <param name="e">event arguments</param>
        private void Form1_KeyUp(object sender, KeyEventArgs e)
        {
            if (!this.mouse.IsLeftDown)
            {
                return;
            }

            if (e.KeyCode == Keys.Z)
            {
                this.DownDelta();
                this.mouse.ResetMoveX();
                this.mouse.ResetMoveY();
                this.ShowToolTipBig();
            }

            if (e.KeyCode == Keys.X)
            {
                this.UpDelta();
                this.mouse.ResetMoveX();
                this.mouse.ResetMoveY();
                this.ShowToolTipBig();
            }
        }

        /// <summary>
        /// 調整ボタンが押されたとき。
        /// </summary>
        /// <param name="sender">The sender object</param>
        /// <param name="e">event arguments</param>
        private void On_btn_Adjust_Click(object sender, EventArgs e)
        {
            this.AdjustValue();
        }
        #endregion

        //// -----------------------------------------------------------------

        /// <summary>
        /// Set the value member and notifies changes.
        /// Works like the Value property, but the isChanging flag can be tweaked.
        /// </summary>
        /// <param name="newValue">The new value to set.</param>
        /// <param name="isChanging">The flag telling whether the value
        /// is changing or has changed.</param>
        private void SetValueInternal(float newValue, bool isChanging)
        {
            float oldValue = this.Value;

            if (this.IsCoercerActive && this.Coercer != null)
            {
                newValue = this.Coercer.Coerce(newValue);
            }

            newValue = MathUtility.RoundToMinimumDigit(newValue, this.RoundValue);
            newValue = MathUtility.Clamp(newValue, this.Minimum, this.Maximum);

            this.LogicalTreeElementExtender.SetValue(ref this.value, newValue, BindingUpdateType.PropertyChanged, "Value");
            this.OnValueChangedInternal(oldValue, isChanging);
            this.UpdateTextBox();
        }

        /// <summary>
        /// テキストボックスを更新します。
        /// </summary>
        private void UpdateTextBox()
        {
            this.txb_Value.ValueChanged -= this.On_txb_Value_ValueChanged;
            this.txb_Value.Value = this.value;
            this.txb_Value.ValueChanged += this.On_txb_Value_ValueChanged;
        }

        /// <summary>
        /// Set the Value from a given string, if parsing do not fail.
        /// </summary>
        /// <param name="textValue">The textual value to set the Value from.</param>
        /// <returns>Returns true if parsing succeeded, false otherwise.</returns>
        private bool SetValueFromString(string textValue)
        {
            float v;

            bool result = float.TryParse(textValue, out v);
            if (result)
            {
                this.SetValueInternal(v, false);
            }

            return result;
        }

        /// <summary>
        /// Set the Value from a given string, if parsing do not fail.
        /// </summary>
        /// <param name="value">The textual value to set the Value from.</param>
        private void SetValue(float value)
        {
            this.SetValueInternal(value, false);
        }

        /// <summary>
        /// 小さなツールチップを表示します。
        /// </summary>
        private void ShowToolTipSmall()
        {
            string text = string.Format(Properties.Resources.DialSliderToolTipSmall, this.Delta);
            this.toolTip.SetToolTip(this.btn_Drag, text);
        }

        /// <summary>
        /// 大きなツールチップを表示します。
        /// </summary>
        private void ShowToolTipBig()
        {
            string text = string.Format(Properties.Resources.DialSliderToolTipBig, this.Delta);
            int duration = int.MaxValue;
            this.toolTip.Show(text, this.btn_Drag, -35, -90, duration);
        }

        /// <summary>
        /// 増減値を１段階、引下げます。
        /// </summary>
        private void DownDelta()
        {
            this.deltaPos--;

            if (this.IntegerOnly && this.deltaPos < 4)
            {
                this.deltaPos = 4;
            }
            else if (this.deltaPos < 4 - this.roundValue)
            {
                // 丸め値以下を指定できないようにする
                this.deltaPos = 4 - this.roundValue;
            }
            else if (this.deltaPos < 0)
            {
                this.deltaPos = 0;
            }

            this.btn_Drag.Text = this.Delta.ToString();
        }

        /// <summary>
        /// 増減値を１段階、引き揚げます。
        /// </summary>
        private void UpDelta()
        {
            this.deltaPos++;

            if (deltaArray.Length <= this.deltaPos)
            {
                this.deltaPos = deltaArray.Length - 1;
            }

            this.btn_Drag.Text = this.Delta.ToString();
        }
    }

    /// <summary>
    /// ダイアルスライダのユーティリティ.
    /// </summary>
    public class UtilDialSlider
    {
        /// <summary>
        /// テキスト形式の計算式を評価します。
        /// </summary>
        /// <param name="text">入力するテキスト。</param>
        /// <param name="result">結果。</param>
        /// <returns>True on success.</returns>
        public static bool ParseTextMath(string text, out float result)
        {
            result = 0.0f;
            try
            {
                var eval = new System.Data.DataTable();
                var tmpStr = eval.Compute(text, string.Empty).ToString();
                if (float.TryParse(tmpStr, out result))
                {
                    return true;
                }
            }
            catch
            {
                return false;
            }

            return false;
        }

        /// <summary>
        /// 値を丸めます。
        /// </summary>
        /// <param name="src">入力する値。</param>
        /// <returns>丸められた値。</returns>
        public static float Round(float src)
        {
            if (int.MaxValue <= src)
            {
                return src;
            }

            double scale = 10000;

            double v = Math.Round(src * scale, 0, MidpointRounding.AwayFromZero);
            double result;
            int level = 1;

            for (int i = 0; i < 15; i++)
            {
                if (int.MaxValue / 10 < level)
                {
                    return src;
                }

                level *= 10;

                if (v % level != 0 && Math.Abs(v) > level)
                {
                    result = v / level;
                    result = result >= 0.0f ? (float)Math.Floor(result) : (float)Math.Floor(result) + 1.0f;
                    result = result / scale * level;
                    if (result == 0)
                    {
                        return src;
                    }

                    return (float)result;
                }
            }

            return src;
        }
    }

    /// <summary>
    /// マウス状態を管理するクラス。
    /// </summary>
    public class UtilMouse
    {
        /// <summary>
        /// マウスの移動量の平均値への加算量減衰率
        /// </summary>
        private const float MoveAverageAttenuation = 0.8f;

        /// <summary>
        /// マウスの移動量平均値の最大値
        /// </summary>
        private const float MoveAverageLimit = 1000.0f;

        /// <summary>
        /// 位置.
        /// </summary>
        private Point pos;

        /// <summary>
        /// ドラッグの開始位置.
        /// </summary>
        private Point startPos;

        /// <summary>
        /// ドラッグ時の前回位置.
        /// </summary>
        private Point prePos;

        /// <summary>
        /// 移動の平均値Ｘ.
        /// </summary>
        private float averageX = 0.0f;

        /// <summary>
        /// 移動の平均値Ｙ.
        /// </summary>
        private float averageY = 0.0f;

        /// <summary>
        /// 移動の平均値Ｘ.
        /// </summary>
        public float AverageX
        {
            get { return this.averageX; }
            set { this.averageX = value; }
        }

        /// <summary>
        /// 移動の平均値Ｙ.
        /// </summary>
        public float AverageY
        {
            get { return this.averageY; }
            set { this.averageY = value; }
        }

        /// <summary>
        /// 位置.
        /// </summary>
        public Point Pos
        {
            get { return this.pos; }
            set { this.pos = value; }
        }

        /// <summary>
        /// 開始位置.
        /// </summary>
        public Point StartPos
        {
            get { return this.startPos; }
            set { this.startPos = value; }
        }

        /// <summary>
        /// 前回の位置.
        /// </summary>
        public Point PrePos
        {
            get { return this.prePos; }
            set { this.prePos = value; }
        }

        /// <summary>
        /// 左ボタンが押されているか？
        /// </summary>
        public bool IsLeftDown
        {
            get { return (Control.MouseButtons & MouseButtons.Left) == MouseButtons.Left; }
        }

        /// <summary>
        /// 右ボタンが押されているか？
        /// </summary>
        public bool IsRightDown
        {
            get { return (Control.MouseButtons & MouseButtons.Right) == MouseButtons.Right; }
        }

        /// <summary>
        /// 移動量.
        /// </summary>
        public Point Delta
        {
            get { return new Point(this.pos.X - this.prePos.X, this.pos.Y - this.prePos.Y); }
        }

        /// <summary>
        /// ドラッグ開始位置からの移動量.
        /// </summary>
        public Point TotalDelta
        {
            get { return new Point(this.pos.X - this.startPos.X, this.pos.Y - this.startPos.Y); }
        }

        /// <summary>
        /// 移動量をリセットする.
        /// </summary>
        public void ResetMoveY()
        {
            this.startPos.Y = this.pos.Y;
            this.averageY = 0;
        }

        /// <summary>
        /// 移動量をリセットする.
        /// </summary>
        public void ResetMoveX()
        {
            this.startPos.X = this.pos.X;
            this.averageX = 0;
        }

        /// <summary>
        /// マウスのボタンが押されたとき.
        /// </summary>
        /// <param name="e">event arguments</param>
        public void OnMouseDown(MouseEventArgs e)
        {
            if (this.IsLeftDown)
            {
                this.pos.X = e.Location.X;
                this.pos.Y = e.Location.Y;
                this.startPos.X = e.Location.X;
                this.startPos.Y = e.Location.Y;
                this.averageX = 0.0f;
                this.averageY = 0.0f;
            }
        }

        /// <summary>
        /// マウスが移動したとき.
        /// </summary>
        /// <param name="e">event arguments</param>
        public void OnMouseMove(MouseEventArgs e)
        {
            this.prePos.X = this.pos.X;
            this.prePos.Y = this.pos.Y;
            this.pos.X = e.Location.X;
            this.pos.Y = e.Location.Y;

            // 移動平均の計算
            this.averageX = (this.averageX * MoveAverageAttenuation) + Math.Abs(this.Delta.X);
            this.averageY = (this.averageY * MoveAverageAttenuation) + Math.Abs(this.Delta.Y);

            // 移動平均の最大値制限
            if (this.averageX > MoveAverageLimit)
            {
                this.averageX = MoveAverageLimit;
            }

            if (this.averageY > MoveAverageLimit)
            {
                this.averageY = MoveAverageLimit;
            }
        }
    }
}
