﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;
using System.Drawing.Drawing2D;
using EffectMaker.Foundation.Primitives;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.UIControls.Specifics.ColorPicker
{
    /// <summary>
    /// Class for HSV color element slider controller.
    /// </summary>
    public class ColorSliderControllerHSV
    {
        /// <summary>
        /// Ｈスライダ。
        /// </summary>
        private ColorElementSlider sliderH = null;

        /// <summary>
        /// Ｓスライダ。
        /// </summary>
        private ColorElementSlider sliderS = null;

        /// <summary>
        /// Ｖスライダ。
        /// </summary>
        private ColorElementSlider sliderV = null;

        /// <summary>
        /// Constructor.
        /// </summary>
        public ColorSliderControllerHSV()
        {
        }

        /// <summary>
        /// Event triggered when the color value is changed.
        /// </summary>
        public event SequentialValueChangedEventHandler ValueChanged = null;

        /// <summary>
        /// Get or set the color value.
        /// </summary>
        public ColorRgba Value
        {
            get
            {
                if (this.sliderH == null || this.sliderS == null || this.sliderV == null)
                {
                    return new ColorRgba(1.0f, 1.0f, 1.0f, 1.0f);
                }

                ColorHsva color = new ColorHsva(
                                      this.sliderH.Value,
                                      this.sliderS.Value,
                                      this.sliderV.Value,
                                      1.0f);
                return ColorUtility.ToColorRgba(color);
            }

            set
            {
                if (this.sliderH == null || this.sliderS == null || this.sliderV == null)
                {
                    return;
                }

                var color = ColorUtility.ToColorHsva(value);

                var oldH = this.sliderH.Value;
                var oldS = this.sliderS.Value;
                var oldV = this.sliderV.Value;

                // スライダ位置を出来る限り維持する
                if ((color.H == 0.0f) || (color.H == 1.0f))
                {
                    color.H = (oldH > 0.5) ? 1.0f : 0.0f;
                }

                this.sliderH.Value = color.H;
                this.sliderS.Value = color.S;
                this.sliderV.Value = color.V;

                // スライダ位置を出来る限り維持する
                if (color.S == 0.0f)
                {
                    this.sliderH.Value = oldH;
                }

                if (color.V == 0.0f)
                {
                    this.sliderH.Value = oldH;
                    this.sliderS.Value = oldS;
                }

                this.UpdateSliderHImage();
                this.UpdateSliderSImage();
                this.UpdateSliderVImage();
            }
        }

        /// <summary>
        /// Set color element sliders.
        /// </summary>
        /// <param name="sliderH">Hue color slider.</param>
        /// <param name="sliderS">Saturation color slider.</param>
        /// <param name="sliderV">Value color slider.</param>
        public void SetColorElementSliders(
                    ColorElementSlider sliderH,
                    ColorElementSlider sliderS,
                    ColorElementSlider sliderV)
        {
            this.sliderH = sliderH;
            this.sliderS = sliderS;
            this.sliderV = sliderV;

            this.sliderH.RequestUpdateBitmap += this.OnSliderRequestUpdateBitmap;
            this.sliderS.RequestUpdateBitmap += this.OnSliderRequestUpdateBitmap;
            this.sliderV.RequestUpdateBitmap += this.OnSliderRequestUpdateBitmap;

            this.sliderH.ValueChanged += this.OnSliderValueChanged;
            this.sliderS.ValueChanged += this.OnSliderValueChanged;
            this.sliderV.ValueChanged += this.OnSliderValueChanged;

            this.UpdateSliderHImage();
            this.UpdateSliderSImage();
            this.UpdateSliderVImage();
        }

        /// <summary>
        /// Clear the controlling color element sliders.
        /// </summary>
        public void ClearColorElementSliders()
        {
            if (this.sliderH != null)
            {
                this.sliderH.RequestUpdateBitmap -= this.OnSliderRequestUpdateBitmap;
                this.sliderH.ValueChanged        -= this.OnSliderValueChanged;

                this.sliderH = null;
            }

            if (this.sliderS != null)
            {
                this.sliderS.RequestUpdateBitmap -= this.OnSliderRequestUpdateBitmap;
                this.sliderS.ValueChanged        -= this.OnSliderValueChanged;

                this.sliderS = null;
            }

            if (this.sliderV != null)
            {
                this.sliderV.RequestUpdateBitmap -= this.OnSliderRequestUpdateBitmap;
                this.sliderV.ValueChanged        -= this.OnSliderValueChanged;

                this.sliderV = null;
            }
        }

        /// <summary>
        /// Update the hue slider background image.
        /// </summary>
        private void UpdateSliderHImage()
        {
            if (this.sliderH == null || this.sliderS == null || this.sliderV == null)
            {
                return;
            }

            ColorHsva color = new ColorHsva(0.0f, this.sliderS.Value, this.sliderV.Value, 1.0f);

            Bitmap img = this.sliderH.ColorBarBitmap;
            if (img == null)
            {
                return;
            }

            if (img.Width <= 0)
            {
                return;
            }

            using (var g = Graphics.FromImage(img))
            using (var brush = new LinearGradientBrush(
                                                   g.VisibleClipBounds,
                                                   Color.Red,
                                                   Color.Red,
                                                   0.0f,
                                                   false))
            {
                Color[] colorArray  = new Color[7];
                float[] posArray    = new float[7];

                for (int i = 0; i < 7; ++i)
                {
                    float fvalue = (float)i / 6.0f;

                    color.H = fvalue;

                    posArray[i]   = fvalue;
                    colorArray[i] = ColorUtility.ToWinColor(ColorUtility.ToColorRgba(color));
                }

                ColorBlend colorBlend = new ColorBlend();
                colorBlend.Colors     = colorArray;
                colorBlend.Positions  = posArray;

                brush.InterpolationColors = colorBlend;

                g.FillRectangle(brush, g.VisibleClipBounds);
            }

            this.sliderH.Invalidate();
        }

        /// <summary>
        /// Update the saturation slider background image.
        /// </summary>
        private void UpdateSliderSImage()
        {
            if (this.sliderH == null || this.sliderS == null || this.sliderV == null)
            {
                return;
            }

            this.UpdateSliderImage(
                 this.sliderS,
                 new ColorHsva(this.sliderH.Value, 0.0f, this.sliderV.Value, 1.0f),
                 new ColorHsva(this.sliderH.Value, 1.0f, this.sliderV.Value, 1.0f));
        }

        /// <summary>
        /// Update the value slider background image.
        /// </summary>
        private void UpdateSliderVImage()
        {
            if (this.sliderH == null || this.sliderS == null || this.sliderV == null)
            {
                return;
            }

            this.UpdateSliderImage(
                 this.sliderV,
                 new ColorHsva(this.sliderH.Value, this.sliderS.Value, 0.0f, 1.0f),
                 new ColorHsva(this.sliderH.Value, this.sliderS.Value, 1.0f, 1.0f));
        }

        /// <summary>
        /// Update a single color element slider background image.
        /// </summary>
        /// <param name="target">The slider to update.</param>
        /// <param name="color0">The start color of the gradation.</param>
        /// <param name="color1">The end color of the gradation.</param>
        private void UpdateSliderImage(
                     ColorElementSlider target,
                     ColorHsva color0,
                     ColorHsva color1)
        {
            if (target == null)
            {
                return;
            }

            Bitmap img = target.ColorBarBitmap;
            if (img == null)
            {
                return;
            }

            if (img.Width <= 0)
            {
                return;
            }

            Color winColor0 = ColorUtility.ToWinColor(ColorUtility.ToColorRgba(color0));
            Color winColor1 = ColorUtility.ToWinColor(ColorUtility.ToColorRgba(color1));

            using (var g = Graphics.FromImage(img))
            using (var brush = new LinearGradientBrush(
                                                   g.VisibleClipBounds,
                                                   winColor0,
                                                   winColor1,
                                                   0.0f,
                                                   false))
            {
                g.FillRectangle(brush, g.VisibleClipBounds);
            }

            target.Invalidate();
        }

        /// <summary>
        /// Handle RequestUpdateBitmap event for the sliders.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnSliderRequestUpdateBitmap(object sender, EventArgs e)
        {
            if (sender == null)
            {
                return;
            }
            else if (sender == this.sliderH)
            {
                this.UpdateSliderHImage();
            }
            else if (sender == this.sliderS)
            {
                this.UpdateSliderSImage();
            }
            else if (sender == this.sliderV)
            {
                this.UpdateSliderVImage();
            }
        }

        /// <summary>
        /// Handle ValueChanged event for the sliders.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnSliderValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            if (sender == null)
            {
                return;
            }

            if (sender == this.sliderH)
            {
                this.UpdateSliderSImage();
                this.UpdateSliderVImage();
            }
            else if (sender == this.sliderS)
            {
                this.UpdateSliderHImage();
                this.UpdateSliderVImage();
            }
            else if (sender == this.sliderV)
            {
                this.UpdateSliderHImage();
                this.UpdateSliderSImage();
            }
            else
            {
                return;
            }

            // Fire the event.
            if (this.ValueChanged != null)
            {
                this.ValueChanged(this, e);
            }
        }
    }
}
