﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;
using System.Windows.Forms;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.Foundation.Primitives;
using EffectMaker.Foundation.Utility;
using EffectMaker.UIControls.BaseControls;

namespace EffectMaker.UIControls.Specifics.ColorPicker
{
    /// <summary>
    /// カラープレビューパネル
    /// </summary>
    public class ColorPreviewPanel : UIPanel
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ColorPreviewPanel()
        {
            OptionStore.OptionChanged += this.OnGammaCorrectionChanged;
        }

        /// <summary>
        /// アルファチャンネルか？
        /// </summary>
        public bool AlphaChannel
        {
            get; set;
        }

        /// <summary>
        /// <see cref="T:System.Windows.Forms.Control"/> とその子コントロールが使用しているアンマネージ リソースを解放します。オプションで、マネージ リソースも解放します。
        /// </summary>
        /// <param name="disposing">マネージ リソースとアンマネージ リソースの両方を解放する場合は true。アンマネージ リソースだけを解放する場合は false。</param>
        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                OptionStore.OptionChanged -= this.OnGammaCorrectionChanged;
            }

            base.Dispose(disposing);
        }

        /// <summary>
        /// <see cref="E:System.Windows.Forms.Control.EnabledChanged"/> イベントを発生させます。
        /// </summary>
        /// <param name="e">イベント データを格納している <see cref="T:System.EventArgs"/>。</param>
        protected override void OnEnabledChanged(EventArgs e)
        {
            base.OnEnabledChanged(e);

            this.Invalidate();
        }

        /// <summary>
        /// コントロールの背景を描画します。
        /// </summary>
        /// <param name="e">イベント データを格納している <see cref="T:System.Windows.Forms.PaintEventArgs"/>。</param>
        protected override void OnPaintBackground(PaintEventArgs e)
        {
            var rect = this.ClientRectangle;
            rect.Width -= 1;
            rect.Height -= 1;

            if (this.Enabled)
            {
                if (this.AlphaChannel)
                {
                    e.Graphics.FillRectangle(Brushes.Black, rect);

                    using (var brush = new SolidBrush(Color.FromArgb(this.BackColor.R, 0xFF, 0xFF, 0xFF)))
                    {
                        e.Graphics.FillRectangle(brush, rect);
                    }
                }
                else
                {
                    using (var brush = new SolidBrush(ColorUtility.ApplyColorGamma(0xFF, this.BackColor)))
                    {
                        e.Graphics.FillRectangle(brush, rect);
                    }
                }

                e.Graphics.DrawRectangle(Pens.Black, rect);
            }
            else
            {
                e.Graphics.FillRectangle(Brushes.LightGray, rect);
                e.Graphics.DrawRectangle(Pens.DarkGray, rect);
            }
        }

        /// <summary>
        /// Handle GammaCorrectionChanged event from application.
        /// </summary>
        /// <param name="sender">呼び出し元</param>
        /// <param name="e">イベント引数</param>
        private void OnGammaCorrectionChanged(object sender, EventArgs e)
        {
            this.Invalidate();
        }
    }
}
