﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.ComponentModel;
using System.Windows.Forms;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.Foundation.Core;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.Behaviors;
using EffectMaker.UILogic.Commands;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.UIControls.Specifics.Behaviors
{
    /// <summary>
    /// フォルダを開くBehaviorです。
    /// </summary>
    public class OpenFolderBehavior : Behavior<UIButton>, INotifyPropertyChanged
    {
        /// <summary>
        ///  フォルダパスです。
        /// </summary>
        private string folderPath = string.Empty;

        /// <summary>
        /// ファイルタイプです。
        /// </summary>
        private LastAccessDirectoryTypes fileType = LastAccessDirectoryTypes.Any;

        /// <summary>
        /// Raised when the value of a property changes.
        /// </summary>
        public new event PropertyChangedEventHandler PropertyChanged;

        /// <summary>
        /// フォルダパスを取得または設定します。
        /// </summary>
        public string FolderPath
        {
            get
            {
                return this.folderPath;
            }

            set
            {
                this.folderPath = value;

                using (new CommandCombiner())
                {
                    var handler = this.PropertyChanged;
                    if (handler != null)
                    {
                        handler(this, new PropertyChangedEventArgs("FolderPath"));
                    }
                }
            }
        }

        /// <summary>
        /// ファイルタイプを取得または設定します。
        /// </summary>
        public LastAccessDirectoryTypes FileType
        {
            get
            {
                return this.fileType;
            }

            set
            {
                this.fileType = value;

                var handler = this.PropertyChanged;
                if (handler != null)
                {
                    handler(this, new PropertyChangedEventArgs("FileType"));
                }
            }
        }

        /// <summary>
        /// UIButtonにアタッチするときの処理を行います。
        /// </summary>
        protected override void OnAttached()
        {
            this.AssociatedObject.Click += this.OnClicked;
        }

        /// <summary>
        /// UIButtonからデタッチするときの処理を行います。
        /// </summary>
        protected override void OnDetaching()
        {
            this.AssociatedObject.Click -= this.OnClicked;
        }

        /// <summary>
        /// ボタンが押されたときの処理を行います。
        /// </summary>
        /// <param name="sender">sender</param>
        /// <param name="e">event</param>
        private void OnClicked(object sender, EventArgs e)
        {
            // フォルダ選択ダイアログを作成
            FolderBrowserDialog dialog = new FolderBrowserDialog();

            dialog.Description = string.Empty;
            dialog.ShowNewFolderButton = true;

            // 初期ディレクトリを設定
            dialog.SelectedPath = ApplicationIOManager.GetLastAccessedDir(this.fileType);

            // ダイアログを表示
            DialogResult result = dialog.ShowDialog(this.AssociatedObject);

            if (result != DialogResult.OK)
            {
                return;
            }

            // 初期ディレクトリを更新
            ApplicationIOManager.SetLastAccessedDir(
                this.FileType,
                dialog.SelectedPath);

            // 選択されたフォルダを取得
            this.FolderPath = dialog.SelectedPath;
        }
    }
}
