﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using EffectMaker.Foundation.Utility;
using EffectMaker.UIControls.Behaviors;
using System;
using System.ComponentModel;
using System.IO;
using System.Linq;
using System.Text;
using System.Windows.Forms;

namespace EffectMaker.UIControls.Specifics.Behaviors
{
    /// <summary>
    /// ファイルのドラッグ&ドロップを処理するビヘイビアです。
    /// </summary>
    public class FileDragDropBehavior : Behavior<Control>, INotifyPropertyChanged
    {
        /// <summary>
        /// ドロップされたパスです。
        /// </summary>
        private string dropResult;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public FileDragDropBehavior()
        {
            this.AllowFile              = true;
            this.AllowAllExtensions     = false;
            this.AllowExtensions        = string.Empty;
            this.AllowDirectory         = false;
            this.AllowMultiByteName     = true;
            this.ConvertToDirectoryPath = false;
            this.BasePath               = string.Empty;
            this.MaxByteCount           = 0;
            this.DropResult             = string.Empty;
        }

        /// <summary>
        /// DropResult プロパティの値が変更された場合に発生します。
        /// </summary>
        public EventHandler DropResultChanged;

        /// <summary>
        /// ファイルのドラッグ&ドロップを許可するかどうか取得または設定します。
        /// </summary>
        public bool AllowFile { get; set; }

        /// <summary>
        /// 全ての拡張子のファイルのドラッグ&ドロップを許可するかどうか取得または設定します。
        /// </summary>
        public bool AllowAllExtensions { get; set; }

        /// <summary>
        /// ドラッグ&ドロップを許可するファイルの拡張子を取得または設定します。
        /// コロン ":" で区切ることで複数指定できます。
        /// </summary>
        public string AllowExtensions { get; set; }

        /// <summary>
        /// ディレクトリのドラッグ&ドロップを許可するかどうか取得または設定します。
        /// </summary>
        public bool AllowDirectory { get; set; }

        /// <summary>
        /// マルチバイト文字を含んだ名前を許可するかどうか取得または設定します。
        /// </summary>
        public bool AllowMultiByteName { get; set; }

        /// <summary>
        /// ファイルパスをディレクトリパスに変換するかどうか取得または設定します。
        /// </summary>
        public bool ConvertToDirectoryPath { get; set; }

        /// <summary>
        /// ファイルパスを相対パスに変換するときのベースパスを取得または設定します。
        /// </summary>
        public string BasePath { get; set; }

        /// <summary>
        /// パスの最大バイト数を取得または設定します。
        /// 制限しないときは 0 を指定します。
        /// </summary>
        public int MaxByteCount { get; set; }

        /// <summary>
        /// ドロップされたパスを取得します。
        /// </summary>
        public string DropResult
        {
            get { return this.dropResult; }
            set { }  // xaml用のダミーセッター
        }

        /// <summary>
        /// コントロールにアタッチするときの処理を行います。
        /// </summary>
        protected override void OnAttached()
        {
            this.AssociatedObject.DragEnter += this.OnDragEnter;
            this.AssociatedObject.DragDrop += this.OnDragDrop;
        }

        /// <summary>
        /// コントロールからデタッチするときの処理を行います。
        /// </summary>
        protected override void OnDetaching()
        {
            this.AssociatedObject.DragEnter -= this.OnDragEnter;
            this.AssociatedObject.DragDrop -= this.OnDragDrop;
        }

        /// <summary>
        /// ドラッグ中のマウスカーソルがコントロール領域に入ったときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void OnDragEnter(object sender, DragEventArgs e)
        {
            string[] paths = this.FilterDragEvent(e);

            // パスが有効なとき、カーソルのアイコンを変える
            if (paths != null && paths.Length > 0)
            {
                e.Effect = DragDropEffects.Copy;
            }
        }

        /// <summary>
        /// ドロップしたときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void OnDragDrop(object sender, DragEventArgs e)
        {
            string[] paths = this.FilterDragEvent(e);

            // パスが有効なとき、最終チェックをしてからドロップを確定する
            if (paths != null && paths.Length > 0)
            {
                string path = paths[0];

                // 名前にマルチバイト文字が使われていないかチェック
                if (this.AllowMultiByteName == false)
                {
                    string name = Path.GetFileName(path);
                    byte[] nameUtf8 = Encoding.UTF8.GetBytes(name);

                    if (name.Length != nameUtf8.Length)
                    {
                        MessageBox.Show(Properties.Resources.FileDragDropBehaviorWarningMultiByteName, Properties.Resources.FileDragDropBehaviorWarningCaption);
                        return;
                    }
                }

                // パスの文字列が最大バイト数を超えないかチェック
                if (this.MaxByteCount > 0)
                {
                    byte[] pathUtf8 = Encoding.UTF8.GetBytes(path);

                    if (pathUtf8.Length > this.MaxByteCount)
                    {
                        MessageBox.Show(Properties.Resources.FileDragDropBehaviorWarningByteCount, Properties.Resources.FileDragDropBehaviorWarningCaption);
                        return;
                    }
                }

                // DropResultを更新
                this.dropResult = path;

                this.PropertyChanged?.Invoke(this, new PropertyChangedEventArgs("DropResult"));
                this.DropResultChanged?.Invoke(this, EventArgs.Empty);
            }
        }

        /// <summary>
        /// ドラッグイベント情報をフィルタリングします。
        /// </summary>
        /// <param name="e">イベント情報</param>
        private string[] FilterDragEvent(DragEventArgs e)
        {
            // ドラッグデータを取得
            string[] paths = (string[])e.Data.GetData(DataFormats.FileDrop);

            // ドラッグデータがファイルまたはディレクトリでないとき
            if (paths == null)
            {
                return new string[0];
            }

            string[] allowExtensions = new string[0];

            // コロン ":" で区切られた文字列を分割
            if (this.AllowExtensions != null)
            {
                allowExtensions = this.AllowExtensions.Split(':');
            }

            // パスリストをフィルタにかける
            string[] allowPaths = paths.Where(path =>
            {
                // ファイルパスをフィルタにかける
                if (File.Exists(path))
                {
                    if (this.AllowFile)
                    {
                        if (this.AllowAllExtensions)
                        {
                            return true;
                        }

                        string extension = Path.GetExtension(path);

                        if (allowExtensions.Any(ext => ext == extension))
                        {
                            return true;
                        }

                        return false;
                    }
                }
                // ディレクトリパスをフィルタにかける
                else if (Directory.Exists(path))
                {
                    if (this.AllowDirectory)
                    {
                        return true;
                    }

                    return false;
                }

                return false;
            }).ToArray();

            // ファイルパスをディレクトリパスに変換する
            if (this.ConvertToDirectoryPath)
            {
                for (int i = 0; i < allowPaths.Length; ++i)
                {
                    if (File.Exists(allowPaths[i]))
                    {
                        allowPaths[i] = Path.GetDirectoryName(allowPaths[i]);
                    }
                }
            }

            // パスを相対パスに変換する
            if (string.IsNullOrEmpty(this.BasePath) == false)
            {
                for (int i = 0; i < allowPaths.Length; ++i)
                {
                    allowPaths[i] = PathUtility.GetRelativePath(this.BasePath, allowPaths[i]);
                }
            }

            return allowPaths;
        }

        #region INotifyPropertyChanged members

        /// <summary>
        /// プロパティの値が変更されたときに発生します。
        /// </summary>
        public new event PropertyChangedEventHandler PropertyChanged;

        #endregion
    }
}
