﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.ToolDevelopmentKit
{
    using System;
    using NintendoWare.ToolDevelopmentKit.ComponentModel;

    /// <summary>
    /// クォータニオンを表します。
    /// </summary>
    public sealed class Quaternion : ObservableObject, ICloneable, IEquatable<Quaternion>, ISettable
    {
        //-----------------------------------------------------------------
        private float x;
        private float y;
        private float z;
        private float w;

        //-----------------------------------------------------------------
        // オブジェクトの生成
        //-----------------------------------------------------------------

        /// <summary>
        /// デフォルトコンストラクタです。
        /// </summary>
        public Quaternion()
        {
        }

        /// <summary>
        /// クォータニオンの成分を指定の値に設定するコンストラクタです。
        /// </summary>
        /// <param name="x">X 成分です。</param>
        /// <param name="y">Y 成分です。</param>
        /// <param name="z">Z 成分です。</param>
        /// <param name="w">W 成分です。</param>
        public Quaternion(float x, float y, float z, float w)
        {
            this.Set(x, y, z, w);
        }

        /// <summary>
        /// 回転軸と回転角度を指定の値に設定するコンストラクタです。
        /// </summary>
        /// <param name="axis">回転軸です。</param>
        /// <param name="radian">回転角度 (ラジアン単位)です。</param>
        public Quaternion(Vector3 axis, float radian)
        {
            this.SetRotate(axis, radian);
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">コピー元のクォータニオンです。</param>
        public Quaternion(Quaternion source)
        {
            this.Set(source);
        }

        //-----------------------------------------------------------------
        // プロパティの取得または設定

        //-----------------------------------------------------------------

        /// <summary>
        /// x 成分を取得または設定します。
        /// </summary>
        public float X
        {
            get
            {
                return this.x;
            }

            set
            {
                this.x = value;
                OnPropertyChanged("X");
            }
        }

        /// <summary>
        /// y 成分を取得または設定します。
        /// </summary>
        public float Y
        {
            get
            {
                return this.y;
            }

            set
            {
                this.y = value;
                OnPropertyChanged("Y");
            }
        }

        /// <summary>
        /// z 成分を取得または設定します。
        /// </summary>
        public float Z
        {
            get
            {
                return this.z;
            }

            set
            {
                this.z = value;
                OnPropertyChanged("Z");
            }
        }

        /// <summary>
        /// w 成分を取得または設定します。
        /// </summary>
        public float W
        {
            get
            {
                return this.w;
            }

            set
            {
                this.w = value;
                OnPropertyChanged("W");
            }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 長さを取得します。
        /// </summary>
        public float Length
        {
            get
            {
                return (float)Math.Sqrt(this.LengthSquared);
            }
        }

        /// <summary>
        /// 長さの 2 乗を取得します。
        /// </summary>
        public float LengthSquared
        {
            get { return (this.X * this.X) + (this.Y * this.Y) + (this.Z * this.Z) + (this.W * this.W); }
        }

        //-----------------------------------------------------------------
        // オペレータ
        //-----------------------------------------------------------------

        /// <summary>
        /// 2 つのクォータニオンを加算します。
        /// </summary>
        /// <param name="lhs">演算の左のクォータニオンです。</param>
        /// <param name="rhs">演算の右のクォータニオンです。</param>
        /// <returns>演算結果です。</returns>
        public static Quaternion operator +(Quaternion lhs, Quaternion rhs)
        {
            Ensure.Argument.NotNull(lhs);
            Ensure.Argument.NotNull(rhs);

            return new Quaternion(
                lhs.X + rhs.X, lhs.Y + rhs.Y, lhs.Z + rhs.Z, lhs.W + rhs.W);
        }

        /// <summary>
        /// 2 つのクォータニオンを減算します。
        /// </summary>
        /// <param name="lhs">演算の左のクォータニオンです。</param>
        /// <param name="rhs">演算の右のクォータニオンです。</param>
        /// <returns>演算結果です。</returns>
        public static Quaternion operator -(Quaternion lhs, Quaternion rhs)
        {
            Ensure.Argument.NotNull(lhs);
            Ensure.Argument.NotNull(rhs);

            return new Quaternion(
                lhs.X - rhs.X, lhs.Y - rhs.Y, lhs.Z - rhs.Z, lhs.W - rhs.W);
        }

        /// <summary>
        /// クォータニオンとスカラーを乗算します。
        /// </summary>
        /// <param name="quat">演算の左のクォータニオンです。</param>
        /// <param name="scalar">演算の右のスカラーです。</param>
        /// <returns>演算結果です。</returns>
        public static Quaternion operator *(Quaternion quat, float scalar)
        {
            Ensure.Argument.NotNull(quat);

            return new Quaternion(
                quat.X * scalar, quat.Y * scalar, quat.Z * scalar, quat.W * scalar);
        }

        /// <summary>
        /// クォータニオンとスカラーを除算します。
        /// </summary>
        /// <param name="quat">演算の左のクォータニオンです。</param>
        /// <param name="scalar">演算の右のスカラーです。</param>
        /// <returns>演算結果です。</returns>
        public static Quaternion operator /(Quaternion quat, float scalar)
        {
            Ensure.Argument.NotNull(quat);
            Ensure.Operation.DividerNotZero(scalar);

            return new Quaternion(
                quat.X / scalar, quat.Y / scalar, quat.Z / scalar, quat.W / scalar);
        }

        /// <summary>
        /// クォータニオンとベクトルを乗算します。
        /// </summary>
        /// <remarks>
        /// 演算結果はベクトルにクォータニオンの回転を適用したものです。
        /// </remarks>
        /// <param name="quat">演算の左のクォータニオンです。</param>
        /// <param name="vec">演算の右のベクトルです。</param>
        /// <returns>演算結果です。</returns>
        public static Vector3 operator *(Quaternion quat, Vector3 vec)
        {
            Ensure.Argument.NotNull(quat);
            Ensure.Argument.NotNull(vec);

            return quat.Transform(vec);
        }

        /// <summary>
        /// 2 つのクォータニオンを乗算します。
        /// </summary>
        /// <param name="lhs">演算の左のクォータニオンです。</param>
        /// <param name="rhs">演算の右のクォータニオンです。</param>
        /// <returns>演算結果です。</returns>
        public static Quaternion operator *(Quaternion lhs, Quaternion rhs)
        {
            Ensure.Argument.NotNull(lhs);
            Ensure.Argument.NotNull(rhs);

            // RevolutionSDK 準拠 quat.c C_QUATMultiply
            return new Quaternion(
                (lhs.W * rhs.X) + (lhs.X * rhs.W) + (lhs.Y * rhs.Z) - (lhs.Z * rhs.Y),
                (lhs.W * rhs.Y) + (lhs.Y * rhs.W) + (lhs.Z * rhs.X) - (lhs.X * rhs.Z),
                (lhs.W * rhs.Z) + (lhs.Z * rhs.W) + (lhs.X * rhs.Y) - (lhs.Y * rhs.X),
                (lhs.W * rhs.W) - (lhs.X * rhs.X) - (lhs.Y * rhs.Y) - (lhs.Z * rhs.Z));
        }

        /// <summary>
        /// 2 つのクォータニオンの線形補間を計算します。
        /// </summary>
        /// <param name="lhs">演算の左のクォータニオンです。</param>
        /// <param name="rhs">演算の右のクォータニオンです。</param>
        /// <param name="t">補間の比率です。 t = 0 であれば　p が、t = 1 であれば　quat が
        /// 計算結果となります。</param>
        /// <returns>演算結果です。</returns>
        public static Quaternion Lerp(Quaternion lhs, Quaternion rhs, float t)
        {
            Ensure.Argument.NotNull(lhs);
            Ensure.Argument.NotNull(rhs);

            return (lhs * (1 - t)) + (rhs * t);
        }

        /// <summary>
        /// 2 つのクォータニオンの球面線形補間を計算します。
        /// </summary>
        /// <param name="lhs">演算の左のクォータニオンです。</param>
        /// <param name="rhs">演算の右のクォータニオンです。</param>
        /// <param name="t">補間の比率です。 t = 0 であれば　p が、t = 1 であれば　quat が
        /// 計算結果となります。</param>
        /// <returns>演算結果です。</returns>
        public static Quaternion Slerp(Quaternion lhs, Quaternion rhs, float t)
        {
            Ensure.Argument.NotNull(lhs);
            Ensure.Argument.NotNull(rhs);

            // RevolutionSDK 準拠 quat.c C_QUATSlerp
            float angle, sin, cos, tp, tq;

            cos = lhs.Dot(rhs);
            tq = 1.0f;

            if (cos < 0.0f)
            {
                cos = -cos;
                tq = -tq;
            }

            if (FloatUtility.NearlyEqual(cos, 1.0f))
            {
                // cos(theta) is close to 1.0F -> linear
                tp = 1f - t;
                tq = tq * t;
            }
            else
            {
                angle = (float)Math.Acos(cos);
                sin = (float)Math.Sin(angle);

                float inverseSin = 1f / sin;

                tp = (float)Math.Sin((1f - t) * angle) * inverseSin;
                tq *= (float)Math.Sin(t * angle) * inverseSin;
            }

            return (lhs * tp) + (rhs * tq);
        }

        /// <summary>
        /// 2 つのクォータニオンの球面 3 次補間を計算します。
        /// </summary>
        /// <remarks>
        /// Slerp( Slerp(lhs, rhs, t), Slerp(lhsController, rhsController, t), 2*t*(1 - t) )
        /// を計算します。
        /// t = 0 であれば　lhs が、t = 1 であれば　rhs が 計算結果となります。
        /// </remarks>
        /// <param name="lhs">球面 3 次補間の始端となるクォータニオンです。</param>
        /// <param name="lhsController">球面 3 次補間の始端制御用クォータニオンです。</param>
        /// <param name="rhsController">球面 3 次補間の終端制御用クォータニオンです。</param>
        /// <param name="rhs">球面 3 次補間の終端となるクォータニオンです。</param>
        /// <param name="t">補間の比率です。 </param>
        /// <returns>演算結果です。</returns>
        public static Quaternion Squad(
            Quaternion lhs,
            Quaternion lhsController,
            Quaternion rhsController,
            Quaternion rhs,
            float t)
        {
            Ensure.Argument.NotNull(lhs);
            Ensure.Argument.NotNull(lhsController);
            Ensure.Argument.NotNull(rhsController);
            Ensure.Argument.NotNull(rhs);

            // RevolutionSDK 準拠 quat.c C_QUATSquad
            Quaternion lr, lrc;
            float t2;

            t2 = 2 * t * (1f - t);
            lr = Slerp(lhs, rhs, t);
            lrc = Slerp(lhsController, rhsController, t);
            return Slerp(lr, lrc, t2);
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 指定したクォータニオンの共役なクォータニオンを作成します。
        /// </summary>
        /// <param name="quat">指定のクォータニオンです。</param>
        /// <returns>共役なクォータニオンです。</returns>
        public static Quaternion Conjugate(Quaternion quat)
        {
            Quaternion result = new Quaternion(quat);
            result.Conjugate();
            return result;
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 指定したクォータニオンの指数部を作成します。
        /// </summary>
        /// <param name="quat">指定のクォータニオンです。</param>
        /// <returns>指数部であるクォータニオンです。</returns>
        public static Quaternion Exp(Quaternion quat)
        {
            Quaternion result = new Quaternion(quat);
            result.Exp();
            return result;
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// クォータニオンの自然対数を作成します。
        /// </summary>
        /// <param name="quat">指定のクォータニオンです。</param>
        /// <returns>自然対数であるクォータニオンです。</returns>
        public static Quaternion LogN(Quaternion quat)
        {
            Quaternion result = new Quaternion(quat);
            result.LogN();
            return result;
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 2 つのクォータニオンの内積を計算します。
        /// </summary>
        /// <param name="lhs">演算の左のクォータニオンです。</param>
        /// <param name="rhs">演算の右のクォータニオンです。</param>
        /// <returns>演算結果です。</returns>
        public static float Dot(Quaternion lhs, Quaternion rhs)
        {
            Ensure.Argument.NotNull(lhs);
            Ensure.Argument.NotNull(rhs);

            return (lhs.X * rhs.X) + (lhs.Y * rhs.Y) + (lhs.Z * rhs.Z) + (lhs.W * rhs.W);
        }

        /// <summary>
        /// オブジェクトを複製します。
        /// </summary>
        /// <returns>複製したオブジェクトです。</returns>
        public object Clone()
        {
            return new Quaternion(this);
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// オブジェクトを設定します。
        /// </summary>
        /// <param name="source">設定するオブジェクトです。</param>
        public void Set(object source)
        {
            this.Set(source as Quaternion);
        }

        /// <summary>
        /// クォータニオンを設定します。
        /// </summary>
        /// <param name="source">コピー元のクォータニオンです。</param>
        public void Set(Quaternion source)
        {
            Ensure.Argument.NotNull(source);

            this.X = source.X;
            this.Y = source.Y;
            this.Z = source.Z;
            this.W = source.W;
        }

        /// <summary>
        /// クォータニオンの成分を指定の値に設定します。
        /// </summary>
        /// <param name="x">X 成分です。</param>
        /// <param name="y">Y 成分です。</param>
        /// <param name="z">Z 成分です。</param>
        /// <param name="w">W 成分です。</param>
        public void Set(float x, float y, float z, float w)
        {
            this.X = x;
            this.Y = y;
            this.Z = z;
            this.W = w;
        }

        //-----------------------------------------------------------------
        // 数学演算
        //-----------------------------------------------------------------

        /// <summary>
        /// クォータニオンに回転軸と回転角度を設定します。
        /// </summary>
        /// <param name="axis">回転軸です。</param>
        /// <param name="radian">回転角度 (ラジアン単位)です。</param>
        public void SetRotate(Vector3 axis, float radian)
        {
            Ensure.Argument.NotNull(axis);

            if (axis.LengthSquared != 1f)
            {
                axis.Normalize();
            }

            float sin = (float)Math.Sin(radian * 0.5f);
            float cos = (float)Math.Cos(radian * 0.5f);
            this.Set(axis.X * sin, axis.Y * sin, axis.Z * sin, cos);
        }

        /// <summary>
        /// クォータニオンを指定の回転軸と回転角度で回転します。
        /// </summary>
        /// <param name="axis">回転軸です。</param>
        /// <param name="radian">回転角度 (ラジアン単位)です。</param>
        public void Rotate(Vector3 axis, float radian)
        {
            Ensure.Argument.NotNull(axis);

            Quaternion quat = new Quaternion();
            quat.SetRotate(axis, radian);
            this.Set(quat * this);
        }

        /// <summary>
        /// クォータニオンを指定のクォータニオンで回転します。
        /// </summary>
        /// <param name="quat">クォータニオンです。</param>
        public void Rotate(Quaternion quat)
        {
            Ensure.Argument.NotNull(quat);

            this.Set(quat * this);
        }

        /// <summary>
        /// 共役なクォータニオンに設定します。
        /// </summary>
        public void Conjugate()
        {
            this.Set(-this.X, -this.Y, -this.Z, this.W);
        }

        /// <summary>
        /// クォータニオンの指数部に設定します。
        /// </summary>
        public void Exp()
        {
            Ensure.Operation.True(this.W != 0.0f);

            // RevolutionSDK 準拠 quat.c C_QUATExp
            float angle = (float)Math.Sqrt((this.X * this.X) + (this.Y * this.Y) + (this.Z * this.Z));
            float scale = 1f;
            if (angle > FloatUtility.Epsilon)
            {
                scale = (float)(Math.Sin(angle) / angle);
            }

            this.Set(this * scale);
            this.W = (float)Math.Cos(angle);
        }

        /// <summary>
        /// クォータニオンの自然対数に設定します。
        /// </summary>
        public void LogN()
        {
            // RevolutionSDK 準拠 quat.c C_QUATLogN
            float scale = (float)Math.Sqrt((this.X * this.X) + (this.Y * this.Y) + (this.Z * this.Z));
            float theta = (float)Math.Atan2(scale, this.W);
            if (scale > 0f)
            {
                scale = theta / scale;
            }

            this.Set(this * scale);
            this.W = 0f;
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 正規化したクォータニオンに設定します。
        /// </summary>
        public void Normalize()
        {
            float lengthSquared = this.LengthSquared;
            Ensure.Operation.DividerNotZero(lengthSquared);

            this.Normalize(lengthSquared);
        }

        /// <summary>
        /// 正規化したクォータニオンに設定します。
        /// 長さを 1 に正規化できなかった場合は指定したクォータニオンに設定します。
        /// </summary>
        /// <param name="substitute">長さを 1 に正規化できなかった場合に
        /// 代わりに設定するベクトルです。</param>
        /// <returns>長さを正規化できた場合は true を返します。
        /// それ以外の場合は false を返します。</returns>
        public bool Normalize(Quaternion substitute)
        {
            Ensure.Argument.NotNull(substitute);

            float lengthSquared = this.LengthSquared;
            if (FloatUtility.NearlyEqual(lengthSquared, 0f))
            {
                this.Set(substitute);
                return false;
            }

            this.Normalize(lengthSquared);
            return true;
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// ベクトルに回転を適用します。
        /// </summary>
        /// <param name="vec">回転を適用するベクトルです。</param>
        /// <returns>回転を適用したベクトルです。</returns>
        public Vector3 Transform(Vector3 vec)
        {
            Ensure.Argument.NotNull(vec);

            Quaternion qv = new Quaternion(vec.X, vec.Y, vec.Z, 0);
            Quaternion q = this;
            Quaternion qc = Quaternion.Conjugate(this);
            Quaternion result = q * qv * qc;
            return new Vector3(result.X, result.Y, result.Z);
        }

        /// <summary>
        /// クォータニオンとの内積を計算します。
        /// </summary>
        /// <param name="quat">掛けるクォータニオンです。</param>
        /// <returns>演算結果です。</returns>
        public float Dot(Quaternion quat)
        {
            Ensure.Argument.NotNull(quat);

            return Dot(this, quat);
        }

        //-----------------------------------------------------------------
        // 同値比較
        //-----------------------------------------------------------------

        /// <summary>
        /// 等値であるかどうか比較します。
        /// </summary>
        /// <param name="other">比較対象です。</param>
        /// <returns>等値であれば true を返します。</returns>
        public override bool Equals(object other)
        {
            return this.Equals(other as Quaternion);
        }

        /// <summary>
        /// 等値であるかどうか比較します。
        /// </summary>
        /// <param name="other">比較対象です。</param>
        /// <returns>等値であれば true を返します。</returns>
        public bool Equals(Quaternion other)
        {
            if (other == this)
            {
                return true;
            }

            if (other == null)
            {
                return false;
            }

            return
                (this.X == other.X) &&
                (this.Y == other.Y) &&
                (this.Z == other.Z) &&
                (this.W == other.W);
        }

        /// <summary>
        /// 等値であるかどうか比較します。
        /// </summary>
        /// <param name="other">比較対象です。</param>
        /// <param name="tolerance">各成分毎の差の最大許容値です。</param>
        /// <returns>等値であれば true を返します。</returns>
        public bool Equals(Quaternion other, float tolerance)
        {
            if (other == this)
            {
                return true;
            }

            if (other == null)
            {
                return false;
            }

            return
                FloatUtility.NearlyEqual(this.X, other.X, tolerance) &&
                FloatUtility.NearlyEqual(this.Y, other.Y, tolerance) &&
                FloatUtility.NearlyEqual(this.Z, other.Z, tolerance) &&
                FloatUtility.NearlyEqual(this.W, other.W, tolerance);
        }

        /// <summary>
        /// ハッシュ値を取得します。
        /// </summary>
        /// <returns>ハッシュ値です。</returns>
        public override int GetHashCode()
        {
            return
                this.X.GetHashCode() ^
                this.Y.GetHashCode() ^
                this.Z.GetHashCode() ^
                this.W.GetHashCode();
        }

        //-----------------------------------------------------------------
        // 文字列化
        //-----------------------------------------------------------------

        /// <summary>
        /// 現在のオブジェクトを表す文字列を返します。
        /// </summary>
        /// <returns>現在のオブジェクトを表す文字列です。</returns>
        public override string ToString()
        {
            return string.Format("[{0} {1} {2} {3}]", this.X, this.Y, this.Z, this.W);
        }

        // 長さの２乗を受け取り長さを１に正規化
        private void Normalize(float lengthSquared)
        {
            if (lengthSquared == 1f)
            {
                return;
            }

            float inverseLength = 1f / (float)Math.Sqrt(lengthSquared);
            this.Set(this * inverseLength);
        }
    }
}
