﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;

namespace EffectMaker.Foundation.Extensions
{
    /// <summary>
    /// Extensions methods for SynchronizationContext.
    /// </summary>
    public static class SynchronizationContextExtensions
    {
        /// <summary>
        /// Dispatches a synchronous message.
        /// </summary>
        /// <param name="context">SynchronizationContext to dispatch to.</param>
        /// <param name="action">The message to dispatch.</param>
        public static void Send(this SynchronizationContext context, Action action)
        {
            if (context == null)
            {
                throw new ArgumentNullException("context");
            }

            if (action == null)
            {
                throw new ArgumentNullException("action");
            }

            context.Send(_ => action(), null);
        }

        /// <summary>
        /// Runs code on the SynchronizationContext and returns the produced value.
        /// </summary>
        /// <typeparam name="T">The type of value to produce.</typeparam>
        /// <param name="context">The SynchronizationContext to run code on.</param>
        /// <param name="action">The code that produces the value.</param>
        /// <param name="parameter">A custom parameter.</param>
        /// <returns>Returns a value produced on a difference SynchronizationContext.</returns>
        public static T ProduceSafely<T>(
            this SynchronizationContext context,
            Func<object, T> action,
            object parameter)
        {
            if (context == null)
            {
                throw new ArgumentNullException("context");
            }

            var result = default(T);

            context.Send(p => result = action(p), parameter);

            return result;
        }

        /// <summary>
        /// Runs code on the SynchronizationContext and returns the produced value.
        /// </summary>
        /// <typeparam name="T">The type of value to produce.</typeparam>
        /// <param name="context">The SynchronizationContext to run code on.</param>
        /// <param name="action">The code that produces the value.</param>
        /// <returns>Returns a value produced on a difference SynchronizationContext.</returns>
        public static T ProduceSafely<T>(this SynchronizationContext context, Func<T> action)
        {
            if (context == null)
            {
                throw new ArgumentNullException("context");
            }

            var result = default(T);

            context.Send(_ => result = action(), null);

            return result;
        }

        /// <summary>
        /// Dispatches a asynchronous message.
        /// </summary>
        /// <param name="context">SynchronizationContext to dispatch to.</param>
        /// <param name="action">The message to dispatch.</param>
        public static void Post(this SynchronizationContext context, Action action)
        {
            if (context == null)
            {
                throw new ArgumentNullException("context");
            }

            if (action == null)
            {
                throw new ArgumentNullException("action");
            }

            context.Post(_ => action(), null);
        }
    }
}
