﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.CodeDom.Compiler;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Xml.Serialization;
using EffectMaker.DataModel.DataModels;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;
using Microsoft.CSharp;

namespace EffectMaker.DataModel.Manager
{
    /// <summary>
    /// EmitterCustomShaderSettingManager
    /// </summary>
    public class EmitterCustomShaderSettingManager
    {
        /// <summary>
        /// シングルトンインスタンス
        /// </summary>
        public static readonly EmitterCustomShaderSettingManager Instance =
            new EmitterCustomShaderSettingManager();

        /// <summary>
        /// 最大セッティング数
        /// </summary>
        private const int MaxSettingCount = 8;

        /// <summary>
        /// セッチングの型リスト
        /// </summary>
        private readonly List<Type> settingTypes = new List<Type>();

        /// <summary>
        /// カスタムシェーダデータのアセンブリ情報です.
        /// </summary>
        private Assembly userAssembly = null;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        private EmitterCustomShaderSettingManager()
        {
        }

        /// <summary>
        /// カスタムシェーダセッティング
        /// </summary>
        public List<EmitterCustomShaderSettingInfo> Settings
        {
            get;
            private set;
        }

        /// <summary>
        /// 共通バーテックスシェーダファイルパス
        /// </summary>
        public List<string> GeneralVertexShaderPaths { get; set; }

        /// <summary>
        /// 共通フラグメントシェーダファイルパス
        /// </summary>
        public List<string> GeneralFragmentShaderPaths { get; set; }

        /// <summary>
        /// ユーザーアセンブリを取得します.
        /// </summary>
        public Assembly UserAssembly
        {
            get
            {
                return this.userAssembly;
            }
        }

        /// <summary>
        /// パーティクル定義コードを取得します.
        /// </summary>
        public string ParticleGlslSrc
        {
            get
            {
                string exeDir = PathUtility.GetDirectoryName(System.Reflection.Assembly.GetExecutingAssembly().Location);
                return this.ConcatenateFiles(new string[] { @"Converter\shader\eft_ParticleDeclaration.glsl" }, exeDir);
            }
        }

        /// <summary>
        /// パーティクルバーテックス定義コードを取得します.
        /// </summary>
        public string ParticleDecVshSrc
        {
            get
            {
                string exeDir = PathUtility.GetDirectoryName(System.Reflection.Assembly.GetExecutingAssembly().Location);
                return this.ConcatenateFiles(new string[] { @"Converter\shader\eft_ParticleDeclaration.vsh" }, exeDir);
            }
        }

        /// <summary>
        /// パーティクルフラグメント定義コードを取得します.
        /// </summary>
        public string ParticleDecFshSrc
        {
            get
            {
                string exeDir = PathUtility.GetDirectoryName(System.Reflection.Assembly.GetExecutingAssembly().Location);
                return this.ConcatenateFiles(new string[] { @"Converter\shader\eft_ParticleDeclaration.fsh" }, exeDir);
            }
        }

        /// <summary>
        /// パーティクルバーテックスシェーダコードを取得します.
        /// </summary>
        public string VertexShaderSrc
        {
            get
            {
                string exeDir = PathUtility.GetDirectoryName(System.Reflection.Assembly.GetExecutingAssembly().Location);
                return this.ConcatenateFiles(new string[] { @"Converter\shader\eft_Particle.vsh" }, exeDir);
            }
        }

        /// <summary>
        /// パーティクルフラグメントシェーダコードを取得します.
        /// </summary>
        public string FragShaderSrc
        {
            get
            {
                string exeDir = PathUtility.GetDirectoryName(System.Reflection.Assembly.GetExecutingAssembly().Location);
                return this.ConcatenateFiles(new string[] { @"Converter\shader\eft_Particle.fsh" }, exeDir);
            }
        }

        /// <summary>
        /// ストライプバーテックスシェーダコードを取得します.
        /// </summary>
        public string StripShaderSrc
        {
            get
            {
                string exeDir = PathUtility.GetDirectoryName(System.Reflection.Assembly.GetExecutingAssembly().Location);
                return this.ConcatenateFiles(new string[] { @"Converter\shader\eft_Stripe.vsh" }, exeDir);
            }
        }

        /// <summary>
        /// ストリームアウトバーテックスシェーダコードを取得します.
        /// </summary>
        public string StreamOutVshSrc
        {
            get
            {
                string exeDir = PathUtility.GetDirectoryName(System.Reflection.Assembly.GetExecutingAssembly().Location);
                return this.ConcatenateFiles(new string[] { @"Converter\shader\eft_StreamOut.vsh" }, exeDir);
            }
        }

        /// <summary>
        /// 共通バーテックスシェーダコードを取得します.
        /// </summary>
        public string GeneralVshSrc
        {
            get
            {
                string exeDir = PathUtility.GetDirectoryName(System.Reflection.Assembly.GetExecutingAssembly().Location);
                return this.ConcatenateFiles(this.GeneralVertexShaderPaths.ToArray(), exeDir);
            }
        }

        /// <summary>
        /// 共通フラグメントシェーダコードを取得します.
        /// </summary>
        public string GeneralFshSrc
        {
            get
            {
                string exeDir = PathUtility.GetDirectoryName(System.Reflection.Assembly.GetExecutingAssembly().Location);
                return this.ConcatenateFiles(this.GeneralFragmentShaderPaths.ToArray(), exeDir);
            }
        }

        /// <summary>
        /// カスタムバーテックスシェーダコードを取得します.
        /// </summary>
        public string[] CustomVshSrc
        {
            get
            {
                string exeDir = PathUtility.GetDirectoryName(System.Reflection.Assembly.GetExecutingAssembly().Location);
                string[] customVshSrc = new string[8];

                for (int i = 0; i < this.Settings.Count; ++i)
                {
                    List<string> vshPaths = new List<string>(this.Settings[i].VertexShaderPaths.Count);
                    vshPaths.AddRange(this.Settings[i].VertexShaderPaths);

                    customVshSrc[i] = this.ConcatenateFiles(vshPaths.ToArray(), exeDir);
                }

                return customVshSrc;
            }
        }

        /// <summary>
        /// カスタムフラグメントシェーダコードを取得します.
        /// </summary>
        public string[] CustomFshSrc
        {
            get
            {
                string exeDir = PathUtility.GetDirectoryName(System.Reflection.Assembly.GetExecutingAssembly().Location);
                string[] customFshSrc = new string[8];

                for (int i = 0; i < this.Settings.Count; ++i)
                {
                    List<string> fshPaths = new List<string>(this.Settings[i].FragmentShaderPaths.Count);
                    fshPaths.AddRange(this.Settings[i].FragmentShaderPaths);

                    customFshSrc[i] = this.ConcatenateFiles(fshPaths.ToArray(), exeDir);
                }

                return customFshSrc;
            }
        }

        /// <summary>
        /// インスタンスの作成.
        /// </summary>
        public static void Initialize()
        {
            Instance.InitializeInternal();
        }

        /// <summary>
        /// ユーザーデータモデルをロードする.
        /// </summary>
        public void LoadUserDataModels()
        {
            this.settingTypes.Clear();

            List<string> codeFiles = new List<string>();

            foreach (var setting in this.Settings)
            {
                if (string.IsNullOrEmpty(setting.DataModelFileName) == false)
                {
                    codeFiles.Add(setting.DataModelFileName);
                }

                if (string.IsNullOrEmpty(setting.ConversionInfoFileName) == false)
                {
                    codeFiles.Add(setting.ConversionInfoFileName);
                }
            }

            var asmInfo = UserDataModelManager.LoadDataModelFromFiles(codeFiles.ToArray());
            if (asmInfo != null)
            {
                this.userAssembly = asmInfo.Assembly;

                Type baseClassType = typeof(EmitterCustomShaderSettingData);
                var types = from t in asmInfo.Assembly.GetExportedTypes()
                            where t.IsClass && t.IsSubclassOf(baseClassType)
                            select t;
                this.settingTypes.AddRange(types);
            }

            while (this.settingTypes.Count() < MaxSettingCount)
            {
                this.settingTypes.Add(null);
            }
        }

        /// <summary>
        /// Create data model for the specified custom shader setting data model.
        /// </summary>
        /// <param name="index">The index to the custom shader setting data model.</param>
        /// <returns>The created data model.</returns>
        public EmitterCustomShaderSettingData CreateDataModelForSetting(int index)
        {
            if (index < 0 || index >= this.settingTypes.Count)
            {
                return null;
            }

            if (this.settingTypes[index] != null)
            {
                return Activator.CreateInstance(this.settingTypes[index]) as EmitterCustomShaderSettingData;
            }
            else
            {
                return null;
            }
        }

        /// <summary>
        /// エクストラデータを取得する.
        /// </summary>
        /// <returns>エクストラデータの配列.</returns>
        public IEnumerable<Type> GetExtraTypes()
        {
            return this.settingTypes.Where(x => x != null);
        }

        /// <summary>
        /// Check if the custom shader setting at the specified index exists.
        /// </summary>
        /// <param name="index">The index.</param>
        /// <returns>True if the setting exists.</returns>
        public bool HasSetting(int index)
        {
            if (index < 0 || index >= this.Settings.Count)
            {
                return false;
            }
            else if (this.Settings[index] == null)
            {
                return false;
            }
            else
            {
                return true;
            }
        }

        /// <summary>
        /// セッティングを登録する.
        /// </summary>
        /// <param name="settings">セッティング</param>
        public void SetSettings(List<EmitterCustomShaderSettingInfo> settings)
        {
            this.Settings = settings;

            this.LoadUserDataModels();
        }

        /// <summary>
        /// インスタンスの作成.
        /// </summary>
        private void InitializeInternal()
        {
            this.Settings = new List<EmitterCustomShaderSettingInfo>();
        }

        /// <summary>
        /// シェーダコードを連結します.
        /// </summary>
        /// <param name="filePaths">ファイルパス</param>
        /// <param name="rootDir">ルートディレクトリ</param>
        /// <returns>シェーダコードを返します.</returns>
        private string ConcatenateFiles(string[] filePaths, string rootDir)
        {
            StringBuilder sb = new StringBuilder();

            for (int i = 0; i < filePaths.Length; ++i)
            {
                string path;

                // 相対パスをexeから辿った絶対パスに変換
                if (Path.IsPathRooted(filePaths[i]) == false)
                {
                    path = Path.Combine(rootDir, filePaths[i]);
                }
                else
                {
                    path = filePaths[i];
                }

                try
                {
                    byte[] data = File.ReadAllBytes(path);
                    sb.Append(System.Text.Encoding.UTF8.GetString(data));
                }
                catch (Exception e)
                {
                    Logger.Log(LogLevels.Warning, "Cannot open file: " + path);
                    Logger.Log(LogLevels.Warning, "Error message : " + e.Message);
                }
            }

            return sb.ToString();
        }
    }
}
