﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Reflection;
using EffectMaker.DataModel.DataModels;

namespace EffectMaker.DataModel.Manager
{
    /// <summary>
    /// DataModelManager.
    /// </summary>
    public class DataModelManager
    {
        #region Member variables

        /// <summary>DataModelBaseを継承するクラス情報.</summary>
        private static Dictionary<Type, DataModelBase> dataModelBaseContainer =
                   new Dictionary<Type, DataModelBase>();

        /// <summary>DataModelBaseを継承するユーザーデータモデルクラス情報.</summary>
        private static Dictionary<Type, DataModelBase> userDataModelBaseContainer =
                   new Dictionary<Type, DataModelBase>();

        #endregion // Member variables

        /// <summary>
        /// Default constructor.
        /// </summary>
        static DataModelManager()
        {
            Assembly asm = System.Reflection.Assembly.GetExecutingAssembly();

            Type[] types = asm.GetTypes();

            Type dataModelBaseType = typeof(DataModelBase);

            foreach (Type type in types)
            {
                if (dataModelBaseType.IsAssignableFrom(type) && type.IsAbstract == false)
                {
                    var instance = Activator.CreateInstance(type) as DataModelBase;

                    dataModelBaseContainer.Add(type, instance);
                }
            }
        }

        /// <summary>
        /// データモデルタイプを追加する.
        /// </summary>
        /// <param name="type">型情報.</param>
        /// <returns>追加に成功した場合、trueを返す.</returns>
        public static bool AddUserDataModel(Type type)
        {
            Type dataModelBaseType = typeof(DataModelBase);
            if (dataModelBaseType.IsAssignableFrom(type) && type.IsAbstract == false)
            {
                var instance = Activator.CreateInstance(type) as DataModelBase;

                userDataModelBaseContainer.Add(type, instance);
                return true;
            }

            return false;
        }

        /// <summary>
        /// データモデルタイプを削除する.
        /// </summary>
        /// <param name="type">型情報.</param>
        /// <returns>削除に成功した場合、trueを返す.</returns>
        public static bool RemoveUserDataModel(Type type)
        {
            if (userDataModelBaseContainer.ContainsKey(type))
            {
                userDataModelBaseContainer.Remove(type);
                return true;
            }

            return false;
        }

        /// <summary>
        /// ユーザーデータモデルタイプ情報コンテナのクリア.
        /// </summary>
        public static void ClearUserDataModel()
        {
            userDataModelBaseContainer.Clear();
        }

        /// <summary>
        /// 指定したタイプのプロパティ名のデフォルト値を取得する.
        /// </summary>
        /// <param name="type">クラスタイプ.</param>
        /// <param name="propertyName">取得したい値のプロパティ名.</param>
        /// <param name="result">結果の値.</param>
        /// <returns>成功した場合trueを返す.</returns>
        public static bool GetDataModelDefaultValue(
            Type type, string propertyName, out object result)
        {
            DataModelBase instance;
            if (dataModelBaseContainer.TryGetValue(type, out instance) == false)
            {
                // ユーザーデータモデルコンテナをチェック.
                if (userDataModelBaseContainer.TryGetValue(type, out instance) == false)
                {
                    result = null;
                    return false;
                }
            }

            PropertyInfo pi = instance.GetType().GetProperty(propertyName);
            if (pi == null)
            {
                result = null;
                return false;
            }

            result = pi.GetValue(instance, null);
            return true;
        }

        /// <summary>
        /// 指定した名前のデータモデルを取得する.
        /// </summary>
        /// <param name="dataModelType">データモデルのクラス名.</param>
        /// <returns>成功した場合trueを返す.</returns>
        public static DataModelBase GetDefaultDataModel(Type dataModelType)
        {
            DataModelBase instance = null;
            if (dataModelBaseContainer.TryGetValue(dataModelType, out instance) == false)
            {
                // ユーザーデータモデルコンテナをチェック.
                if (userDataModelBaseContainer.TryGetValue(dataModelType, out instance) == false)
                {
                    return null;
                }
            }

            return instance;
        }
    }
}
