﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using EffectMaker.DataModel.DataModels;

using EffectMaker.Foundation.Interfaces;

namespace EffectMaker.DataModel.Extensions
{
    /// <summary>
    /// Extensions for data models.
    /// </summary>
    public static class DataModelExtension
    {
        /// <summary>
        /// Helper method for a property getter which returns a data model.
        /// When the backing field is null, this method create a new instance
        /// of the data model to the backing field.
        /// </summary>
        /// <typeparam name="T">The type of the returning data model.</typeparam>
        /// <param name="self">The extending data model.</param>
        /// <param name="field">The backing field of the property.</param>
        /// <param name="allowNull">False to create instance to the backing field when it's null.</param>
        /// <returns>The data model stored in the backing field.</returns>
        public static T GetDataModelFromField<T>(
            this DataModelBase self,
            ref T field,
            bool allowNull) where T : DataModelBase
        {
            if (field == null && allowNull == false)
            {
                field = (T)Activator.CreateInstance(typeof(T));
                field.Parent = self;
            }

            return field;
        }

        /// <summary>
        /// Helper method for setting the data model to the given backing field.
        /// </summary>
        /// <typeparam name="T">The type of the data model.</typeparam>
        /// <param name="self">The extending data model.</param>
        /// <param name="field">The backing field.</param>
        /// <param name="value">The value to set.</param>
        public static void SetDataModelToField<T>(
            this DataModelBase self,
            ref T field,
            T value) where T : DataModelBase
        {
            if (value == null)
            {
                field = null;
            }
            else
            {
                if (field == null ||
                    field.GetType() != value.GetType())
                {
                    field = (T)value.Clone();
                    field.Parent = self;
                }
                else
                {
                    field.Set(value);
                }
            }
        }

        /// <summary>
        /// Helper method for a property getter which returns an object.
        /// When the backing field is null, this method create a new instance
        /// of the object to the backing field.
        /// </summary>
        /// <typeparam name="T">The type of the returning object.</typeparam>
        /// <param name="self">The extending data model.</param>
        /// <param name="field">The backing field of the property.</param>
        /// <param name="allowNull">False to create instance to the backing field when it's null.</param>
        /// <returns>The object stored in the backing field.</returns>
        public static T GetObjectFromField<T>(
            this DataModelBase self,
            ref T field,
            bool allowNull)
        {
            if (field == null && allowNull == false)
            {
                field = (T)Activator.CreateInstance(typeof(T));
            }

            return field;
        }

        /// <summary>
        /// Helper method for setting the ISettable to the given backing field.
        /// </summary>
        /// <typeparam name="T">The type of the data model.</typeparam>
        /// <param name="self">The extending data model.</param>
        /// <param name="field">The backing field.</param>
        /// <param name="value">The value to set.</param>
        public static void SetISettableToField<T>(
            this DataModelBase self,
            ref T field,
            T value) where T : ISettable
        {
            if (value == null)
            {
                field = default(T);
            }
            else
            {
                if (field == null)
                {
                    field = (T)Activator.CreateInstance(typeof(T));
                }

                field.Set(value);
            }
        }

        /// <summary>
        /// Helper method for a property getter which returns a data model list.
        /// When the backing field is null, return an empty list.
        /// </summary>
        /// <typeparam name="T">The type of the returning data model.</typeparam>
        /// <param name="self">The extending data model.</param>
        /// <param name="field">The backing field of the property.</param>
        /// <returns>The data model list stored in the backing field.</returns>
        public static List<T> GetDataModelListFromField<T>(
            this DataModelBase self,
            ref List<T> field) where T : DataModelBase
        {
            if (field == null)
            {
                field = new List<T>();
            }

            return field;
        }

        /// <summary>
        /// Helper method for setting the data model list to the given backing field.
        /// </summary>
        /// <typeparam name="T">The type of the data model.</typeparam>
        /// <param name="self">The extending data model.</param>
        /// <param name="field">The backing field.</param>
        /// <param name="value">The value to set.</param>
        public static void SetDataModelListToField<T>(
            this DataModelBase self,
            ref List<T> field,
            List<T> value) where T : DataModelBase
        {
            if (field == null)
            {
                field = new List<T>();
            }

            field.Clear();
            field.AddRange(value.Select(it => it.Clone() as T));
            field.ForEach(it => it.Parent = self);
        }

        /// <summary>
        /// Helper method for a property getter which enumerates data models.
        /// When the backing field is null, return an empty enumerable.
        /// </summary>
        /// <typeparam name="T">The type of the returning data model.</typeparam>
        /// <param name="self">The extending data model.</param>
        /// <param name="field">The backing field of the property.</param>
        /// <returns>The data model enumerable stored in the backing field.</returns>
        public static IEnumerable<T> GetDataModelEnumerableFromField<T>(
            this DataModelBase self,
            ref IEnumerable<T> field) where T : DataModelBase
        {
            if (field == null)
            {
                return Enumerable.Empty<T>();
            }
            else
            {
                return field;
            }
        }

        /// <summary>
        /// Helper method for a property getter which returns a list.
        /// When the backing field is null, return an empty list.
        /// </summary>
        /// <typeparam name="T">The type of the the list elements.</typeparam>
        /// <param name="self">The extending data model.</param>
        /// <param name="field">The backing field of the property.</param>
        /// <returns>The list stored in the backing field.</returns>
        public static List<T> GetValueListFromField<T>(
            this DataModelBase self,
            ref List<T> field)
        {
            if (field == null)
            {
                field = new List<T>();
            }

            return field;
        }

        /// <summary>
        /// Helper method for setting the list to the given backing field.
        /// </summary>
        /// <typeparam name="T">The type of the list elements.</typeparam>
        /// <param name="self">The extending data model.</param>
        /// <param name="field">The backing field.</param>
        /// <param name="value">The value to set.</param>
        public static void SetValueListToField<T>(
            this DataModelBase self,
            ref List<T> field,
            List<T> value)
        {
            if (field == null)
            {
                field = new List<T>();
            }

            field.Clear();
            field.AddRange(value);
        }

        /// <summary>
        /// Helper method for setting the list of ICloneable to the given backing field.
        /// </summary>
        /// <typeparam name="T">The type of the list elements.</typeparam>
        /// <param name="self">The extending data model.</param>
        /// <param name="field">The backing field.</param>
        /// <param name="value">The value to set.</param>
        public static void SetICloneableListToField<T>(
            this DataModelBase self,
            ref List<T> field,
            List<T> value) where T : ICloneable
        {
            if (field == null)
            {
                field = new List<T>();
            }

            field.Clear();
            field.AddRange(value.Select(it => (T)it.Clone()));
        }
    }
}
