﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Text;
using NintendoWare.ToolDevelopmentKit;

namespace NWCore.DataModel
{
    #region Enumerator for the emitter asset types

    /// <summary>
    /// Enumerator for the emitter asset types.
    /// </summary>
    public enum EmitterAssetTypes
    {
        EmitterTexture0        = 0x00000101,
        EmitterTexture1        = 0x00000102,
        EmitterTexture2        = 0x00000103,
        ChildTexture0          = 0x00000104,

        EmitterVolumePrimitive = 0x00000204,
        EmitterPrimitive       = 0x00000205,
        ChildPrimitive         = 0x00000206,

        AssetTypeMask          = 0x0000FF00,

        TextureAsset           = 0x00000100,
        PrimitiveAsset         = 0x00000200,

        ForceDWORD             = 0x7FFFFFFF
    }

    #endregion

    #region Class for storing emitter texture information

    /// <summary>
    /// Class for storing emitter texture information.
    /// </summary>
    public class EmitterAssetInfo
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="emitter">The owner emitter document.</param>
        /// <param name="texType">The type of the texture.</param>
        /// <param name="filePath">The asset file path or name.</param>
        public EmitterAssetInfo( NWCore.DataModel.IEmitterDocument emitter,
                                 EmitterAssetTypes texType,
                                 string filePath )
        {
            if ( emitter==null )
                this.EmitterSetDocument = null;
            else
                this.EmitterSetDocument = emitter.EmitterSetDocument;

            this.EmitterDocument = emitter;
            this.AssetType       = texType;
            this.FilePath        = filePath;
        }


        /// <summary>Get the owner emitter set document of the asset.</summary>
        public NWCore.DataModel.IEmitterSetDocument EmitterSetDocument { get; private set; }

        /// <summary>Get the owner emitter document of the asset.</summary>
        public NWCore.DataModel.IEmitterDocument EmitterDocument { get; private set; }

        /// <summary>Get the type of the asset.</summary>
        public EmitterAssetTypes AssetType { get; private set; }

        /// <summary>Get or set the file path or name of the asset file.</summary>
        public string FilePath { get; set; }
    }

    #endregion

    /// <summary>
    /// エミッタドキュメント
    /// </summary>
    public interface IEmitterDocument : IDocument, IShowable
    {
        /// <summary>
        /// accessor for emitter data model encapsulated in this document
        /// </summary>
        EmitterData EmitterData { get; }

        /// <summary>
        /// Emitter type
        /// </summary>
        EmitterType EmitterType { get; }

        /// <summary>
        /// Texture path list
        /// </summary>
        IEnumerable<string> TexturePaths { get; }

        /// <summary>
        /// 関連するテクスチャを取得します
        /// </summary>
        IEnumerable<EmitterAssetInfo> Textures { get; }

        /// <summary>
        /// Primitive path list
        /// </summary>
        IEnumerable<string> PrimitivePaths { get; }

        /// <summary>
        /// 関連するプリミティブを取得します
        /// </summary>
        IEnumerable<EmitterAssetInfo> Primitives { get; }

        /// <summary>
        /// Parent
        /// </summary>
        IEmitterSetDocument EmitterSetDocument { get; }

        /// <summary>
        /// Verify the texture paths of this emitter and/or its child are set.
        /// </summary>
        /// <param name="docPath">Use this path as document path to verify the textures.</param>
        /// <returns>True if all the texture paths are set.</returns>
        bool VerifyTexturePathsAreSet(string docPath);

        /// <summary>
        /// Verify the texture paths of this emitter and/or its child.
        /// </summary>
        /// <param name="docPath">Use this path as document path to verify the textures.</param>
        /// <returns>True if all the texture paths are valid.</returns>
        bool VerifyTexturePathsAreReachable(string docPath);

        /// <summary>
        /// Verify the primitve paths of this emitter and/or its child.
        /// </summary>
        /// <param name="docPath">Use this path as document path to verify the textures.</param>
        /// <param name="bCheckPath">True to check if the path is reachable.</param>
        /// <returns>True if all the primitive paths are valid.</returns>
        bool VerifyPrimitivePathsAreReachable( string docPath );
    }
}
