﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using NintendoWare.ToolDevelopmentKit;

namespace NWCore.DataModel.Major_1.Minor_1.Build_0.Revision_0
{
    /// <summary>
    /// チャイルド
    /// </summary>
    public sealed class ChildData : BaseDataModel<ChildData>
    {
        #region Construtors

        /// <summary>
        /// デフォルトコンストラクタ
        /// </summary>
        public ChildData()
        {
            this.Color = new RgbaColor();        // 色
            this.TextureRes = new TextureRes();  // チャイルド用テクスチャ
            this.Scale = new Vector2(1, 1); ;    // スケール値
            this.RandVel = new Vector3();        // ランダム速度
            this.UserShaderParams = new float[32];
        }

        /// <summary>
        /// コピーコンストラクタ
        /// </summary>
        /// <param name="source">コピー元のベクトル</param>
        public ChildData(ChildData source)
            : this()
        {
            this.Set(source);
        }

        /// <summary>
        /// 初期化
        /// </summary>
        public override void Initialize()
        {
            this.AirRegist = 1.0f;
            this.Alpha = 1.0f;
            this.EmitRate = 1;
            this.EmitStep = 10;
            this.EmitTiming = 60;
            this.ScaleInheritRate = 1.0f;
            this.VelInheritRate = 0.0f;
            this.VelInheritRate = 1.0f;
            this.UseSoftParticle = false;
            this.SoftParticleFadeDistance = 0.0f;
            this.SoftParticleVolumeParam = 0.0f;
            this.ShaderType = ShaderType.Particle;
            this.ShaderParam0 = 0.0f;
            this.ShaderParam1 = 0.0f;

            this.UserShaderFlags = 0;
            for (int i = 0; i < 32; ++i)
            {
                this.UserShaderParams[i] = 0.0f;
            }

            this.RandomScale = 0.0f;

            float worldScale = s_WorldScaleRange / 100.0f;
            {
                this.FigureVelocity = 0.1f * worldScale;
                this.Scale.Set(1*worldScale, 1*worldScale);
            }
        }

        #endregion

        #region Utility Functions

        /// <summary>
        /// ベクトルを設定します。
        /// </summary>
        /// <param name="source">設定するベクトルです。</param>
        public override void Set(ChildData source)
        {
            Ensure.Argument.NotNull(source);

            this.Color.Set(source.Color);
            this.TextureRes.Set(source.TextureRes);
            this.Scale.Set(source.Scale);
            this.RandVel.Set(source.RandVel);

            this.AirRegist = source.AirRegist;
            this.Alpha = source.Alpha;
            this.AlphaInit = source.AlphaInit;
            this.AlphaTarget = source.AlphaTarget;
            this.BillboardType = source.BillboardType;
            this.BillboardPrimitiveShape = source.BillboardPrimitiveShape;
            this.CombinerType = source.CombinerType;
            this.EmitRate = source.EmitRate;
            this.EmitStep = source.EmitStep;
            this.EmitTiming = source.EmitTiming;
            this.FigureVelocity = source.FigureVelocity;
            this.InitPosRand = source.InitPosRand;
            this.RandomScale = source.RandomScale;
            this.ScaleInheritRate = source.ScaleInheritRate;
            this.VelInheritRate = source.VelInheritRate;
            this.ZBufATestType = source.ZBufATestType;
            this.UseSoftParticle = source.UseSoftParticle;
            this.SoftParticleFadeDistance = source.SoftParticleFadeDistance;
            this.SoftParticleVolumeParam = source.SoftParticleVolumeParam;
            this.ShaderType = source.ShaderType;
            this.ShaderSetting = source.ShaderSetting;
            this.ShaderParam0 = source.ShaderParam0;
            this.ShaderParam1 = source.ShaderParam1;
            this.UserShaderFlags = source.UserShaderFlags;
            for (int i = 0; i < 32; ++i)
            {
                this.UserShaderParams[i] = source.UserShaderParams[i];
            }
        }

        /// <summary>
        /// オブジェクトを複製します。
        /// </summary>
        /// <returns>複製したオブジェクトです。</returns>
        public override ChildData Clone()
        {
            return new ChildData(this);
        }

        #endregion

        #region Properties

        /// <summary>
        /// 放出量
        /// </summary>
        public int EmitRate { get; set; }

        /// <summary>
        /// タイミング
        /// </summary>
        public int EmitTiming { get; set; }

        /// <summary>
        /// 放出間隔
        /// </summary>
        public int EmitStep { get; set; }

        /// <summary>
        /// 速度継承率
        /// </summary>
        public float VelInheritRate { get; set; }

        /// <summary>
        /// 形状速度
        /// </summary>
        public float FigureVelocity { get; set; }

        /// <summary>
        /// ランダム速度
        /// </summary>
        public Vector3 RandVel { get; set; }

        /// <summary>
        /// 初期位置のランダム量
        /// </summary>
        public float InitPosRand { get; set; }

        /// <summary>
        /// ビルボードタイプ
        /// </summary>
        public BillboardType BillboardType { get; set; }

        /// <summary>
        /// プリミティブ形状
        /// </summary>
        public int BillboardPrimitiveShape { get; set; }

        /// <summary>
        /// Ｚバッファー、αテストタイプ
        /// </summary>
        public ZBufATestType ZBufATestType { get; set; }

        /// <summary>
        /// チャイルド用テクスチャ
        /// </summary>
        public TextureRes TextureRes { get; set; }

        /// <summary>
        /// 色
        /// </summary>
        public RgbaColor Color { get; set; }

        /// <summary>
        /// アルファ
        /// </summary>
        public float Alpha { get; set; }

        /// <summary>
        /// アルファアニメのターゲット
        /// </summary>
        public float AlphaTarget { get; set; }

        /// <summary>
        /// アルファアニメの初期値
        /// </summary>
        public float AlphaInit { get; set; }

        /// <summary>
        /// スケール継承率
        /// </summary>
        public float ScaleInheritRate { get; set; }

        /// <summary>
        /// スケール値
        /// </summary>
        public Vector2 Scale { get; set; }

        /// <summary>
        /// ランダム値
        /// </summary>
        public float RandomScale { get; set; }

        /// <summary>
        /// コンバイナタイプ
        /// </summary>
        public CombinerType CombinerType { get; set; }

        /// <summary>
        /// 空気抵抗
        /// </summary>
        public float AirRegist { get; set; }

        /// <summary>
        /// シェーダータイプ：ソフトパーティクルを利用する
        /// </summary>
        public bool UseSoftParticle { get; set; }

        /// <summary>
        /// Fade distance for soft particle.
        /// </summary>
        public float SoftParticleFadeDistance { get; set; }

        /// <summary>
        /// Volume parameter for the soft particle
        /// </summary>
        public float SoftParticleVolumeParam { get; set; }

        /// <summary>
        /// シェーダータイプ
        /// </summary>
        public ShaderType ShaderType { get; set; }

        /// <summary>
        /// シェーダセッティング
        /// </summary>
        public uint ShaderSetting { get; set; }

        /// <summary>
        /// シェーダパラメータ
        /// </summary>
        public float ShaderParam0 { get; set; }

        /// <summary>
        /// シェーダパラメータ
        /// </summary>
        public float ShaderParam1 { get; set; }

        /// <summary>
        /// ユーザーシェーダフラグ
        /// </summary>
        public uint UserShaderFlags { get; set; }

        #region Accessing each bit

        public bool UserShaderFlag0
        {
            get { return ((UserShaderFlags & 1) != 0); }
            set { UserShaderFlags = ((value ? (uint)1 : 0) | (UserShaderFlags & ~(uint)1)); }
        }

        public bool UserShaderFlag1
        {
            get { return ((UserShaderFlags & 2) != 0); }
            set { UserShaderFlags = ((value ? (uint)2 : 0) | (UserShaderFlags & ~(uint)2)); }
        }

        public bool UserShaderFlag2
        {
            get { return ((UserShaderFlags & 4) != 0); }
            set { UserShaderFlags = ((value ? (uint)4 : 0) | (UserShaderFlags & ~(uint)4)); }
        }

        public bool UserShaderFlag3
        {
            get { return ((UserShaderFlags & 8) != 0); }
            set { UserShaderFlags = ((value ? (uint)8 : 0) | (UserShaderFlags & ~(uint)8)); }
        }

        public bool UserShaderFlag4
        {
            get { return ((UserShaderFlags & 16) != 0); }
            set { UserShaderFlags = ((value ? (uint)16 : 0) | (UserShaderFlags & ~(uint)16)); }
        }

        public bool UserShaderFlag5
        {
            get { return ((UserShaderFlags & 32) != 0); }
            set { UserShaderFlags = ((value ? (uint)32 : 0) | (UserShaderFlags & ~(uint)32)); }
        }

        public bool UserShaderFlag6
        {
            get { return ((UserShaderFlags & 64) != 0); }
            set { UserShaderFlags = ((value ? (uint)64 : 0) | (UserShaderFlags & ~(uint)64)); }
        }

        public bool UserShaderFlag7
        {
            get { return ((UserShaderFlags & 128) != 0); }
            set { UserShaderFlags = ((value ? (uint)128 : 0) | (UserShaderFlags & ~(uint)128)); }
        }

        #endregion

        /// <summary>
        /// ユーザーシェーダパラメータ
        /// </summary>
        public float[] UserShaderParams { get; set; }

        #endregion
    }
}
