﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Diagnostics;
using NintendoWare.ToolDevelopmentKit;
using NWCore.DataModel;

namespace NWCore.DataModel.Major_0.Minor_0.Build_0.Revision_0
{
    /// <summary>
    /// 初期設定データ
    /// </summary>
    public sealed class InitSettingsData : BaseDataModel<InitSettingsData>
    {
        #region Constructors

        private const int USER_DATA_BIT_LENGTH = 16;
        private const int USER_NUMERICAL_VALUE_COUNT = 2;
        /// <summary>
        /// デフォルトコンストラクタ
        /// </summary>
        public InitSettingsData()
        {
            Initialize();
        }

        /// <summary>
        /// コピーコンストラクタ
        /// </summary>
        /// <param name="source">コピー元のベクトル</param>
        public InitSettingsData(InitSettingsData source)
            : this()
        {
            this.Set(source);
        }

        #endregion

        #region Utility Functions

        /// <summary>
        /// 初期化
        /// </summary>
        public override void Initialize()
        {
            this.InitConfigFilename = "";
            this.EditEnvFilename = "";
            this.FrameRate = InitSettingFrameRate.FPS60;
            this.Resolution = InitSettingResolution.RES_960_540;
            this.ClearColor = new RgbaColor(0.2f, 0.2f, 0.2f, 0.0f);
            this.WorldScaleRange = 100.0f;
            this.WorldScaleTime = 1000;
            this.GridScale = 1.0f;
            this.BitLabel = new string[USER_DATA_BIT_LENGTH];
            for (int i=0; i<USER_DATA_BIT_LENGTH; ++i)
            {
                this.BitLabel[i] = "[" + i + "]";
            }
            this.NumberLabel = new string[USER_NUMERICAL_VALUE_COUNT];
            for (int i = 0; i < USER_NUMERICAL_VALUE_COUNT; ++i)
            {
                this.NumberLabel[i] =
                    res.Strings.INIT_CONFIG_DATA_PROPERTY_USER_DATA_NUMERICAL_VALUE_LABEL + i;
            }
        }

        /// <summary>
        /// オブジェクトを設定します。
        /// </summary>
        public override void Set(InitSettingsData source)
        {
            this.InitConfigFilename = source.InitConfigFilename;
            this.EditEnvFilename = source.EditEnvFilename;
            this.FrameRate = source.FrameRate;
            this.Resolution = source.Resolution;
            this.ClearColor.Set(source.ClearColor);
            this.WorldScaleRange = source.WorldScaleRange;
            this.WorldScaleTime = source.WorldScaleTime;
            this.GridScale = source.GridScale;
            for (int i = 0 ; i < USER_DATA_BIT_LENGTH; ++i)
            {
                this.BitLabel[i] = source.BitLabel[i];
            }
            for (int i=0; i<USER_NUMERICAL_VALUE_COUNT; ++i)
            {
                this.NumberLabel[i] = source.NumberLabel[i];
            }
        }


        /// <summary>
        /// オブジェクトを複製します。
        /// </summary>
        /// <returns>複製したオブジェクトです。</returns>
        public override InitSettingsData Clone()
        {
            return new InitSettingsData(this);
        }

        #endregion

        /// <summary>
        /// 初期設定ファイル
        /// </summary>
        public string InitConfigFilename { get; set; }

        /// <summary>
        /// 自動で読み込むエディット環境ファイル
        /// </summary>
        public string EditEnvFilename { get; set; }

        /// <summary>
        /// フレームレート
        /// </summary>
        public InitSettingFrameRate FrameRate { get; set; }

        /// <summary>
        /// 解像度
        /// </summary>
        public InitSettingResolution Resolution { get; set; }

        /// <summary>
        /// クリアカラー
        /// </summary>
        public RgbaColor ClearColor { get; set; }

        /// <summary>
        /// ワールドスケール 位置・速度・サイズレンジ
        /// </summary>
        public float WorldScaleRange { get; set; }

        /// <summary>
        /// ワールドスケール 時間
        /// </summary>
        public int WorldScaleTime { get; set; }

        /// <summary>
        /// ワールドスケール グリッドと床の表示スケール
        /// </summary>
        public float GridScale { get; set; }

        /// <summary>
        /// ユーザーデータビットのラベル
        /// </summary>
        public string[] BitLabel { get; set; }

        /// <summary>
        /// ユーザーデータ数値のラベル
        /// </summary>
        public string[] NumberLabel { get; set; }
    }
}
