﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using NintendoWare.ToolDevelopmentKit;

namespace NWCore.DataModel
{
    /// <summary>
    /// カラーアニメーション編集データ
    /// </summary>
    public sealed class ColorEditData : BaseDataModel<ColorEditData>
    {
        #region プロパティ

        /// <summary>
        /// カラー設定タイプ
        /// </summary>
        public Constances.ColorSettingType ColorType { get; set; }

        /// <summary>
        /// ランダムカラー
        /// </summary>
        public RgbaColor[] RandomColors { get; set; }

        /// <summary>
        /// アニメーションカラー
        /// </summary>
        public RgbaColor Color1 { get; set; }

        /// <summary>
        /// アニメーションカラー
        /// </summary>
        public RgbaColor Color2 { get; set; }

        /// <summary>
        /// アニメーションカラー
        /// </summary>
        public RgbaColor Color3 { get; set; }

        /// <summary>
        /// アニメーションタイム
        /// </summary>
        public int Time1 { get; set; }

        /// <summary>
        /// アニメーションタイム
        /// </summary>
        public int Time2 { get; set; }

        /// <summary>
        /// アニメーションタイム
        /// </summary>
        public int Time3 { get; set; }

        /// <summary>
        /// リピート数
        /// </summary>
        public int NumRepeat { get; set; }

        /// <summary>
        /// 繰り返し開始位置
        /// </summary>
        public bool RandInitColor { get; set; }

        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ColorEditData()
        {
            this.Color1 = new RgbaColor();
            this.Color2 = new RgbaColor();
            this.Color3 = new RgbaColor();

            this.RandomColors = null;
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ColorEditData(ColorEditData source)
            : this()
        {
            this.Set(source);
        }

        /// <summary>
        /// 初期化
        /// </summary>
        public override void Initialize()
        {
            this.ColorType = Constances.ColorSettingType.Constant;
            this.Color1.Set(1.0f, 1.0f, 1.0f, 1.0f);
            this.Color2.Set(1.0f, 1.0f, 1.0f, 1.0f);
            this.Color3.Set(1.0f, 1.0f, 1.0f, 1.0f);
            this.Time1 = 25;
            this.Time2 = 50;
            this.Time3 = 75;
            this.NumRepeat = 1;
            this.RandInitColor = false;

            this.RandomColors    = new RgbaColor[1];
            this.RandomColors[0] = new RgbaColor( 1.0f, 1.0f, 1.0f, 1.0f );
        }

        /// <summary>
        /// 設定
        /// </summary>
        public override void Set(ColorEditData source)
        {
            this.ColorType = source.ColorType;
            this.Color1.Set(source.Color1);
            this.Color2.Set(source.Color2);
            this.Color3.Set(source.Color3);
            this.Time1 = source.Time1;
            this.Time2 = source.Time2;
            this.Time3 = source.Time3;
            this.NumRepeat = source.NumRepeat;
            this.RandInitColor = source.RandInitColor;

            if ( source.RandomColors!=null )
            {
                // Make an array of the same size with the source array.
                if ( this.RandomColors==null ||
                     this.RandomColors.Length!=source.RandomColors.Length )
                {
                    this.RandomColors = new RgbaColor[source.RandomColors.Length];
                }

                // Copy the colors.
                for ( int i=0;i<this.RandomColors.Length;++i )
                {
                    this.RandomColors[i] = new RgbaColor( source.RandomColors[i] );
                }
            }
            else
            {
                this.RandomColors = null;
            }
        }

        /// <summary>
        /// オブジェクトを複製します。
        /// </summary>
        /// <returns>複製したオブジェクトです。</returns>
        public override ColorEditData Clone()
        {
            return new ColorEditData(this);
        }
    }

}
