﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

//#define LoadAsynchronously

// Turn this switch on to convert old eset files
// that texture pattern table was still an array.
#define CONVERT_OLD_ESET_FILE

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Text.RegularExpressions;
using System.Windows.Forms;
using System.Xml.Serialization;
using App.Data;
using NintendoWare.ToolDevelopmentKit.Xml;
using NWCore.DataModel;
using NWCore.Serializer;
using NWCore.Viewer;
using NWCore;

namespace App.IO
{
    /// <summary>
    /// Command_FileNewProject用コマンド引数です。
    /// </summary>
    public class FileNewProjectArgs
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public FileNewProjectArgs(bool isRequireInputName)
        {
            this.IsRequireInputName = isRequireInputName;
        }

        /// <summary>
        /// 名前入力を要求します。
        /// </summary>
        public bool IsRequireInputName { get; set; }
    }


    /// <summary>
    /// ドキュメント入出力。
    /// </summary>
    public static class DocumentIO
    {
        private static bool s_requireInputName = true;
        private const int DEF_MEMRY_BUFFER = (1024 * 1024 * 10);
        const string FILEEXT_FTX = ".ftx";
        const string FILEEXT_TGA = ".tga";
        public const int s_MaxFileNameLength = 128;

        /// <summary>
        /// クリップボードID
        /// </summary>
        public const string CLIPBOARDID_PROJECTNODE = "EM4F_ProjectTree_DocumentClipboardData";

        /// <summary>
        /// 名前入力を求めるようにしますか？
        /// </summary>
        /// <remarks>
        /// =true..新規生成で名前入力を求めます。
        /// =false..新規生成でデフォルト名のまま生成します。
        /// </remarks>
        public static bool IsRequireInputName
        {
            get { return s_requireInputName; }
            set { s_requireInputName = value; }
        }

        #region 新規作成

        #endregion

        #region 開く

        /// <summary>
        /// Load emitter set file without creating the emitter set document.
        /// </summary>
        /// <param name="filePath">File path of the eset file.</param>
        /// <returns>The loaded emitter set data serializer.</returns>
        public static EmitterSetDataXml DeserializeEmitterSet(string filePath)
        {
            if (File.Exists(filePath) == false)
            {
                TheApp.OutputLogMsg(LogLevels.Error,
                                    res.Strings.CONSOLE_FILE_NOT_FOUND_MSG,
                                    filePath);
                return null;
            }

            EmitterSetDataXml xml = null;
            try
            {
                byte[] buffer = File.ReadAllBytes(filePath);

                #if CONVERT_OLD_ESET_FILE
                using (var stream = new MemoryStream(buffer))
                {
                    using (var reader = new StreamReader(stream))
                    {
                        // 旧データのファイルなあらば、新しいデータに変換します。
                        string textContent = reader.ReadToEnd();

                        Regex expReplace = new Regex(@"<\s*UI_texPatTbl\[(\d+)\]>");

                        string replacedContent =
                            expReplace.Replace(textContent, "<UI_texPatTbl_$1>");

                        buffer =
                            System.Text.Encoding.UTF8.GetBytes(replacedContent);
                    }
                }
                #endif

                using (var stream = new MemoryStream(buffer))
                {
                    // Load and update the data
                    UpdaterErrorTypes err;
                    object dataModel =
                        TheApp.UpdaterManager.LoadDataModel(DocumentConstants.DotEset,
                                                             stream,
                                                             out err);
                    // 読み込みエラー
                    if (err != UpdaterErrorTypes.Succeed)
                    {
                        TheApp.OutputLogMsg(LogLevels.Warning,
                                             res.Strings.IO_LOADFILE_FAILED_MSG,
                                             filePath);
                    }
                    else
                    {
                        xml = dataModel as EmitterSetDataXml;
                        if (xml == null)
                        {
                            TheApp.OutputLogMsg(LogLevels.Warning,
                                                 res.Strings.IO_LOADFILE_FAILED_MSG,
                                                 filePath);
                        }
                    }
                }
            }
            catch (System.Threading.ThreadAbortException e)
            {
                // The possible cast is that this function call is made by EffectBrowser
                // loader thread, and the thread task has been canceled because, for
                // example, the user has switched to another folder in EffectBrowser.
                DebugConsole.WriteLine("The thread that loads the emitter set ({0}) has been aborted. {1}", filePath, e.Message);
            }
            catch
            {
                TheApp.OutputLogMsg(LogLevels.Warning,
                                     res.Strings.IO_LOADFILE_FAILED_MSG,
                                     filePath);
            }

            return xml;
        }

        /// <summary>
        /// load emitter set file
        /// </summary>
        /// <param name="filepath">path to load</param>
        /// <param name="project">The project to add the emitter set to.</param>
        /// <param name="loadTexture">=true(DEF)..テクスチャも読む</param>
        public static EmitterSetDocument LoadEmitterSetFile( string filePath,
                                                             IProjectDocument project,
                                                             bool loadTexture = true,
                                                             bool bSilent = false )
        {
            if ( ProjectManager.ActiveProject!=null &&
                 ProjectManager.ActiveProject.Count>=TheApp.MAX_EMITTERSET_COUNT )
            {
                // EmitterSetの最大数に達しているとき
                return null;
            }

            EmitterSetDataXml xml = DeserializeEmitterSet(filePath);
            if ( xml==null )
                return null;

            var emitterSetDocument = EmitterSetDocument.CreateFromDeserializer(project, xml);
            emitterSetDocument.SetFileLocation(Path.GetDirectoryName(filePath));
            string fileTitle = Path.GetFileNameWithoutExtension(filePath);
            emitterSetDocument.FileTitle = fileTitle;
            emitterSetDocument.Name = fileTitle;
            emitterSetDocument.OnFinishedLoading( loadTexture, bSilent );
            return emitterSetDocument;
        }

        /// <summary>
        /// デシリアライズ：ストリームからＸＭＬファイルを読み込みます。
        /// </summary>
        /// <typeparam name="TType">シリアライズする型です。</typeparam>
        /// <param name="filePath">読み込みファイルパスです。</param>
        /// <returns>デシリアライズ結果です。</returns>
        public static TType DeserializeFromFile<TType>(Stream reader)
            where TType : class
        {
            // 読み込み
            XmlSerializer xmlSerializer = XmlUtility.CreateCustomXmlSerializer<TType>();
            return xmlSerializer.Deserialize(reader) as TType;
        }


        #endregion

        #region エディット環境読み込み・保存

        /// <summary>
        /// コマンドハンドラ：エディット環境ファイルの読み込み
        /// </summary>
        public static bool Command_FileEnvLoad( IProjectDocument project,
                                                string xmlFilePath )
        {
            // 現在の設定が破棄されることを警告を出す
            if ( ProjectManager.ActiveProject.Count>0 )
            {
                ////var result = ThreadSafeMsgBox.Show( res.Strings.WANING_LOAD_ENVFILE,
                ////                                    res.Strings.WARNING_CAPTION,
                ////                                    MessageBoxButtons.YesNo, MessageBoxIcon.Exclamation );
                ////if ( result!=DialogResult.Yes )
                ////{
                ////    // キャンセルしたメッセージを出します。
                ////    ThreadSafeMsgBox.Show( res.Strings.WANING_LOADENV_CANCEL,
                ////                           res.Strings.WARNING_CAPTION,
                ////                           MessageBoxButtons.OK);
                ////    return false;
                ////}
            }

            ////// 全てを閉じる
            ////Command_FileCloseAll(new Command.MenuCommandArgs(false, null, null));

            // FENVファイル読み込み
            EnvConfigDataXml xmlModel = null;
            try
            {
                xmlModel = XmlUtility.DeserializeFromFile<EnvConfigDataXml>(xmlFilePath);
            }
            catch (Exception ex)
            {
                DebugConsole.WriteLine(ex.Message);
                return false;
            }

            // Load .prev files
            EmitterSetDocument   firstEsetDoc     = null;
            GameSettingsDocument gameSettingsDoc  = null;
            List<string>         missingEsetFiles = new List<string>();
            foreach ( GameConfigDataXml data in xmlModel.PrevDataList )
            {
                if ( missingEsetFiles.IndexOf( data.EmitterSetPath.ToLower() )>=0 )
                    continue;

                EmitterSetDocument esetDoc =
                    GetEmitterSetForGameSettingsDocument( project,
                                                          Path.GetDirectoryName(xmlFilePath),
                                                          data.EmitterSetPath,
                                                          null,
                                                          true );

                if ( esetDoc==null )
                {
                    missingEsetFiles.Add(data.EmitterSetPath.ToLower());
                    ////ThreadSafeMsgBox.Show(StringResource.Get("ERR_ESET_FILE_NOT_FOUND",
                    ////                                         data.EmitterSetPath),
                    ////                      res.Strings.WARNING_CAPTION,
                    ////                      MessageBoxButtons.OK,
                    ////                      MessageBoxIcon.Exclamation);
                    continue;
                }

                // Does the game settings document already exist?
                if ( esetDoc.FindGameSettingsDocument( data.Name )!=null )
                {
                    ////ThreadSafeMsgBox.Show( res.Strings.ERR_PREV_FILE_ALREADY_EXISTS,
                    ////                       res.Strings.ERROR_LOAD_PREV_FILE_CAPTION,
                    ////                       MessageBoxButtons.OK,
                    ////                       MessageBoxIcon.Exclamation );
                    return false;
                }

                // Load the game settings document.
                gameSettingsDoc = LoadGameSettingsDocument( data, esetDoc, null );
                gameSettingsDoc.SetFileLocation( esetDoc.FileLocation );

                // Load the game settings document.
                gameSettingsDoc.LoadFromDeserializer( data );

                // Remember the first emitter set document, we want to select it later.
                if ( firstEsetDoc==null )
                    firstEsetDoc = esetDoc;
            }

            // Convert all the preview model file paths to absolute path.
            foreach (PreviewModelDocument modelDoc in project.PreviewDocument.PreviewModelDocuments)
            {
                var modelData =
                    xmlModel.ConfigData.PreviewConfigData.GetModelData(modelDoc.DocumentIndex);

                // Convert model file relative path.
                if ((modelData != null) &&
                    (string.IsNullOrEmpty(modelData.ModelFilePath) == false) &&
                    (Path.IsPathRooted(modelData.ModelFilePath) == false))
                {
                    // The model file path is a relative path, convert it to an absolute path.
                    // Path.GetFullPath() compose the combined path again, so that it won't look
                    // like "C:\folder1\folder2\..\..\folder3\folder4\file.ext".
                    modelData.ModelFilePath =
                        Path.GetFullPath(Path.Combine(Path.GetDirectoryName(xmlFilePath),
                                                      modelData.ModelFilePath));
                }

                // Convert animation folder relative path.
                if ((modelData != null) &&
                    (string.IsNullOrEmpty(modelData.AnimationFolder) == false) &&
                    (Path.IsPathRooted(modelData.AnimationFolder) == false))
                {
                    // The animation folder path is a relative path, convert it to an absolute path.
                    // Path.GetFullPath() compose the combined path again, so that it won't look
                    // like "C:\folder1\folder2\..\..\folder3\folder4".
                    modelData.AnimationFolder =
                        Path.GetFullPath(Path.Combine(Path.GetDirectoryName(xmlFilePath),
                                                      modelData.AnimationFolder));
                }
            }

            // ゲーム設定の反映
            project.EnvConfigData.Set( xmlModel.ConfigData );

            // Load user functions
            foreach ( PreviewModelDocument modelDoc in project.PreviewDocument.PreviewModelDocuments )
            {
                modelDoc.LoadFromDeserializer( xmlModel );
                modelDoc.UpdateAnimation();
            }

            // Set the file location to the project.
            project.SetFileLocation( Path.GetDirectoryName(xmlFilePath) );
            project.FileTitle = Path.GetFileNameWithoutExtension( xmlFilePath );

            // Select a document.
            ////App.PropertyEdit.ObjectPropertyDialog activeDialog =
            ////    App.PropertyEdit.ObjectPropertyDialog.GetFocusedInstance();
            ////if ( activeDialog!=null )
            ////{
            ////    if ( firstEsetDoc!=null )
            ////        activeDialog.SelectDocument( firstEsetDoc );
            ////    else if ( project is EffectProjectDocument )
            ////        activeDialog.SelectDocument( (project as EffectProjectDocument).WorkSpaceDocument );
            ////}

            // ツリー更新
            project.BuildTreeNodes();


            // 全情報を送信します。
//            Viewer.Message.SendUtility.SendAll();
            return true;
        }


        #endregion

        #region プレビュー読み込み・保存

        /// <summary>
        /// コマンドハンドラ：プレビューファイルの読み込み
        /// </summary>
        public static bool Command_FilePrevLoad( IProjectDocument project,
                                                 string filePath,
                                                 List<string> loadedEmitterSets = null )
        {
            if ( (project is EffectProjectDocument)==false )
                return false;

            GameSettingsDocument gameSettingsDoc = null;

            string dirPath = Path.GetDirectoryName( filePath );

            // .prevファイル読み込み
            GameConfigDataXml content = null;
            try
            {
                content = XmlUtility.DeserializeFromFile<GameConfigDataXml>( filePath );

                EmitterSetDocument esetDoc = null;
                using ( MCSDisableBlock block = new MCSDisableBlock() )
                {
                    esetDoc =
                        GetEmitterSetForGameSettingsDocument( project,
                                                              dirPath,
                                                              content.EmitterSetPath,
                                                              loadedEmitterSets );
                }
                if ( esetDoc==null )
                {
                    ////ThreadSafeMsgBox.Show( StringResource.Get( "ERR_ESET_FILE_NOT_FOUND",
                    ////                                           content.EmitterSetPath ),
                    ////                       res.Strings.WARNING_CAPTION,
                    ////                       MessageBoxButtons.OK,
                    ////                       MessageBoxIcon.Exclamation );
                    return false;
                }

                // Does the game settings document already exist?
                if ( esetDoc.FindGameSettingsDocument( content.Name )!=null )
                {
                    ////ThreadSafeMsgBox.Show( res.Strings.ERR_PREV_FILE_ALREADY_EXISTS,
                    ////                       res.Strings.ERROR_LOAD_PREV_FILE_CAPTION,
                    ////                       MessageBoxButtons.OK,
                    ////                       MessageBoxIcon.Exclamation );
                    return false;
                }

                // Load the game settings document.
                gameSettingsDoc = LoadGameSettingsDocument( content, esetDoc, null );
                gameSettingsDoc.SetFileLocation( dirPath );
            }
            catch ( Exception ex )
            {
                DebugConsole.WriteLine(ex.Message);
                return false;
            }

            // ツリー更新
            project.BuildTreeNodes();

            ////// Make sure the UI definition has been loaded.
            ////MainFrame.SynchronizationContext.Send(() =>
            ////    TheApp.UIManager.UpdatePropertyPanel(PropertyEdit.PropertyPanelID.GameSettingsPropertyPanel));

            ////// Load the game settings document.
            ////gameSettingsDoc.LoadFromDeserializer( content );

            ////// Refresh the page content by showing the first property page of the game settings panel.
            ////MainFrame.SynchronizationContext.Send(() =>
            ////    TheApp.UIManager.UpdatePropertyPanel(PropertyEdit.PropertyPanelID.GameSettingsPropertyPanel));

            // Send the emitter set to the viewer
//            Viewer.Message.SendUtility.SendEmitterSetBinary( gameSettingsDoc.EmitterSetDocument );

            return true;
        }

        /// <summary>
        /// コマンドハンドラ：プレビューファイルの読み込み
        /// </summary>
        public static bool Command_FilePrevLoad( IProjectDocument project,
                                                 EmitterSetDocument owner,
                                                 GameSettingsDocument targetDoc )
        {
            if ( (project is EffectProjectDocument)==false )
                return false;

            if ( owner==null )
                return false;

            if ( targetDoc==null )
                return false;

            System.Windows.Forms.OpenFileDialog dialog =
                new System.Windows.Forms.OpenFileDialog();

            dialog.Filter             = res.Strings.IO_FILEFILTER_CAFE_PREV;
            dialog.Multiselect        = true;
            dialog.InitialDirectory   = Config.Data.DocumentIO.GetLastAccessedDir( DocumentConstants.Prev );
            dialog.FilterIndex        = 0;
            dialog.AutoUpgradeEnabled = Config.Data.DocumentIO.AutoUpgradeEnabled;

            ////if ( dialog.ShowDialog(TheApp.MainFrame)!=System.Windows.Forms.DialogResult.OK )
            ////    return false;

            Config.Data.DocumentIO.SetLastAccessedDir( DocumentConstants.Prev,
                                                       dialog.FileName );

            string filePath = dialog.FileName;
            string dirPath  = Path.GetDirectoryName( filePath );

            string name = Path.GetFileNameWithoutExtension(filePath);
            if (name.Length > DocumentIO.s_MaxFileNameLength)
            {
                DocumentIO.ErrorFileNameLengthLoad(name);
                return false;
            }

            // .prevファイル読み込み
            GameConfigDataXml content = null;
            try
            {
                content = XmlUtility.DeserializeFromFile<GameConfigDataXml>( filePath );

                ////GameSettingsDocumentImportCommand cmd =
                ////    new GameSettingsDocumentImportCommand( content, targetDoc );

                ////TheApp.CommandManager.ScheduleExec( cmd,
                ////                                    cmd.Description,
                ////                                    cmd.DataSrcDesc );
            }
            catch ( Exception ex )
            {
                DebugConsole.WriteLine(ex.Message);
                return false;
            }

            // ツリー更新
            project.BuildTreeNodes();

            // Send the emitter set to the viewer
//            Viewer.Message.SendUtility.SendEmitterSetBinary( owner );

            return true;
        }


        /// <summary>
        /// Load game settings document from the loaded XML content.
        /// </summary>
        /// <param name="content">The XML contents to load the game settings from.</param>
        /// <returns>The loaded game settings document.</returns>
        private static GameSettingsDocument LoadGameSettingsDocument( GameConfigDataXml content,
                                                                      EmitterSetDocument esetDoc,
                                                                      GameSettingsDocument targetDoc )
        {
            IProjectDocument project = ProjectManager.ActiveProject;

            // Process post-deserialize event.
            content.PostDeserialize();

            // Create the game settings document.
            GameSettingsDocument gameSettingsDoc = targetDoc;
            if ( gameSettingsDoc==null )
            {
                gameSettingsDoc = new GameSettingsDocument( esetDoc,
                                                            content.Name );

                // Add it to the emitter set.
                esetDoc.AddGameSettingsDocument( gameSettingsDoc );
            }

            return gameSettingsDoc;
        }


        /// <summary>
        /// Get or load emitter set for the game settings document.
        /// </summary>
        /// <param name="project">The project document.</param>
        /// <returns>The emitter set document.</returns>
        private static EmitterSetDocument
            GetEmitterSetForGameSettingsDocument( IProjectDocument project,
                                                  string basePath,
                                                  string esetPath,
                                                  List<string> loadedEmitterSets,
                                                  bool bLoadFromBasePathWhenFailed = false )
        {
            EmitterSetDocument esetDoc = null;

            // Check if the emitter set has already been loaded.
            string esetName = Path.GetFileNameWithoutExtension( esetPath );
            esetDoc = (project as EffectProjectDocument).FindByName( esetName );

            // The emitter set is not there, load it.
            if ( esetDoc!=null )
                return esetDoc;

            // Compose the full emitter set path.
            string esetFullPath = esetPath;
            if ( Path.IsPathRooted(esetFullPath)==false )
            {
                esetFullPath = Path.GetFullPath(Path.Combine(basePath, esetFullPath));
            }
            else if ( bLoadFromBasePathWhenFailed==true &&
                      File.Exists(esetFullPath) == false)
            {
                esetFullPath = Path.Combine( basePath, Path.GetFileName(esetPath) );
            }

            // The path is not correct, unable to load the emitter set.
            if ( Path.IsPathRooted(esetFullPath)==false ||
                 File.Exists(esetFullPath)==false )
            {
                string msg = string.Format( res.Strings.CONSOLE_FILE_NOT_FOUND_MSG,
                                            esetFullPath );
                TheApp.Logger.Error.AddMessage( msg );
                return null;
            }

            // エミッタセットの読み込み
            esetDoc = DocumentIO.LoadEmitterSetFile( esetFullPath, project );
            if ( esetDoc==null )
                return null;

            if ( loadedEmitterSets!=null )
            {
                // 一緒に読み込んだエミッタセットを通知する
                loadedEmitterSets.Add( esetDoc.FilePath );
            }

            // エミッタセットをプロジェクトへ追加
            ////EmitterSetDocumentAddRemoveCommand cmdAddEmitterSet =
            ////    new EmitterSetDocumentAddRemoveCommand( esetDoc,
            ////                                            project,
            ////                                            EmitterSetDocumentAddRemoveCommand.OpFlg.Add );
            ////TheApp.CommandManager.Execute( cmdAddEmitterSet );

            return esetDoc;
        }

        #endregion

        #region 閉じる

        /// <summary>
        /// コマンドハンドラ。
        /// </summary>
        ////public static void Command_FileCloseAll(MenuCommandArgs args)
        ////{
        ////    if (args.RequireUpdate)
        ////    {
        ////        args.CommandUI.Enabled =
        ////            (ProjectManager.ActiveProject != null
        ////                && ProjectManager.ActiveProject.Count > 0);
        ////        return;
        ////    }

        ////    // 実行
        ////    Debug.Assert(DocumentManager.Documents.Count > 0);

        ////    Viewer.Message.PreviewCommunicator.CloseAllModelInfo();

        ////    DocumentCloser closer = new DocumentCloser();
        ////    if (closer.CloseAll()) { MCSManager.WaitTransmission(); }
        ////}

        #endregion

        #region Utitlity Functions

        /// <summary>
        /// shows a message box to inform name collision
        /// </summary>
        /// <param name="filepath">path already loaded</param>
        public static void ErrorFileNameCollision(string filePath)
        {
            string message = string.Format(res.Strings.IO_FILENAME_COLLISION_MSG, Path.GetFileName(filePath));
            ////ThreadSafeMsgBox.Show( message,
            ////                       res.Strings.WARNING_CAPTION,
            ////                       System.Windows.Forms.MessageBoxButtons.OK,
            ////                       System.Windows.Forms.MessageBoxIcon.Warning );
        }

        /// <summary>
        /// shows a message box to inform name length exceeds maximum
        /// </summary>
        /// <param name="filepath">path already loaded</param>
        public static void ErrorFileNameLengthLoad(string filePath)
        {
            string message = string.Format(res.Strings.IO_FILENAME_LENGTH_LOAD_MSG, Path.GetFileName(filePath));
            //////ThreadSafeMsgBox.Show( message,
            //////                       res.Strings.WARNING_CAPTION,
            //////                       System.Windows.Forms.MessageBoxButtons.OK,
            //////                       System.Windows.Forms.MessageBoxIcon.Warning );
        }

        /// <summary>
        /// shows a message box to inform name length exceeds maximum
        /// </summary>
        /// <param name="filepath">path already loaded</param>
        public static void ErrorFileNameLengthSave(string filePath)
        {
            string message = string.Format(res.Strings.IO_FILENAME_LENGTH_SAVE_MSG, Path.GetFileName(filePath));
            ////ThreadSafeMsgBox.Show(message,
            ////                       res.Strings.WARNING_CAPTION,
            ////                       System.Windows.Forms.MessageBoxButtons.OK,
            ////                       System.Windows.Forms.MessageBoxIcon.Warning);
        }

        /// <summary>
        /// shows a message box to inform name length exceeds maximum
        /// </summary>
        /// <param name="filepath">path already loaded</param>
        public static void ErrorFileNameLengthCopy(string filePath)
        {
            string message = string.Format(res.Strings.IO_FILENAME_LENGTH_COPY_MSG, Path.GetFileName(filePath));
            ////ThreadSafeMsgBox.Show(message,
            ////                       res.Strings.WARNING_CAPTION,
            ////                       System.Windows.Forms.MessageBoxButtons.OK,
            ////                       System.Windows.Forms.MessageBoxIcon.Warning);
        }

        #endregion

        #region Drag/Drop

        /// <summary>
        /// ドラッグ＆ドロップを受け付けられるファイルかチェックします。
        /// </summary>
        public static bool IsAcceptFile(string filePath)
        {
            string fileext = System.IO.Path.GetExtension(filePath).ToLower();
            switch (fileext)
            {
                /*
                 *  ドロップを許可する拡張子を列挙します。
                 */
                case DocumentConstants.DotFEnv:
                    return true;

                case DocumentConstants.DotEset:
                    return (ProjectManager.ActiveProject.Count < TheApp.MAX_EMITTERSET_COUNT);

                case DocumentConstants.DotPrev:
                    return true;

                default:
                    break;
            }
            return false;
        }

        /// <summary>
        /// ファイルのドロップ処理を行います
        /// </summary>
        /// <param name="file">ドロップするファイルのパス</param>
        public static void Command_DropFile(IProjectDocument project, string filePath)
        {
            string fileext = System.IO.Path.GetExtension(filePath).ToLower();
            switch (fileext)
            {
                case DocumentConstants.DotFEnv:
                    {
                        // エディット環境ファイルの読み込み
                        Command_FileEnvLoad(project, filePath);
                    }
                    break;

                case DocumentConstants.DotEset:
                    {
                        // エミッタセットの読み込み
                        var emitterSetDocument =
                            DocumentIO.LoadEmitterSetFile(filePath, project);

                        if (emitterSetDocument != null)
                        {
                            // エミッタセットをプロジェクトへ追加
                            ////TheApp.CommandManager.Execute(
                            ////    new Command.EmitterSetDocumentAddRemoveCommand(
                            ////        emitterSetDocument, project,
                            ////        Command.EmitterSetDocumentAddRemoveCommand.OpFlg.Add));

                            ////// Create default game settings document
                            ////string defName =
                            ////    DocumentCreator.CreateDefaultGameSettingsDocName( emitterSetDocument );
                            ////TheApp.CommandManager.ExecuteIgnoreTargetDocument(
                            ////    new GameSettingsDocumentCreationCommand(defName, emitterSetDocument));

                            ////Viewer.Message.SendUtility.SendEmitterSetBinary(emitterSetDocument);
                        }
                    }
                    break;

                case DocumentConstants.DotPrev:
                    {
                        // エディット環境ファイルの読み込み
                        Command_FilePrevLoad( project, filePath );
                    }
                    break;

                default:
                    break;
            }
        }

        #endregion

        #region テクスチャー読み込み

        /// <summary>
        /// テクスチャーファイルの読み込み
        /// </summary>
        /// <param name="xml">Deserializer of emitter data.</param>
        /// <returns>=true..成功</returns>
        public static bool ReadTextureFile( ComplexEmitterDataXml xml )
        {
            bool bUseRelPath    = xml.UseRelativePath;
            xml.UseRelativePath = false;

            string fullPath = string.Empty;

            ReadTextureFile( string.Empty,
                             xml.EditData.TexPatData0.UI_texPatFileName,
                             xml.EditData.TexRes0,
                             ref fullPath );

            ReadTextureFile( string.Empty,
                             xml.EditData.TexPatData1.UI_texPatFileName,
                             xml.EditData.TexRes1,
                             ref fullPath );

            ReadTextureFile( string.Empty,
                             xml.EditData.TexPatData2.UI_texPatFileName,
                             xml.EditData.TexRes2,
                             ref fullPath );

            ReadTextureFile( string.Empty,
                             xml.EditData.ChildTexPatData.UI_texPatFileName,
                             xml.EditData.ChildData.TextureRes,
                             ref fullPath );

            xml.UseRelativePath = bUseRelPath;

            return true;
        }


        /// <summary>
        /// テクスチャーファイルの読み込み
        /// </summary>
        /// <param name="emitterDoc">EmitterDocument</param>
        /// <returns>=true..成功</returns>
        /// <remarks>fullTexPath..に読み込んだフルパスを出力します。</remarks>
        public static bool ReadTextureFile( string baseFolder,
                                            string texPath,
                                            TextureRes textureRes,
                                            ref string fullTexPath )
        {
            if (String.IsNullOrEmpty(texPath) == true)
            {
                // Clear the images
                textureRes.SetTexture(null, null, 0, 0, -1);
                textureRes.TextureInfo.Initialize();
                return false;
            }

            if ( Path.IsPathRooted( texPath )==true )
            {
                fullTexPath = texPath;
            }
            else if ( DocumentConstants.LocateTextureByFileName(baseFolder,
                                                                texPath,
                                                                out fullTexPath)==false )
            {
                textureRes.SetTexture( null, null, 0, 0, -1 );
                textureRes.TextureInfo.Initialize();
                return false;
            }

            return true;
        }

        /// <summary>
        /// Try to load the texture file and see if the file is valid.
        /// </summary>
        /// <param name="path">The file path to the texture file.</param>
        public static bool TryLoadingTextureFile( string path )
        {
/*            TextureInfo texInfo = null;
            Enum        format  = TextureFormat.Unknown;

            texture_info_comp_selValue[] compSelector = null;
            Bitmap[]                     colorImgs    = null;
            Bitmap[]                     alphaImgs    = null;

            LoadTextureResults result;
            TextureManager     texMgr = TheApp.TextureManager;

            using ( new TextureManager.TextureLoadingMsgBoxBlock() )
            {
                result = texMgr.LoadTexture( path,
                                             out texInfo,
                                             out colorImgs,
                                             out alphaImgs,
                                             out format,
                                             out compSelector );
                if ( result!=LoadTextureResults.Success )
                    return false;
            }*/

            return true;
        }

        #endregion

        #region Node Clone/Copy/Paste

        /// <summary>
        /// Clone the current active document node.
        /// </summary>
        /// <param name="srcDocument">The source document to clone from.</param>
        /// <param name="targetParentDoc">
        /// The parent document to add the cloned document to.
        /// Null to add to the same parent document of the source document.
        /// This is only effective when cloning emitters or game settings documents.
        /// </param>
        /// <param name="iIndex">The index to insert the new document to.</param>
        /// <param name="bUseSrcDocName">Use the name of the source document.</param>
        /// <returns>True on success.</returns>
        public static bool CloneDocumentNode( IDocument srcDocument = null,
                                              IDocument targetParentDoc = null,
                                              int iIndex = -1,
                                              bool bUseSrcDocName = false,
                                              string docName = "" )
        {
////            DocumentClipboardData data = null;
////            IProjectDocument project = ProjectManager.ActiveProject;
////            if ( project == null )
////                return false;

////            // Copy the source document.
////            IDocument srcDoc = srcDocument;
////            if ( srcDoc==null )
////                srcDoc = project.ActiveDocument;
////            if ( srcDoc is EmitterSetDocument )
////            {
////                if ( project.NumEmitterSetDocuments>=TheApp.MAX_EMITTERSET_COUNT )
////                    return false;

////                // Copy emitter set document.
////                data = new DocumentClipboardData( srcDoc as EmitterSetDocument );
////            }
////            else if ( srcDoc is EmitterDocument )
////            {
////                IEmitterSetDocument parentDoc = (srcDoc as EmitterDocument).EmitterSetDocument;
////                if ( parentDoc==null ||
////                     parentDoc.Count>=TheApp.MAX_EMITTER_COUNT )
////                {
////                    return false;
////                }

////                // Copy emitter document.
////                data = new DocumentClipboardData( srcDoc as EmitterDocument );
////            }
////            else if ( srcDoc is GameSettingsDocument )
////            {
////                // Copy game settings document.
////                data = new DocumentClipboardData( srcDoc as GameSettingsDocument );
////            }
////            else
////            {
////                return false;
////            }

////            IDocument toSelect = null;
////            IEmitterSetDocument toSend = null;

////            // Paste the destination document
////            if ( data.DocumentType==GuiObjectID.EmitterSet )
////            {
////                EmitterSetDocument destDoc =
////                    data.DeserializeEmitterSetDocument( project,
////                                                        res.Strings.COMMAND_DESC_CLONE_DOC_NODE );
////                if (destDoc != null)
////                {
////                    toSelect = destDoc;
////                    toSend = destDoc;
////                }
////            }
////            else if ( data.DocumentType==GuiObjectID.Emitter )
////            {
////                IEmitterSetDocument parentDoc = targetParentDoc as EmitterSetDocument;
////                if ( parentDoc==null )
////                {
////                    parentDoc = (srcDoc as EmitterDocument).EmitterSetDocument;
////                    if ( parentDoc==null )
////                        return false;
////                }

////                EmitterDocument destDoc =
////                    data.DeserializeEmitterDocument( parentDoc as EmitterSetDocument,
////                                                     true,
////                                                     true,
////                                                     res.Strings.COMMAND_DESC_CLONE_DOC_NODE,
////                                                     iIndex,
////                                                     bUseSrcDocName );
////                if (destDoc != null)
////                {
////                    toSelect = destDoc;
////                    toSend = destDoc.EmitterSetDocument;
////                }
////            }
////            else if ( data.DocumentType==GuiObjectID.GameSettings )
////            {
////                IEmitterSetDocument parentDoc = targetParentDoc as EmitterSetDocument;
////                if ( parentDoc==null )
////                {
////                    parentDoc = (srcDoc as GameSettingsDocument).EmitterSetDocument;
////                    if ( parentDoc==null )
////                        return false;
////                }

////                GameSettingsDocument destDoc =
////                    data.DeserializeGameSettingsDocument( parentDoc as EmitterSetDocument,
////                                                          true,
////                                                          res.Strings.COMMAND_DESC_CLONE_DOC_NODE,
////                                                          iIndex,
////                                                          bUseSrcDocName );
////                if (destDoc != null)
////                {
////                    toSelect = destDoc;
////                    toSend = destDoc.EmitterSetDocument;
////                }
////            }

////            if (null == toSend)
////                return false;

////            ProjectManager.ActiveDocument = toSelect;

////            // パケット送信
//////            Viewer.Message.SendUtility.SendEmitterSetBinary(toSend);////            DocumentClipboardData data = null;
////            IProjectDocument project = ProjectManager.ActiveProject;
////            if ( project == null )
////                return false;

////            // Copy the source document.
////            IDocument srcDoc = srcDocument;
////            if ( srcDoc==null )
////                srcDoc = project.ActiveDocument;
////            if ( srcDoc is EmitterSetDocument )
////            {
////                if ( project.NumEmitterSetDocuments>=TheApp.MAX_EMITTERSET_COUNT )
////                    return false;

////                // Copy emitter set document.
////                data = new DocumentClipboardData( srcDoc as EmitterSetDocument );
////            }
////            else if ( srcDoc is EmitterDocument )
////            {
////                IEmitterSetDocument parentDoc = (srcDoc as EmitterDocument).EmitterSetDocument;
////                if ( parentDoc==null ||
////                     parentDoc.Count>=TheApp.MAX_EMITTER_COUNT )
////                {
////                    return false;
////                }

////                // Copy emitter document.
////                data = new DocumentClipboardData( srcDoc as EmitterDocument );
////            }
////            else if ( srcDoc is GameSettingsDocument )
////            {
////                // Copy game settings document.
////                data = new DocumentClipboardData( srcDoc as GameSettingsDocument );
////            }
////            else
////            {
////                return false;
////            }

////            IDocument toSelect = null;
////            IEmitterSetDocument toSend = null;

////            // Paste the destination document
////            if ( data.DocumentType==GuiObjectID.EmitterSet )
////            {
////                EmitterSetDocument destDoc =
////                    data.DeserializeEmitterSetDocument( project,
////                                                        res.Strings.COMMAND_DESC_CLONE_DOC_NODE );
////                if (destDoc != null)
////                {
////                    toSelect = destDoc;
////                    toSend = destDoc;
////                }
////            }
////            else if ( data.DocumentType==GuiObjectID.Emitter )
////            {
////                IEmitterSetDocument parentDoc = targetParentDoc as EmitterSetDocument;
////                if ( parentDoc==null )
////                {
////                    parentDoc = (srcDoc as EmitterDocument).EmitterSetDocument;
////                    if ( parentDoc==null )
////                        return false;
////                }

////                EmitterDocument destDoc =
////                    data.DeserializeEmitterDocument( parentDoc as EmitterSetDocument,
////                                                     true,
////                                                     true,
////                                                     res.Strings.COMMAND_DESC_CLONE_DOC_NODE,
////                                                     iIndex,
////                                                     bUseSrcDocName );
////                if (destDoc != null)
////                {
////                    toSelect = destDoc;
////                    toSend = destDoc.EmitterSetDocument;
////                }
////            }
////            else if ( data.DocumentType==GuiObjectID.GameSettings )
////            {
////                IEmitterSetDocument parentDoc = targetParentDoc as EmitterSetDocument;
////                if ( parentDoc==null )
////                {
////                    parentDoc = (srcDoc as GameSettingsDocument).EmitterSetDocument;
////                    if ( parentDoc==null )
////                        return false;
////                }

////                GameSettingsDocument destDoc =
////                    data.DeserializeGameSettingsDocument( parentDoc as EmitterSetDocument,
////                                                          true,
////                                                          res.Strings.COMMAND_DESC_CLONE_DOC_NODE,
////                                                          iIndex,
////                                                          bUseSrcDocName );
////                if (destDoc != null)
////                {
////                    toSelect = destDoc;
////                    toSend = destDoc.EmitterSetDocument;
////                }
////            }

////            if (null == toSend)
////                return false;

////            ProjectManager.ActiveDocument = toSelect;

////            // パケット送信
//////            Viewer.Message.SendUtility.SendEmitterSetBinary(toSend);

            return true;
        }


        /// <summary>
        /// Check if the specified document can be copied on the project tree.
        /// </summary>
        /// <param name="doc">The document.</param>
        /// <returns>True if the document can be copied.</returns>
        public static bool CanCopyDocument( IDocument doc )
        {
            if ( doc is EmitterDocument ||
                 doc is GameSettingsDocument ||
                 doc is FieldDocument ||
                 doc is FluctuationDocument ||
                 doc is ChildDocument )
            {
                return true;
            }

            return false;
        }

        #endregion
    }
}
