﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.IO;
using System.Windows.Forms;
using System.Xml;
using System.Xml.Schema;
using System.Xml.Serialization;
using System.Collections.Generic;
using App.IO;

namespace App
{
    /// <summary>
    /// コンフィグ
    /// </summary>
    public static class Config
    {
        #region Varialbles

        #if BUILD_FOR_CTR
            const string STR_FILE_DEF_SETTING = "files.DefaultSettings_CTR.config";
        #else
            const string STR_FILE_DEF_SETTING = "files.DefaultSettings.config";
        #endif
        const string STR_FILE_CONFIGDATA  = "files.ConfigData_";
        const string STR_CONFIG_EXT       = ".config";

        private static CfConfigData  s_configData = null;
        private static XmlSerializer s_serializer = new XmlSerializer(typeof(CfConfigData));

        private static RecentDocumentManager s_emitterRecentDocuments = new RecentDocumentManager();

        #endregion

        #region Properties

        /// <summary>
        /// コンフィグデータ
        /// </summary>
        public static CfConfigData Data
        {
            get
            {
                return s_configData;
            }
        }

        /// <summary>
        /// コンフィグファイル名を取得します
        /// </summary>
        public static string ConfigFileName
        {
            get
            {
                return Path.GetFileName(ConfigFilePath);
            }
        }

        /// <summary>
        /// コンフィグファイルのパスを取得します
        /// </summary>
        public static string ConfigFilePath
        {
            get
            {
                return Path.ChangeExtension(Application.ExecutablePath, STR_CONFIG_EXT);
            }
        }

        public static RecentDocumentManager EmitterRecentDocuments
        {
            get
            {
                return s_emitterRecentDocuments;
            }
        }

        #endregion

        #region Init / cleanup

        //---------------------------------------------------------------------
        /// <summary>
        /// 初期化
        /// </summary>
        public static bool Initialize()
        {
            bool   bResult = false;
            Stream stream  = null;

            try
            {
                string targetVersion = new CfConfigData().version;
                string configPath    = ConfigFilePath;

                if ( File.Exists(configPath)==true )
                {
                    // バージョンアップ
                    XMLVersionUpdater updater = new XMLVersionUpdater( ResolveSchema,
                                                                       ResolveStylesheet );

                    stream = new FileStream( configPath, FileMode.Open, FileAccess.Read );
                    stream = updater.Update( stream, targetVersion );

                    stream.Position = 0;

                    // Load the default config file.
                    XmlDocument defaultConfigDoc = new XmlDocument();
                    defaultConfigDoc.Load( TheApp.GetManifestResourceStream(STR_FILE_DEF_SETTING) );

                    // Load the actual application config file.
                    XmlDocument currentConfigDoc = new XmlDocument();
                    currentConfigDoc.Load( stream );

                    // To prevent error caused by reading an config file of old version,
                    // we merge the config file with the default config.
                    // The missing elements or attributes will be appended from the
                    // default config file.
                    MergeDefaultConfig( defaultConfigDoc, currentConfigDoc );

                    // Write the content of the merged XML document to a memory
                    // stream and replace the original stream with this one.
                    MemoryStream tmpStream = new MemoryStream();
                    currentConfigDoc.Save( tmpStream );

                    // Close the original file stream.
                    stream.Close();

                    // Replace the file stream with the memory stream.
                    stream          = tmpStream;
                    stream.Position = 0;
                }
                else
                {
                    // ファイルが存在しない場合はデフォルト設定を読み込む
                    stream = TheApp.GetManifestResourceStream(STR_FILE_DEF_SETTING);

                    #if BUILD_FOR_CTR
                    #else
                    if ( TheApp.ConsoleMode==false )
                    {
                        ////App.Controls.ThreadSafeMsgBox.Show( res.Strings.DEFAULT_CONFIG_LINEAR_EDIT_MODE_ENABLED_MESSAGE,
                        ////                                    res.Strings.INFORMATION_CAPTION,
                        ////                                    MessageBoxButtons.OK,
                        ////                                    MessageBoxIcon.Information );
                    }
                    #endif
                }

                XmlReaderSettings settings = new XmlReaderSettings();
                using ( XmlReader reader = XmlReader.Create( stream, settings ) )
                {
                    try
                    {
                        s_configData = (CfConfigData)s_serializer.Deserialize(reader);
                        bResult = true;
                    }
                    catch
                    {
                        // 例外を握りつぶす
                    }
                }

                s_emitterRecentDocuments.Initialize();
            }
//            catch ( Exception exception )
            catch (Exception)
            {
                ////string message = string.Format(res.Strings.APP_CONFIG_LOAD_FAILED, ConfigFileName, exception.Message);
                ////ThreadSafeMsgBox.Show( message,
                ////                       res.Strings.ERROR_CAPTION,
                ////                       System.Windows.Forms.MessageBoxButtons.OK,
                ////                       System.Windows.Forms.MessageBoxIcon.Error );
                bResult = false;
            }
            finally
            {
                if ( stream!=null )
                    stream.Close();
            }

            return bResult;
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// 後始末
        /// </summary>
        public static void Cleanup()
        {
            s_emitterRecentDocuments.Flush();
        }

        #endregion

        #region Utility methods

        /// <summary>
        /// Compare the default config file and the actual application config file.
        /// If there are any missing required elements or attributes, the method
        /// take them from the default config file and append it to the application
        /// config.
        /// </summary>
        /// <param name="defaultDoc">The XML document of the default config file.</param>
        /// <param name="currentDoc">The XML document of the applicaiton config file.</param>
        /// <returns>True on success.</returns>
        private static bool MergeDefaultConfig( XmlDocument defaultDoc,
                                                XmlDocument currentDoc )
        {
            if ( defaultDoc==null || currentDoc==null )
                return false;

            XmlElement rootDefault = defaultDoc.DocumentElement;
            XmlElement rootCurrent = currentDoc.DocumentElement;
            if ( rootDefault==null || rootCurrent==null )
                return false;

            MergeDefaultConfigElement( rootDefault, rootCurrent );

            return true;
        }


        /// <summary>
        /// Compare the XML elements from the default config file and the actual
        /// application config file.
        /// If there are any missing required elements or attributes, the method
        /// take them from the default config file and append it to the application
        /// config.
        /// </summary>
        /// <param name="defElement"></param>
        /// <param name="cfgElement"></param>
        /// <returns></returns>
        private static bool MergeDefaultConfigElement( XmlElement defElement,
                                                       XmlElement cfgElement )
        {
            // Merge child elements
            List<XmlNode> missingNodes = new List<XmlNode>();
            foreach ( XmlNode node in defElement.ChildNodes )
            {
                if ( node.NodeType!=XmlNodeType.Element )
                    continue;

                XmlElement child = node as XmlElement;
                if ( child==null )
                    continue;

                XmlNodeList foundChildren = cfgElement.GetElementsByTagName( child.Name );
                if ( foundChildren==null ||
                     foundChildren.Count<=0 )
                {
                    // This child element is not found.
                    // Append it from the default config to the config file.
                    XmlNode importedNode = cfgElement.OwnerDocument.ImportNode( child, true );
                    missingNodes.Add( importedNode );
                }
                else
                {
                    // The element is found, process these elements.
                    foreach ( XmlNode foundNode in foundChildren )
                    {
                        if ( foundNode.NodeType!=XmlNodeType.Element )
                            continue;

                        XmlElement element = foundNode as XmlElement;
                        if ( element==null )
                            continue;

                        // Recursively process all the matching child elements.
                        MergeDefaultConfigElement( child, element );
                    }
                }
            }

            // Merge attributes
            foreach ( XmlAttribute attr in defElement.Attributes )
            {
                if ( attr==null )
                    continue;

                // The attribute is not found, append it from default config.
                if ( cfgElement.HasAttribute( attr.Name )==false )
                {
                    cfgElement.SetAttribute( attr.Name, attr.Value );
                }
            }

            // Add the missing nodes
            foreach ( XmlNode node in missingNodes )
            {
                cfgElement.AppendChild( node );
            }

            return true;
        }


        /// <summary>
        /// Get the config that if the message box should be shown.
        /// </summary>
        /// <param name="name">The name of the message box config data.</param>
        /// <param name="dlgResult">The previous result of the message box.</param>
        /// <returns>True to show the message box.</returns>
        public static bool ShouldShowMessageBox( string name,
                                                 out DialogResult dlgResult )
        {
            dlgResult = DialogResult.None;
            if ( Config.Data.OptionalMsgBoxShowList==null )
                return true;

            foreach ( CfOptionalMsgBox data in Config.Data.OptionalMsgBoxShowList )
            {
                if ( data!=null &&
                     data.ConfigName.Length==name.Length &&
                     data.ConfigName==name )
                {
                    DialogResult result;
                    if ( Enum.TryParse<DialogResult>( data.DialogResult,
                                                      out result )==true )
                    {
                        dlgResult = result;
                    }

                    return data.Show;
                }
            }

            return true;
        }


        /// <summary>
        /// Enable / disable the message box.
        /// </summary>
        /// <param name="name">The config name for the message box.</param>
        /// <param name="bEnable">True to show the message box.</param>
        /// <param name="dlgResult">The result of the message box.</param>
        public static void EnableMessageBox( string name,
                                             bool bEnable,
                                             DialogResult dlgResult )
        {
            // Does the settings of the message box already exist?
            if ( Config.Data.OptionalMsgBoxShowList!=null )
            {
                foreach ( CfOptionalMsgBox data in Config.Data.OptionalMsgBoxShowList )
                {
                    if ( data!=null &&
                         data.ConfigName.Length==name.Length &&
                         data.ConfigName==name )
                    {
                        data.Show         = bEnable;
                        data.DialogResult = dlgResult.ToString();

                        return;
                    }
                }
            }

            // The data was not found, create it.
            int iNumData = 1;
            if ( Config.Data.OptionalMsgBoxShowList!=null )
                iNumData = Config.Data.OptionalMsgBoxShowList.Length + 1;

            CfOptionalMsgBox[] list = new CfOptionalMsgBox[ iNumData ];
            if ( Config.Data.OptionalMsgBoxShowList!=null )
            {
                // Copy the original data
                for ( int i=0;i<Config.Data.OptionalMsgBoxShowList.Length;++i )
                {
                    list[i] = Config.Data.OptionalMsgBoxShowList[i];
                }
            }

            // Create a new data for the message box.
            CfOptionalMsgBox msgBoxData = new CfOptionalMsgBox();
            msgBoxData.ConfigName = string.Copy( name );
            msgBoxData.Show       = bEnable;

            list[iNumData-1] = msgBoxData;

            Config.Data.OptionalMsgBoxShowList = list;
        }

        #endregion

        #region Delegated methods for resolving the config file

        /// <summary>
        /// スキーマ解決
        /// </summary>
        public static Stream ResolveSchema( string version )
        {
            return TheApp.GetManifestResourceStream(
                STR_FILE_CONFIGDATA + version.Replace( '.', '_' ) + ".xsd" );
        }

        /// <summary>
        /// スタイルシート解決
        /// </summary>
        public static Stream ResolveStylesheet( string version )
        {
            return TheApp.GetManifestResourceStream(
                STR_FILE_CONFIGDATA + version.Replace( '.', '_' ) + ".xsl" );
        }

        #endregion
    }
}
