﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;

using NWCore.DataModel;
using NWCore.Serializer;
using NWCore.Utility;

namespace App.Data
{
    /// <summary>
    /// Class as a manager and intermediate data layer for animation table enabled
    /// data models.
    /// </summary>
    public class AnimationTableManager
    {
        #region Class for storing the informations about animation tables

        /// <summary>
        /// Class for storing the informations about animation tables.
        /// </summary>
        private class AnimTableInfo
        {
            #region Constructors

            /// <summary>
            /// Constructor.
            /// </summary>
            /// <param name="owner">The owner document of the animation tables.</param>
            /// <param name="animTableList">The keeper list of the animation tables.</param>
            public AnimTableInfo( IDocument owner,
                                  AnimTableList animTableList )
            {
                this.OwnerDocument      = owner;
                this.AnimationTableList = animTableList;
            }

            #endregion

            #region Properties

            /// <summary>
            /// Get or set the owner document of the animation tables.
            /// </summary>
            public IDocument OwnerDocument { get; set; }


            /// <summary>
            /// Get or set the keeper list of the animation tables.
            /// </summary>
            public AnimTableList AnimationTableList { get; set; }

            #endregion
        }

        #endregion

        #region Constructor

        /// <summary>
        /// Constructor.
        /// </summary>
        public AnimationTableManager()
        {
        }

        #endregion

        #region Get animation table from data source

        /// <summary>
        /// Get animation table from the specified data source.
        /// </summary>
        /// <param name="dataSrcPathStr">The data source path.</param>
        /// <param name="target">The applied target of the animation table.</param>
        /// <returns>The source animation table data.</returns>
        public AnimTableData GetAnimationTable( string dataSrcPathStr,
                                                AnimTableTargetTypes target )
        {
            Path dataSrcPath = new Path( dataSrcPathStr );
            return GetAnimationTable( dataSrcPath, target );
        }


        /// <summary>
        /// Get animation table from the specified data source.
        /// </summary>
        /// <param name="dataSrcPath">The data source path.</param>
        /// <param name="target">The applied target of the animation table.</param>
        /// <returns>The source animation table data.</returns>
        public AnimTableData GetAnimationTable( Path dataSrcPath,
                                                AnimTableTargetTypes target )
        {
            AnimTableInfo info = null;

            // Find a registered animation table list from the
            // closest parent of the data source path.
            Path docPath = dataSrcPath.Clone();
            while ( docPath.Length>0 )
            {
                PathTreeNode node = m_animTableInfos.FindPathData( docPath, true );
                if ( node!=null && node.Data!=null )
                {
                    info = node.Data as AnimTableInfo;
                    break;
                }

                docPath.ToUpperLevel();
            }

            if ( info==null || info.AnimationTableList==null )
                return null;

            return info.AnimationTableList.FindAnimationTableData( target );
        }

        #endregion

        #region Set animation table to data source

        /// <summary>
        /// Set animation table to the specified data source.
        /// </summary>
        /// <param name="dataSrcPathStr">The data source path.</param>
        /// <param name="data">The animation key frame list to set.</param>
        /// <returns>True on success.</returns>
        public bool SetAnimationTable( string dataSrcPathStr,
                                       BaseKeyFrameList data )
        {
            Path dataSrcPath = new Path( dataSrcPathStr );
            return SetAnimationTable( dataSrcPath, data );
        }


        /// <summary>
        /// Set animation table to the specified data source.
        /// </summary>
        /// <param name="dataSrcPath">The data source path.</param>
        /// <param name="data">The animation key frame list to set.</param>
        /// <returns>True on success.</returns>
        public bool SetAnimationTable( Path dataSrcPath,
                                       BaseKeyFrameList data )
        {
            AnimTableInfo info = null;

            // Find a registered animation table list from the
            // closest parent of the data source path.
            Path docPath = dataSrcPath.Clone();
            while ( docPath.Length>0 )
            {
                PathTreeNode node = m_animTableInfos.FindPathData( docPath, true );
                if ( node!=null && node.Data!=null )
                {
                    info = node.Data as AnimTableInfo;
                    break;
                }

                docPath.ToUpperLevel();
            }

            if ( info==null || info.AnimationTableList==null )
                return false;

            for ( int i=0;i<data.FieldCount;++i )
            {
                AnimTableData dstData =
                    info.AnimationTableList.FindAnimationTableData( data.GetFieldTarget(i) );
                if ( dstData==null )
                    return false;

                if ( ConvertAnimation(data, i, dstData)==false )
                    return false;
            }

            return true;
        }

        #endregion

        #region Register animation table

        /// <summary>
        /// Register the animation table list.
        /// </summary>
        /// <param name="owner">The owner document of the animation tables.</param>
        /// <param name="animTableList">The animation table list.</param>
        public void RegisterAnimationTableList( IDocument owner,
                                                AnimTableList animTableList )
        {
            AnimTableInfo info = new AnimTableInfo(owner, animTableList);
            PathTreeNode  node =  m_animTableInfos.AddPath( owner.DataScrPath,
                                                            info,
                                                            true );

            // The node already exists? AddPath() fails and returns null
            // when the node already exists.
            if ( node==null )
            {
                // Replace the node data with the new one.
                node = m_animTableInfos.FindPathData( owner.DataScrPath, true );
                if ( node!=null )
                    node.Data = info;
            }
        }


        /// <summary>
        /// Unregister the animation table list.
        /// </summary>
        /// <param name="owner">The owner document of the animation tables.</param>
        public void UnregisterAnimationTableList( IDocument owner )
        {
            m_animTableInfos.RemovePath( owner.DataScrPath, true );
        }

        #endregion

        #region Convert animation key frames

        /// <summary>
        /// Convert animation key frame data.
        /// </summary>
        /// <param name="src">The source data.</param>
        /// <param name="iChannelIndex">The animation channel in the key frames.</param>
        /// <param name="dst">The destination data.</param>
        /// <returns>True on success.</returns>
        public bool ConvertAnimation( BaseKeyFrameList src,
                                      int iChannelIndex,
                                      AnimTableData dst )
        {
            dst.Initialize();

            dst.Name                      = src.GetFieldTarget( iChannelIndex );
            dst.InterpolationType         = src.InterpolationType;
            dst.IsEnabled                 = src.IsEnabled;
            dst.IsLoop                    = src.IsLoopEnabled;
            dst.NumLoopFrames             = src.NumLoopFrames;
            dst.IsRandomStartFrameEnabled = src.IsRandomStartFrameEnabled;

            if ( src is KeyFrameList<int> )
            {
                foreach ( KeyFrameList<int>.KeyFrameData srcKeyFrame in src )
                {
                    IntAnimTableKeyFrameData dstKeyFrame = new IntAnimTableKeyFrameData();
                    dstKeyFrame.Initialize();

                    dstKeyFrame.KeyFrame = srcKeyFrame.KeyFrame;
                    dstKeyFrame.Value    = srcKeyFrame[iChannelIndex];

                    dst.KeyFrames.Add( dstKeyFrame );
                }
            }
            else if ( src is KeyFrameList<float> )
            {
                foreach ( KeyFrameList<float>.KeyFrameData srcKeyFrame in src )
                {
                    FloatAnimTableKeyFrameData dstKeyFrame = new FloatAnimTableKeyFrameData();
                    dstKeyFrame.Initialize();

                    dstKeyFrame.KeyFrame = srcKeyFrame.KeyFrame;
                    dstKeyFrame.Value    = srcKeyFrame[iChannelIndex];

                    dst.KeyFrames.Add( dstKeyFrame );
                }
            }
            else
            {
                return false;
            }

            return true;
        }


        /// <summary>
        /// Convert animation key frame data.
        /// </summary>
        /// <param name="src">The source data.</param>
        /// <returns>The converted array of animation table data.</returns>
        public AnimTableListXml ConvertAnimation( BaseKeyFrameList src )
        {
            EmitterAnimTableListXml animations = new EmitterAnimTableListXml();
            for ( int i=0;i<src.FieldCount;++i )
            {
                AnimTableData dst = new AnimTableData();
                dst.Initialize();

                dst.Name                      = src.GetFieldTarget( i );
                dst.InterpolationType         = src.InterpolationType;
                dst.IsEnabled                 = src.IsEnabled;
                dst.IsLoop                    = src.IsLoopEnabled;
                dst.NumLoopFrames             = src.NumLoopFrames;
                dst.IsRandomStartFrameEnabled = src.IsRandomStartFrameEnabled;

                if ( src is KeyFrameList<int> )
                {
                    foreach ( KeyFrameList<int>.KeyFrameData srcKeyFrame in src )
                    {
                        IntAnimTableKeyFrameData dstKeyFrame = new IntAnimTableKeyFrameData();
                        dstKeyFrame.Initialize();

                        dstKeyFrame.KeyFrame = srcKeyFrame.KeyFrame;
                        dstKeyFrame.Value    = srcKeyFrame[i];

                        dst.KeyFrames.Add( dstKeyFrame );
                    }
                }
                else if ( src is KeyFrameList<float> )
                {
                    foreach ( KeyFrameList<float>.KeyFrameData srcKeyFrame in src )
                    {
                        FloatAnimTableKeyFrameData dstKeyFrame = new FloatAnimTableKeyFrameData();
                        dstKeyFrame.Initialize();

                        dstKeyFrame.KeyFrame = srcKeyFrame.KeyFrame;
                        dstKeyFrame.Value    = srcKeyFrame[i];

                        dst.KeyFrames.Add( dstKeyFrame );
                    }
                }
                else
                {
                    return null;
                }

                // Add the animation table data.
                animations.AnimationList.Animations.Add( dst );
            }

            return animations;
        }


        /// <summary>
        /// Convert animation key frame data.
        /// </summary>
        /// <param name="src">The source data.</param>
        /// <param name="iChannelIndex">The animation channel in the key frames.</param>
        /// <param name="dst">The destination data.</param>
        /// <returns>True on success.</returns>
        public bool ConvertAnimation( AnimTableData src,
                                      int iChannelIndex,
                                      KeyFrameList<int> dst )
        {
            dst.InterpolationType         = src.InterpolationType;
            dst.IsEnabled                 = src.IsEnabled;
            dst.IsLoopEnabled             = src.IsLoop;
            dst.NumLoopFrames             = src.NumLoopFrames;
            dst.IsRandomStartFrameEnabled = src.IsRandomStartFrameEnabled;

            for ( int i=0;i<src.KeyFrames.Count;++i )
            {
                IntAnimTableKeyFrameData srcKeyFrame =
                    src.KeyFrames[i] as IntAnimTableKeyFrameData;
                if ( srcKeyFrame==null )
                    continue;

                KeyFrameList<int>.KeyFrameData dstKeyFrame;
                if ( i>=dst.Count )
                    dstKeyFrame = dst.AddNew() as KeyFrameList<int>.KeyFrameData;
                else
                    dstKeyFrame = dst[i] as KeyFrameList<int>.KeyFrameData;

                if ( dstKeyFrame==null )
                    continue;

                dstKeyFrame.KeyFrame       = srcKeyFrame.KeyFrame;
                dstKeyFrame[iChannelIndex] = srcKeyFrame.Value;
            }

            return true;
        }


        /// <summary>
        /// Convert animation key frame data.
        /// </summary>
        /// <param name="src">The source data.</param>
        /// <param name="iChannelIndex">The animation channel in the key frames.</param>
        /// <param name="dst">The destination data.</param>
        /// <returns>True on success.</returns>
        public bool ConvertAnimation( AnimTableData src,
                                      int iChannelIndex,
                                      KeyFrameList<float> dst )
        {
            dst.InterpolationType         = src.InterpolationType;
            dst.IsEnabled                 = src.IsEnabled;
            dst.IsLoopEnabled             = src.IsLoop;
            dst.NumLoopFrames             = src.NumLoopFrames;
            dst.IsRandomStartFrameEnabled = src.IsRandomStartFrameEnabled;

            for ( int i=0;i<src.KeyFrames.Count;++i )
            {
                FloatAnimTableKeyFrameData srcKeyFrame =
                    src.KeyFrames[i] as FloatAnimTableKeyFrameData;
                if ( srcKeyFrame==null )
                    continue;

                KeyFrameList<float>.KeyFrameData dstKeyFrame;
                if ( i>=dst.Count )
                    dstKeyFrame = dst.AddNew() as KeyFrameList<float>.KeyFrameData;
                else
                    dstKeyFrame = dst[i] as KeyFrameList<float>.KeyFrameData;

                if ( dstKeyFrame==null )
                    continue;

                dstKeyFrame.KeyFrame       = srcKeyFrame.KeyFrame;
                dstKeyFrame[iChannelIndex] = srcKeyFrame.Value;
            }

            return true;
        }


        /// <summary>
        /// Convert animation key frame data.
        /// </summary>
        /// <param name="srcList">The array of source animation data.</param>
        /// <param name="dst">The destination data.</param>
        /// <returns>True on success.</returns>
        public bool ConvertAnimation( AnimTableData[] srcList,
                                      BaseKeyFrameList dst )
        {
            KeyFrameList<int>   dstInt   = dst as KeyFrameList<int>;
            KeyFrameList<float> dstFloat = dst as KeyFrameList<float>;
            if ( dstInt==null && dstFloat==null )
                return false;

            for ( int i=0;i<srcList.Length;++i )
            {
                AnimTableData src = srcList[i];
                if ( src==null )
                    continue;

                if ( dstInt!=null )
                {
                    if ( ConvertAnimation( src, i, dstInt )==false )
                        return false;
                }
                else
                {
                    if ( ConvertAnimation( src, i, dstFloat )==false )
                        return false;
                }
            }

            return true;
        }

        #endregion

        #region Member variables

        private PathTree m_animTableInfos = new PathTree( null );

        #endregion
    }

    #region Interface IKeyFrameData

    /// <summary>
    /// Interface for a key frame data.
    /// </summary>
    public interface IKeyFrameData : ICloneable
    {
        /// <summary>Get the parent key frame list.</summary>
        BaseKeyFrameList Parent { get; set; }

        /// <summary>Get or set the key frame number.</summary>
        int KeyFrame { get; set; }

        /// <summary>Get the number of the values of the key frame data.</summary>
        int Count { get; }

        /// <summary>
        /// *** This property is for the data grid to access the values. ***
        /// Get or set the value of the key frame.
        /// </summary>
        /// <remarks>
        /// The value get from this property is only for the data grid view to
        /// display on UI, the value can be converted for displaying purpose,
        /// so DO NOT access this property directly if you want to get the
        /// actual value.
        /// Eq. the value you get from this property can be a degree but a
        /// radian internally.
        /// </remarks>
        object UIValue0 { get; set; }

        /// <summary>
        /// *** This property is for the data grid to access the values. ***
        /// Get or set the value of the key frame.
        /// </summary>
        /// <remarks>
        /// The value get from this property is only for the data grid view to
        /// display on UI, the value can be converted for displaying purpose,
        /// so DO NOT access this property directly if you want to get the
        /// actual value.
        /// Eq. the value you get from this property can be a degree but a
        /// radian internally.
        /// </remarks>
        object UIValue1 { get; set; }

        /// <summary>
        /// *** This property is for the data grid to access the values. ***
        /// Get or set the value of the key frame.
        /// </summary>
        /// <remarks>
        /// The value get from this property is only for the data grid view to
        /// display on UI, the value can be converted for displaying purpose,
        /// so DO NOT access this property directly if you want to get the
        /// actual value.
        /// Eq. the value you get from this property can be a degree but a
        /// radian internally.
        /// </remarks>
        object UIValue2 { get; set; }

        /// <summary>
        /// *** This property is for the data grid to access the values. ***
        /// Get or set the value of the key frame.
        /// </summary>
        /// <remarks>
        /// The value get from this property is only for the data grid view to
        /// display on UI, the value can be converted for displaying purpose,
        /// so DO NOT access this property directly if you want to get the
        /// actual value.
        /// Eq. the value you get from this property can be a degree but a
        /// radian internally.
        /// </remarks>
        object UIValue3 { get; set; }

        /// <summary>
        /// Copy the given key frame data.
        /// </summary>
        /// <param name="src">The source key frame data to copy from.</param>
        void Copy( object src );

        /// <summary>
        /// Compare this key frame with the given key frame data.
        /// </summary>
        /// <param name="data">The key frame data to compare to.</param>
        /// <returns>True if equal.</returns>
        bool IsEqual( IKeyFrameData data );
    }

    #endregion

    #region Event handler delegation for converting the key frame values

    /// <summary>
    /// The event handler delegation for converting the key frame values
    /// to and from the data grid view on UI.
    /// </summary>
    /// <param name="keyFrame">The key frame data.</param>
    /// <param name="originalValue">The original value.</param>
    /// <param name="convertedValue">The converted value.</param>
    /// <returns>
    /// True on success, false indicates that the key frame should use the default conversion
    /// instead of the converted value returned from this handler.
    /// </returns>
    public delegate bool KeyFrameValueConvertEventHandler( IKeyFrameData keyFrame,
                                                           object originalValue,
                                                           out object convertedValue );

    #endregion

    public class KeyFrameListSelectionInfo
    {
        /// <summary>
        /// Gets the index of the selected key frame.
        /// Warning: this is the index of the key frame, not the key frame 'value'.
        /// </summary>
        public int KeyFrameIndex { get; private set; }

        /// <summary>
        /// Gets the indices of the select field for the given key frame.
        /// </summary>
        public int[] FieldIndices { get; private set; }

        public KeyFrameListSelectionInfo(int keyFrameIndex, params int[] fieldIndices)
            : this(keyFrameIndex, (IEnumerable<int>)fieldIndices)
        {
        }

        public KeyFrameListSelectionInfo(int keyFrameIndex, IEnumerable<int> fieldIndices)
        {
            KeyFrameIndex = keyFrameIndex;
            FieldIndices = fieldIndices != null
                ? fieldIndices.ToArray()
                : new int[0];
        }
    }

    #region Abstract class BaseKeyFrameList

    /// <summary>
    /// Abstract class of the key frame list for public access.
    /// </summary>
    public abstract class BaseKeyFrameList : CollectionBase,
                                             IBindingList,
                                             ICloneable
    {
        #region Enum for the validate insert key frame result

        /// <summary>
        /// Enum for the validate insert key frame result.
        /// </summary>
        public enum InsertKeyFrameValidateResult
        {
            Success,
            ExceedMaxKeyFrameCount,
            ExceedMaxKeyFrameNumber,
            InvalidKeyFrameNumber
        }

        #endregion

        #region Properties

        /// <summary>
        /// Get or set the maximum number of key frames that can be added.
        /// </summary>
        public int MaxNumKeyFrames
        {
            get { return m_iMaxNumKeyFrames; }
            set { m_iMaxNumKeyFrames = value; }
        }


        /// <summary>
        /// Get or set the interpolation type.
        /// </summary>
        public AnimInterpolationTypes InterpolationType { get; set; }


        /// <summary>
        /// Get or set the flag indicating whether the animation is enabled.
        /// </summary>
        public bool IsEnabled { get; set; }


        /// <summary>
        /// Get or set the flag indicating whether the looping is enabled.
        /// </summary>
        public bool IsLoopEnabled { get; set; }


        /// <summary>
        /// Get or set the number of the loop frames.
        /// </summary>
        public int NumLoopFrames { get; set; }


        /// <summary>
        /// Get or set the flag indicating whether the animation start frame should be randomized.
        /// </summary>
        public bool IsRandomStartFrameEnabled { get; set; }

        /// <summary>
        /// Gets or sets the selected key frames and field.
        /// </summary>
        public KeyFrameListSelectionInfo[] SelectedElements { get; set; }

        #endregion

        #region Abstract methods/properties

        /// <summary>
        /// Get or set the key frame data.
        /// </summary>
        /// <param name="iIndex">The index to the key frame data to access.</param>
        /// <returns>The key frame data.</returns>
        public abstract IKeyFrameData this[int iIndex] { get; set; }

        /// <summary>
        /// Get the number of fields in the key frames.
        /// </summary>
        public abstract int FieldCount { get; }

        /// <summary>
        /// Add a key frame data with the default key frame number.
        /// </summary>
        /// <returns>The index to the added key frame.</returns>
        public abstract int AddDefaultKeyFrame();

        /// <summary>
        /// Set the display name on the data grid for the specified key frame value field.
        /// </summary>
        /// <param name="iFieldIndex">The index to the key frame value field.</param>
        /// <param name="name">The display name.</param>
        /// <param name="target">The applied target of the animation table.</param>
        public abstract void SetupField( int iFieldIndex,
                                         string name,
                                         AnimTableTargetTypes target );

        /// <summary>
        /// Get the display name on the data grid for the specified key frame value field.
        /// </summary>
        /// <param name="iFieldIndex">The index to the key frame value field.</param>
        public abstract string GetFieldName( int iFieldIndex );


        /// <summary>
        /// Get the source data name of the specified key frame value field.
        /// </summary>
        /// <param name="iFieldIndex">The index to the key frame value field.</param>
        /// <returns>The applied target of the animation table field.</returns>
        public abstract AnimTableTargetTypes GetFieldTarget( int iFieldIndex );


        /// <summary>
        /// Determine if a new key frame can be inserted after the given row.
        /// </summary>
        /// <param name="iRowIndex">The index of the row.</param>
        /// <param name="iMaxFrameNumber">The maximum key frame number.</param>
        /// <returns>The result.</returns>
        public abstract InsertKeyFrameValidateResult CanInsertKeyFrameAfter( int iRowIndex,
                                                                             int iMaxFrameNumber );

        #endregion

        #region Implementation for IBindingList

        /// <summary>
        /// *** This method is implemented for the DataGridView, do not access it. ***
        /// Create a new key frame data.
        /// </summary>
        /// <returns>The created key frame data.</returns>
        public virtual object AddNew()
        {
            return null;
        }


        /// <summary>
        /// *** This method is implemented for the DataGridView, do not access it. ***
        /// Insert key frame data to the list.
        /// </summary>
        /// <param name="iIndex">The index to insert to.</param>
        /// <param name="value">The value to insert.</param>
        public abstract void Insert( int iIndex,
                                     object value );

        /// <summary>
        /// *** This method is implemented for the DataGridView, do not access it. ***
        /// Create a new key frame data.
        /// </summary>
        /// <returns>The created key frame data.</returns>
        public abstract void Remove( IKeyFrameData value );


        /// <summary>
        /// *** This method is implemented for the DataGridView, do not access it. ***
        /// Get the flag indicating whether allowing editing the items of this list.
        /// </summary>
        bool IBindingList.AllowEdit
        {
            get { return true; }
        }


        /// <summary>
        /// *** This method is implemented for the DataGridView, do not access it. ***
        /// Get the flag indicating whether allowing adding new items to this list.
        /// </summary>
        bool IBindingList.AllowNew
        {
            get { return true; }
        }


        /// <summary>
        /// *** This method is implemented for the DataGridView, do not access it. ***
        /// Get the flag indicating whether allowing removing items from this list.
        /// </summary>
        bool IBindingList.AllowRemove
        {
            get { return true; }
        }


        /// <summary>
        /// *** This method is implemented for the DataGridView, do not access it. ***
        /// Get the flag indicating whether supporting item change notification.
        /// </summary>
        bool IBindingList.SupportsChangeNotification
        {
            get { return true; }
        }


        /// <summary>
        /// *** This method is implemented for the DataGridView, do not access it. ***
        /// Get the flag indicating whether supporting searching of items.
        /// </summary>
        bool IBindingList.SupportsSearching
        {
            get { return false; }
        }


        /// <summary>
        /// *** This method is implemented for the DataGridView, do not access it. ***
        /// Get the flag indicating whether supporting sorting of items.
        /// </summary>
        bool IBindingList.SupportsSorting
        {
            get { return true; }
        }


        /// <summary>
        /// *** This method is implemented for the DataGridView, do not access it. ***
        /// *** This property is not supported. ***
        /// Get the flag indicating whether the list is sorted.
        /// </summary>
        bool IBindingList.IsSorted
        {
            get
            {
                int iPrevKeyFrame = int.MinValue;
                foreach ( object item in this.InnerList )
                {
                    IKeyFrameData data = item as IKeyFrameData;
                    if ( data!=null )
                    {
                        if ( data.KeyFrame<iPrevKeyFrame )
                            return false;

                        iPrevKeyFrame = data.KeyFrame;
                    }
                }

                return true;
            }
        }


        /// <summary>
        /// *** This method is implemented for the DataGridView, do not access it. ***
        /// *** This property is not supported. ***
        /// Get the sort direction of the list.
        /// </summary>
        ListSortDirection IBindingList.SortDirection
        {
            get { return m_sortDirection; }
        }


        /// <summary>
        /// *** This method is implemented for the DataGridView, do not access it. ***
        /// *** This property is not supported. ***
        /// Get the sort property of the list.
        /// </summary>
        PropertyDescriptor IBindingList.SortProperty
        {
            get { return null; }
        }


        /// <summary>
        /// *** This method is implemented for the DataGridView, do not access it. ***
        /// *** This method is not supported. ***
        /// Adds the PropertyDescriptor to the indexes used for searching.
        /// </summary>
        /// <param name="property">The PropertyDescriptor.</param>
        void IBindingList.AddIndex( PropertyDescriptor property )
        {
            throw new NotSupportedException();
        }


        /// <summary>
        /// *** This method is implemented for the DataGridView, do not access it. ***
        /// *** This method is not supported. ***
        /// Sort the list.
        /// </summary>
        /// <param name="property">The properties for sorting.</param>
        /// <param name="direction">The sorting direction.</param>
        void IBindingList.ApplySort( PropertyDescriptor property,
                                     ListSortDirection direction )
        {
            SortKeyFrames();
        }


        /// <summary>
        /// *** This method is implemented for the DataGridView, do not access it. ***
        /// *** This method is not supported. ***
        /// Search item in the list.
        /// </summary>
        /// <param name="property">The properties for searching.</param>
        /// <param name="key">The key of the item to search for.</param>
        int IBindingList.Find( PropertyDescriptor property,
                               object key )
        {
            throw new NotSupportedException();
        }


        /// <summary>
        /// *** This method is implemented for the DataGridView, do not access it. ***
        /// *** This method is not supported. ***
        /// Removes the PropertyDescriptor from the indexes used for searching.
        /// </summary>
        /// <param name="property">The PropertyDescriptor.</param>
        void IBindingList.RemoveIndex( PropertyDescriptor property )
        {
            throw new NotSupportedException();
        }


        /// <summary>
        /// *** This method is implemented for the DataGridView, do not access it. ***
        /// *** This method is not supported. ***
        /// Removes any sort applied using ApplySort.
        /// </summary>
        void IBindingList.RemoveSort()
        {
        }

        #endregion

        #region Utility methods

        /// <summary>
        /// Sort the key frames with the frame number.
        /// </summary>
        /// <returns>True if the order is changed.</returns>
        public bool SortKeyFrames()
        {
            var comparer = new KeyFrameComparer();
            this.InnerList.Sort( comparer );

            return comparer.OrderChanged;
        }


        private class KeyFrameComparer : IComparer
        {
            public bool OrderChanged { get; private set; }

            public int Compare(object x, object y)
            {
                var xx = x as IKeyFrameData;
                if (xx == null)
                    return 0;
                var yy = y as IKeyFrameData;
                if (yy == null)
                    return 0;

                var result = xx.KeyFrame.CompareTo(yy.KeyFrame);

                if (result != 0)
                    OrderChanged = true;

                return result;
            }
        }


        /// <summary>
        /// Clone this instance. Implemented for the ICloneable interface.
        /// </summary>
        /// <returns>The copy of this instance.</returns>
        public virtual object Clone()
        {
            return null;
        }


        /// <summary>
        /// Copy the data.
        /// The field, data source definitions and default key frame data will not be copied.
        /// </summary>
        /// <param name="src">The source key frame list.</param>
        /// <returns>True on success.</returns>
        public virtual bool CopyData( BaseKeyFrameList src )
        {
            if ( src.FieldCount!=this.FieldCount )
                return false;

            if ( src.GetType().Equals(this.GetType())==false )
                return false;

            this.Clear();

            this.IsEnabled                 = src.IsEnabled;
            this.IsLoopEnabled             = src.IsLoopEnabled;
            this.InterpolationType         = src.InterpolationType;
            this.IsRandomStartFrameEnabled = src.IsRandomStartFrameEnabled;
            this.NumLoopFrames             = src.NumLoopFrames;

            // Copy the key frame data.
            this.InnerList.Capacity = src.InnerList.Count;
            foreach ( IKeyFrameData srcData in src.InnerList )
            {
                this.Insert( -1, srcData.Clone() );
            }

            return true;
        }


        /// <summary>
        /// Compare this animation table key frame list with the given list.
        /// </summary>
        /// <param name="list">The animation key frame list to compare to.</param>
        /// <returns>True if equal.</returns>
        public virtual bool IsEqual( BaseKeyFrameList list )
        {
            if ( this.Count!=list.Count )
                return false;

            if ( this.IsEnabled!=list.IsEnabled )
                return false;

            if ( this.IsLoopEnabled!=list.IsLoopEnabled )
                return false;

            if ( this.InterpolationType!=list.InterpolationType )
                return false;

            if ( this.IsRandomStartFrameEnabled!=list.IsRandomStartFrameEnabled )
                return false;

            if ( this.NumLoopFrames!=list.NumLoopFrames )
                return false;

            if ( this.FieldCount!=list.FieldCount )
                return false;

            for ( int i=0;i<this.Count;++i )
            {
                if ( this[i].IsEqual(list[i])==false )
                    return false;
            }

            return true;
        }

        #endregion

        #region Begin / end update

        /// <summary>
        /// Begin a batch of updates.
        /// Between BeginUpdate() and EndUpdate(), ListChange event will not be triggered.
        /// </summary>
        public void BeginUpdate()
        {
            if (m_iBeginUpdateCount == 0)
            {
                m_bFireEvtAtEndUpdate = false;
            }

            ++m_iBeginUpdateCount;
        }


        /// <summary>
        /// End a batch of modifications.
        /// If any modifications between BeginUpdate() and EndUpdate()
        /// request triggering ListChange event, the event will be triggered
        /// here.
        /// </summary>
        public void EndUpdate()
        {
            m_iBeginUpdateCount = Math.Max(m_iBeginUpdateCount - 1, 0);
            if ((m_iBeginUpdateCount == 0) && (m_bFireEvtAtEndUpdate == true))
            {
                TriggerListChangedEvent(new ListChangedEventArgs(ListChangedType.ItemChanged, -1));
            }
        }


        /// <summary>
        /// End a batch of modifications.
        /// If any modifications between BeginUpdate() and EndUpdate()
        /// request triggering ListChange event, the event will be triggered
        /// here.
        /// </summary>
        /// <param name="overrideChangedType">Overridden list changed type.</param>
        /// <param name="iNewIndex">Changed index.</param>
        public void EndUpdate(ListChangedType overrideChangedType,
                              int iNewIndex)
        {
            m_iBeginUpdateCount = Math.Max(m_iBeginUpdateCount - 1, 0);
            TriggerListChangedEvent(new ListChangedEventArgs(overrideChangedType, iNewIndex));
        }

        #endregion

        #region Events

        /// <summary>Event triggered when the list has changed.</summary>
        public event ListChangedEventHandler ListChanged;

        /// <summary>
        /// Trigger ListChanged event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        public void TriggerListChangedEvent( ListChangedEventArgs e )
        {
            if (m_iBeginUpdateCount > 0)
            {
                m_bFireEvtAtEndUpdate = true;
                return;
            }

            if (ListChanged != null)
            {
                ListChanged(this, e);
            }
        }


        /// <summary>
        /// The event triggers when the data grid requesting data from the key frame.
        /// User can convert the internal value to anything for displaying purpose.
        /// </summary>
        public event KeyFrameValueConvertEventHandler ConvertKeyFrameValueToUI;

        /// <summary>
        /// Trigger ConvertKeyFrameValueToUI event.
        /// </summary>
        /// <param name="keyFrame">The key frame.</param>
        /// <param name="originalValue">The original value.</param>
        /// <param name="convertedValue">The converted value.</param>
        /// <returns>
        /// True on success. False indicates that the key frame should use the default conversion
        /// instead of the converted value returned from this method.
        /// </returns>
        public bool TriggerConvertKeyFrameValueToUIEvent( IKeyFrameData keyFrame,
                                                          object originalValue,
                                                          out object convertedValue )
        {
            if ( this.ConvertKeyFrameValueToUI!=null )
            {
                return this.ConvertKeyFrameValueToUI( keyFrame,
                                                      originalValue,
                                                      out convertedValue );
            }
            else
            {
                convertedValue = null;
                return false;
            }
        }

        /// <summary>
        /// The event triggers when the data grid trying to set data to the key frame.
        /// User can convert the displayed value to the internal data.
        /// </summary>
        public event KeyFrameValueConvertEventHandler ConvertUIValueToKeyFrame;

        /// <summary>
        /// Trigger ConvertUIValueToKeyFrame event.
        /// </summary>
        /// <param name="keyFrame">The key frame.</param>
        /// <param name="originalValue">The original value.</param>
        /// <param name="convertedValue">The converted value.</param>
        /// <returns>
        /// True on success. False indicates that the key frame should use the default conversion
        /// instead of the converted value returned from this method.
        /// </returns>
        public bool TriggerConvertUIValueToKeyFrameEvent( IKeyFrameData keyFrame,
                                                          object originalValue,
                                                          out object convertedValue )
        {
            if ( this.ConvertUIValueToKeyFrame!=null )
            {
                return this.ConvertUIValueToKeyFrame( keyFrame,
                                                      originalValue,
                                                      out convertedValue );
            }
            else
            {
                convertedValue = null;
                return false;
            }
        }

        #endregion

        #region Member variables

        private ListSortDirection  m_sortDirection  = ListSortDirection.Ascending;

        // Besides here, you also need to modify UIKeyFrameAnimTable.m_iMaxNumKeyFrames.
        private int m_iMaxNumKeyFrames = 32;

        private int  m_iBeginUpdateCount   = 0;
        private bool m_bFireEvtAtEndUpdate = false;

        #endregion
    }

    #endregion

    #region Class for keeping a list of key frame data

    /// <summary>
    /// Class for keeping a list of key frame data.
    /// </summary>
    /// <typeparam name="T">The type of the data fields.</typeparam>
    public sealed class KeyFrameList<T> : BaseKeyFrameList
    {
        #region Class for storing the key frame data

        /// <summary>
        /// Internal class for storing a key frame data.
        /// </summary>
        /// <typeparam name="T">The type of the data fields.</typeparam>
        public class KeyFrameData : IKeyFrameData
        {
            #region Constructor

            /// <summary>
            /// Constructor.
            /// </summary>
            /// <param name="parent">The parent key frame list.</param>
            /// <param name="iKeyFrame">The key frame number.</param>
            /// <param name="value1">The value.</param>
            public KeyFrameData( KeyFrameList<T> parent,
                                 int iKeyFrame,
                                 T value1 )
            {
                m_parentList = parent;
                m_iKeyFrame  = iKeyFrame;

                m_values     = new T[1];
                m_values[0]  = value1;
            }


            /// <summary>
            /// Constructor.
            /// </summary>
            /// <param name="parent">The parent key frame list.</param>
            /// <param name="iKeyFrame">The key frame number.</param>
            /// <param name="value1">The value.</param>
            /// <param name="value2">The value.</param>
            public KeyFrameData( KeyFrameList<T> parent,
                                 int iKeyFrame,
                                 T value1,
                                 T value2 )
            {
                m_parentList = parent;
                m_iKeyFrame  = iKeyFrame;

                m_values     = new T[2];
                m_values[0]  = value1;
                m_values[1]  = value2;
            }


            /// <summary>
            /// Constructor.
            /// </summary>
            /// <param name="parent">The parent key frame list.</param>
            /// <param name="iKeyFrame">The key frame number.</param>
            /// <param name="value1">The value.</param>
            /// <param name="value2">The value.</param>
            /// <param name="value3">The value.</param>
            public KeyFrameData( KeyFrameList<T> parent,
                                 int iKeyFrame,
                                 T value1,
                                 T value2,
                                 T value3 )
            {
                m_parentList = parent;
                m_iKeyFrame  = iKeyFrame;

                m_values     = new T[3];
                m_values[0]  = value1;
                m_values[1]  = value2;
                m_values[2]  = value3;
            }


            /// <summary>
            /// Constructor.
            /// </summary>
            /// <param name="parent">The parent key frame list.</param>
            /// <param name="iKeyFrame">The key frame number.</param>
            /// <param name="value1">The value.</param>
            /// <param name="value2">The value.</param>
            /// <param name="value3">The value.</param>
            /// <param name="value4">The value.</param>
            public KeyFrameData( KeyFrameList<T> parent,
                                 int iKeyFrame,
                                 T value1,
                                 T value2,
                                 T value3,
                                 T value4 )
            {
                m_parentList = parent;
                m_iKeyFrame  = iKeyFrame;

                m_values     = new T[4];
                m_values[0]  = value1;
                m_values[1]  = value2;
                m_values[2]  = value3;
                m_values[3]  = value4;
            }


            /// <summary>
            /// Copy constructor.
            /// </summary>
            /// <param name="parent">The parent key frame list.</param>
            /// <param name="src">The source key frame data to copy from.</param>
            public KeyFrameData( KeyFrameList<T> parent,
                                 KeyFrameData src )
            {
                m_parentList = parent;

                this.Copy( src );
            }

            #endregion

            #region Properties

            /// <summary>Get the parent key frame list.</summary>
            public BaseKeyFrameList Parent
            {
                get { return m_parentList; }
                set
                {
                    m_parentList = value as KeyFrameList<T>;
                }
            }


            /// <summary>Get or set the key frame number.</summary>
            public int KeyFrame
            {
                get { return m_iKeyFrame; }
                set
                {
                    if ( m_iKeyFrame==value )
                        return;

                    m_iKeyFrame = value;

                    if ( m_parentList!=null )
                        m_parentList.NotifyKeyFrameDataChanged( this );
                }
            }


            /// <summary>Get the number of the values of the key frame data.</summary>
            public int Count
            {
                get { return m_values.Length; }
            }


            /// <summary>
            /// Get or set the values for the key frame.
            /// </summary>
            /// <param name="iIndex">Index to the value to get or set.</param>
            /// <returns>The value.</returns>
            public T this[ int iIndex ]
            {
                get
                {
                    if ( iIndex<0 || iIndex>=m_values.Length )
                        throw new System.ArgumentOutOfRangeException();

                    return m_values[iIndex];
                }
                set
                {
                    if ( iIndex<0 || iIndex>=m_values.Length )
                        throw new System.ArgumentOutOfRangeException();

                    if ( m_values[iIndex].Equals(value)==true )
                        return;

                    m_values[iIndex] = value;

                    if ( m_parentList!=null )
                        m_parentList.NotifyKeyFrameDataChanged( this );
                }
            }


            /// <summary>
            /// *** This property is for the data grid to access the values. ***
            /// Get or set the value of the key frame.
            /// </summary>
            /// <remarks>
            /// The value get from this property is only for the data grid view to
            /// display on UI, the value can be converted for displaying purpose,
            /// so DO NOT access this property directly if you want to get the
            /// actual value.
            /// Eq. the value you get from this property can be a degree but a
            /// radian internally.
            /// </remarks>
            public object UIValue0
            {
                get { return GetValueForUI( 0 ); }
                set { SetValueFromUI( 0, value ); }
            }


            /// <summary>
            /// *** This property is for the data grid to access the values. ***
            /// Get or set the value of the key frame.
            /// </summary>
            /// <remarks>
            /// The value get from this property is only for the data grid view to
            /// display on UI, the value can be converted for displaying purpose,
            /// so DO NOT access this property directly if you want to get the
            /// actual value.
            /// Eq. the value you get from this property can be a degree but a
            /// radian internally.
            /// </remarks>
            public object UIValue1
            {
                get { return GetValueForUI( 1 ); }
                set { SetValueFromUI( 1, value ); }
            }


            /// <summary>
            /// *** This property is for the data grid to access the values. ***
            /// Get or set the value of the key frame.
            /// </summary>
            /// <remarks>
            /// The value get from this property is only for the data grid view to
            /// display on UI, the value can be converted for displaying purpose,
            /// so DO NOT access this property directly if you want to get the
            /// actual value.
            /// Eq. the value you get from this property can be a degree but a
            /// radian internally.
            /// </remarks>
            public object UIValue2
            {
                get { return GetValueForUI( 2 ); }
                set { SetValueFromUI( 2, value ); }
            }


            /// <summary>
            /// *** This property is for the data grid to access the values. ***
            /// Get or set the value of the key frame.
            /// </summary>
            /// <remarks>
            /// The value get from this property is only for the data grid view to
            /// display on UI, the value can be converted for displaying purpose,
            /// so DO NOT access this property directly if you want to get the
            /// actual value.
            /// Eq. the value you get from this property can be a degree but a
            /// radian internally.
            /// </remarks>
            public object UIValue3
            {
                get { return GetValueForUI( 3 ); }
                set { SetValueFromUI( 3, value ); }
            }

            #endregion

            #region Key frame values

            /// <summary>
            /// *** This method is only for the data grid to access the values. ***
            /// Get the key frame values at the given index.
            /// </summary>
            /// <param name="iIndex">The index.</param>
            /// <returns>The value at the index.</returns>
            /// <remarks>
            /// The value get from this method is only for the data grid view to
            /// display on UI, the value can be converted for displaying purpose,
            /// so DO NOT access this method directly if you want to get the
            /// actual value.
            /// Eq. the value you get from this method can be a degree but a
            /// radian internally.
            /// User should always access the value with the this[iIndex]
            /// accessor.
            /// </remarks>
            private object GetValueForUI( int iIndex )
            {
                if ( iIndex<0 || iIndex>=m_values.Length )
                    throw new System.ArgumentOutOfRangeException();

                T value = m_values[iIndex];

                object convertedValue;
                if ( this.Parent.TriggerConvertKeyFrameValueToUIEvent(this,
                                                                      value,
                                                                      out convertedValue)==true )
                {
                    return convertedValue;
                }
                else
                {
                    return value;
                }
            }


            /// <summary>
            /// Set the key frame value at the index.
            /// </summary>
            /// <param name="iIndex">The index.</param>
            /// <param name="value">The value to set.</param>
            /// <remarks>
            /// The method is only for setting values from the data grid and
            /// can be converted internally. DO NOT set value to this method
            /// directly.
            /// Eq. the value data grid set to this method can be a degree
            /// but converted to a radian internally.
            /// So if you set a radian value to this method, the value could
            /// be wrong.
            /// User should always access the value with the this[iIndex]
            /// accessor.
            /// </remarks>
            private void SetValueFromUI( int iIndex,
                                         object value )
            {
                if ( iIndex<0 || iIndex>=m_values.Length )
                    throw new System.ArgumentOutOfRangeException();

                if ( value==null )
                    throw new System.ArgumentNullException();

                T typedValue;
                if ( value is string )
                    typedValue = (T)Convert.ChangeType( value, typeof(T) );
                else if ( value is T )
                    typedValue = (T)value;
                else
                    throw new System.ArgumentException( "The value is of wrong type \"" + value.GetType().ToString() + "\"", "value" );

                object convertedValue;
                if ( this.Parent.TriggerConvertUIValueToKeyFrameEvent(this,
                                                                      typedValue,
                                                                      out convertedValue)==true )
                {
                    if ( convertedValue is T )
                        typedValue = (T)convertedValue;
                }

                if ( m_values[iIndex].Equals( typedValue )==true )
                    return;

                m_values[iIndex] = typedValue;

                if ( m_parentList!=null )
                    m_parentList.NotifyKeyFrameDataChanged( this );
            }

            #endregion

            #region Utility methods

            /// <summary>
            /// Copy the given key frame data.
            /// </summary>
            /// <param name="src">The source key frame data to copy from.</param>
            public void Copy( object src )
            {
                Copy( src as KeyFrameList<T>.KeyFrameData );
            }


            /// <summary>
            /// Copy the given key frame data.
            /// </summary>
            /// <param name="src">The source key frame data to copy from.</param>
            public void Copy( KeyFrameData src )
            {
                if ( src==null )
                    return;

                this.KeyFrame = src.KeyFrame;

                int iLength = src.m_values.Length;

                m_values = new T[iLength];
                Array.Copy( src.m_values, this.m_values, iLength );

                if ( m_parentList!=null )
                    m_parentList.NotifyKeyFrameDataChanged( this );
            }


            /// <summary>
            /// Clone this key frame data. Implemented for ICloneable interface.
            /// </summary>
            /// <returns>The copy of this key frame data.</returns>
            public object Clone()
            {
                return new KeyFrameData( null, this );
            }


            /// <summary>
            /// Compare this key frame with the given key frame data.
            /// </summary>
            /// <param name="data">The key frame data to compare to.</param>
            /// <returns>True if equal.</returns>
            public bool IsEqual( IKeyFrameData data )
            {
                KeyFrameData keyData = data as KeyFrameData;
                if ( data==null  )
                    return false;

                if ( this.KeyFrame!=keyData.KeyFrame )
                    return false;

                if ( this.m_values.Length!=keyData.m_values.Length )
                    return false;

                for ( int i=0;i<this.m_values.Length;++i )
                {
                    if ( this.m_values[i].Equals(keyData.m_values[i])==false )
                        return false;
                }

                return true;
            }

            #endregion

            #region Member variables

            private int             m_iKeyFrame  = 0;
            private T[]             m_values     = null;
            private KeyFrameList<T> m_parentList = null;

            #endregion
        }

        #endregion

        #region Constructor

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="defaultValue1">The default value.</param>
        public KeyFrameList( T defaultValue1 )
        {
            this.IsEnabled                 = false;
            this.IsLoopEnabled             = false;
            this.InterpolationType         = AnimInterpolationTypes.Linear;
            this.NumLoopFrames             = AnimTableData.DEFAULT_NUM_LOOP_FRAMES;
            this.IsRandomStartFrameEnabled = false;

            m_iNumFields  = 1;
            m_defaultData = new KeyFrameData( this, 0, defaultValue1 );

            m_fieldNames.Add( "X" );
            m_srcTargets.Add( AnimTableTargetTypes.unknown );
        }


        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="defaultValue1">The default value.</param>
        /// <param name="defaultValue2">The default value.</param>
        public KeyFrameList( T defaultValue1,
                             T defaultValue2 )
        {
            this.IsEnabled                 = false;
            this.IsLoopEnabled             = false;
            this.InterpolationType         = AnimInterpolationTypes.Linear;
            this.NumLoopFrames             = AnimTableData.DEFAULT_NUM_LOOP_FRAMES;
            this.IsRandomStartFrameEnabled = false;

            m_iNumFields  = 2;
            m_defaultData = new KeyFrameData( this,
                                              0,
                                              defaultValue1,
                                              defaultValue2 );

            m_fieldNames.Add( "X" );
            m_fieldNames.Add( "Y" );
            m_srcTargets.Add( AnimTableTargetTypes.unknown );
            m_srcTargets.Add( AnimTableTargetTypes.unknown );
        }


        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="defaultValue1">The default value.</param>
        /// <param name="defaultValue2">The default value.</param>
        /// <param name="defaultValue3">The default value.</param>
        public KeyFrameList( T defaultValue1,
                             T defaultValue2,
                             T defaultValue3 )
        {
            this.IsEnabled                 = false;
            this.IsLoopEnabled             = false;
            this.InterpolationType         = AnimInterpolationTypes.Linear;
            this.NumLoopFrames             = AnimTableData.DEFAULT_NUM_LOOP_FRAMES;
            this.IsRandomStartFrameEnabled = false;

            m_iNumFields  = 3;
            m_defaultData = new KeyFrameData( this,
                                              0,
                                              defaultValue1,
                                              defaultValue2,
                                              defaultValue3 );

            m_fieldNames.Add( "X" );
            m_fieldNames.Add( "Y" );
            m_fieldNames.Add( "Z" );
            m_srcTargets.Add( AnimTableTargetTypes.unknown );
            m_srcTargets.Add( AnimTableTargetTypes.unknown );
            m_srcTargets.Add( AnimTableTargetTypes.unknown );
        }


        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="defaultValue1">The default value.</param>
        /// <param name="defaultValue2">The default value.</param>
        /// <param name="defaultValue3">The default value.</param>
        /// <param name="defaultValue4">The default value.</param>
        public KeyFrameList( T defaultValue1,
                             T defaultValue2,
                             T defaultValue3,
                             T defaultValue4 )
        {
            this.IsEnabled                 = false;
            this.IsLoopEnabled             = false;
            this.InterpolationType         = AnimInterpolationTypes.Linear;
            this.NumLoopFrames             = AnimTableData.DEFAULT_NUM_LOOP_FRAMES;
            this.IsRandomStartFrameEnabled = false;

            m_iNumFields  = 4;
            m_defaultData = new KeyFrameData( this,
                                              0,
                                              defaultValue1,
                                              defaultValue2,
                                              defaultValue3,
                                              defaultValue4 );

            m_fieldNames.Add( "X" );
            m_fieldNames.Add( "Y" );
            m_fieldNames.Add( "Z" );
            m_fieldNames.Add( "W" );
            m_srcTargets.Add( AnimTableTargetTypes.unknown );
            m_srcTargets.Add( AnimTableTargetTypes.unknown );
            m_srcTargets.Add( AnimTableTargetTypes.unknown );
            m_srcTargets.Add( AnimTableTargetTypes.unknown );
        }


        /// <summary>
        /// Copy constructor.
        /// </summary>
        /// <param name="src">The source key frame list.</param>
        public KeyFrameList( KeyFrameList<T> src )
        {
            this.IsEnabled                 = src.IsEnabled;
            this.IsLoopEnabled             = src.IsLoopEnabled;
            this.InterpolationType         = src.InterpolationType;
            this.NumLoopFrames             = src.NumLoopFrames;
            this.IsRandomStartFrameEnabled = src.IsRandomStartFrameEnabled;

            m_iNumFields  = src.m_iNumFields;
            m_defaultData = new KeyFrameData( this, src.m_defaultData );

            // Copy the field names.
            foreach ( string name in src.m_fieldNames )
            {
                this.m_fieldNames.Add( name );
            }

            // Copy the field source data name.
            foreach ( AnimTableTargetTypes target in src.m_srcTargets )
            {
                this.m_srcTargets.Add( target );
            }

            // Copy the key frame data.
            this.InnerList.Capacity = src.InnerList.Count;
            foreach ( KeyFrameData srcData in src.InnerList )
            {
                this.InnerList.Add( new KeyFrameData(this, srcData) );
            }

            if (src.SelectedElements != null)
            {
                SelectedElements = src.SelectedElements
                    .Select(item => new KeyFrameListSelectionInfo(
                        item.KeyFrameIndex,
                        item.FieldIndices.ToArray()))
                    .ToArray();
            }
        }

        #endregion

        #region Properties

        /// <summary>
        /// Get or set the key frame data.
        /// </summary>
        /// <param name="iIndex">The index to the key frame data to access.</param>
        /// <returns>The key frame data.</returns>
        public override IKeyFrameData this[ int iIndex ]
        {
            get
            {
                return (KeyFrameData)( this.List[iIndex] );
            }
            set
            {
                this.List[iIndex] = value;
            }
        }


        /// <summary>
        /// Get the number of fields in the key frames.
        /// </summary>
        public override int FieldCount
        {
            get { return m_iNumFields; }
        }

        #endregion

        #region Implementation for IBindingList

        /// <summary>
        /// *** This method is implemented for the DataGridView, do not access it. ***
        /// Add a key frame data to the list.
        /// </summary>
        /// <param name="value">The key frame data to add.</param>
        /// <returns>The index to the added key frame data.</returns>
        public int Add( IKeyFrameData value )
        {
            if ( (value is KeyFrameList<T>.KeyFrameData)==false )
                return -1;

            return this.List.Add(value);
        }


        /// <summary>
        /// *** This method is implemented for the DataGridView, do not access it. ***
        /// Create a new key frame data.
        /// </summary>
        /// <returns>The created key frame data.</returns>
        public override object AddNew()
        {
            KeyFrameData data = new KeyFrameData( this, m_defaultData );

            data.KeyFrame = FindNextKeyFrame();

            this.List.Add( data );

            return data;
        }


        /// <summary>
        /// *** This method is implemented for the DataGridView, do not access it. ***
        /// Insert key frame data to the list.
        /// </summary>
        /// <param name="iIndex">The index to insert to.</param>
        /// <param name="value">The value to insert.</param>
        public override void Insert( int iIndex,
                                     object value )
        {
            if ( value==null )
            {
                // This block is made specialized for the key frame animation table.
                if ( iIndex<=0 )
                    throw new ArgumentOutOfRangeException();

                IKeyFrameData prevKeyFrame = this[iIndex-1];
                if ( prevKeyFrame==null )
                    throw new ArgumentOutOfRangeException();

                KeyFrameData data = new KeyFrameData( this, m_defaultData );

                data.KeyFrame = prevKeyFrame.KeyFrame + 1;

                if ( iIndex>=this.Count )
                {
                    iIndex = this.Count;
                    value  = data;
                }
                else
                {
                    value = data;
                }
            }
            else if ( (value is KeyFrameList<T>.KeyFrameData)==false )
            {
                throw new ArgumentException( "The value is of wrong type \"" + value.GetType().ToString() + "\"", "value" );
            }
            else
            {
                if ( iIndex<0 || iIndex>=this.Count )
                    iIndex = this.Count;
            }

            OnInsert( iIndex, value );

            this.InnerList.Insert( iIndex, value );

            OnInsertComplete( iIndex, value );
        }


        /// <summary>
        /// *** This method is implemented for the DataGridView, do not access it. ***
        /// Create a new key frame data.
        /// </summary>
        /// <returns>The created key frame data.</returns>
        public override void Remove( IKeyFrameData value )
        {
            if ( (value is KeyFrameList<T>.KeyFrameData)==false )
                return;

            this.List.Remove( value );
        }


        /// <summary>
        /// *** This method is implemented for the DataGridView, do not access it. ***
        /// Handle Clear event from CollectionBase.
        /// </summary>
        protected override void OnClear()
        {
            foreach ( KeyFrameData data in this.InnerList )
            {
                data.Parent = null;
            }

            base.OnClear();
        }


        /// <summary>
        /// *** This method is implemented for the DataGridView, do not access it. ***
        /// Handle ClearComplete event from CollectionBase.
        /// </summary>
        protected override void OnClearComplete()
        {
            TriggerListChangedEvent( new ListChangedEventArgs(ListChangedType.Reset,
                                                              -1) );
        }


        /// <summary>
        /// *** This method is implemented for the DataGridView, do not access it. ***
        /// Handle InsertComplete event from CollectionBase.
        /// </summary>
        /// <param name="iIndex">The index to the inserted value.</param>
        /// <param name="value">The inserted value.</param>
        protected override void OnInsertComplete( int iIndex,
                                                  object value )
        {
            KeyFrameData data = value as KeyFrameData;
            if ( data==null )
                return;

            data.Parent = this;

            TriggerListChangedEvent( new ListChangedEventArgs(ListChangedType.ItemAdded,
                                                              iIndex) );
        }


        /// <summary>
        /// *** This method is implemented for the DataGridView, do not access it. ***
        /// Handle RemoveComplete event from CollectionBase.
        /// </summary>
        /// <param name="iIndex">The index to the inserted value.</param>
        /// <param name="value">The inserted value.</param>
        protected override void OnRemoveComplete( int iIndex,
                                                  object value )
        {
            KeyFrameData data = value as KeyFrameData;
            if ( data==null )
                return;

            data.Parent = null;

            TriggerListChangedEvent( new ListChangedEventArgs(ListChangedType.ItemDeleted,
                                                              iIndex) );
        }


        /// <summary>
        /// *** This method is implemented for the DataGridView, do not access it. ***
        /// Handle SetComplete event from CollectionBase.
        /// </summary>
        /// <param name="iIndex">The index to the modified value.</param>
        /// <param name="oldValue">The old value.</param>
        /// <param name="newValue">The new value.</param>
        protected override void OnSetComplete( int iIndex,
                                               object oldValue,
                                               object newValue )
        {
            KeyFrameData oldData = oldValue as KeyFrameData;
            KeyFrameData newData = newValue as KeyFrameData;
            if ( oldData==null || newData==null )
                return;

            oldData.Parent = null;
            newData.Parent = this;

            if ( oldData.Count!=newData.Count )
            {
                TriggerListChangedEvent( new ListChangedEventArgs(ListChangedType.ItemChanged,
                                                                  iIndex) );
            }
            else
            {
                for ( int i=0;i<oldData.Count;++i )
                {
                    if ( oldData[i].Equals(newData[i])==false )
                    {
                        TriggerListChangedEvent( new ListChangedEventArgs(ListChangedType.ItemChanged,
                                                                          iIndex) );
                    }
                }
            }
        }


        /// <summary>
        /// Called by key frame data when it's changed.
        /// </summary>
        /// <param name="iIndex">The index to the modified value.</param>
        /// <param name="oldValue">The old value.</param>
        /// <param name="newValue">The new value.</param>
        internal void NotifyKeyFrameDataChanged( KeyFrameData data )
        {
            int iIndex = List.IndexOf( data );
            TriggerListChangedEvent( new ListChangedEventArgs(ListChangedType.ItemChanged,
                                                              iIndex) );
        }

        /// <summary>
        /// Called by key frame data when it's changed.
        /// </summary>
        /// <param name="e">The event argument.</param>
        internal void NotifyKeyFrameDataChanged(ListChangedEventArgs e)
        {
            TriggerListChangedEvent(e);
        }

        #endregion

        #region Add key frame

        /// <summary>
        /// Add a key frame data with the default key frame number.
        /// </summary>
        /// <returns>The index to the added key frame.</returns>
        public override int AddDefaultKeyFrame()
        {
            KeyFrameData data = new KeyFrameData( this, m_defaultData );

            data.KeyFrame = FindNextKeyFrame();

            return this.Add( data );
        }


        /// <summary>
        /// Add or insert a key frame to the list, the key frame numbers
        /// will be in the right order.
        /// </summary>
        /// <param name="iKeyFrame">The key frame number.</param>
        /// <param name="values">The key frame values.</param>
        /// <returns>The index to the added key frame.</returns>
        public int AddKeyFrameSorted(int iKeyFrame,
                                     T[] values)
        {
            if (values.Length < m_iNumFields)
                return -1;

            KeyFrameData data = null;
            if (m_iNumFields == 1)
            {
                data = new KeyFrameData(this,
                                        iKeyFrame,
                                        values[0]);
            }
            else if (m_iNumFields == 2)
            {
                data = new KeyFrameData(this,
                                        iKeyFrame,
                                        values[0],
                                        values[1]);
            }
            else if (m_iNumFields == 3)
            {
                data = new KeyFrameData(this,
                                        iKeyFrame,
                                        values[0],
                                        values[1],
                                        values[2]);
            }
            else
            {
                data = new KeyFrameData(this,
                                        iKeyFrame,
                                        values[0],
                                        values[1],
                                        values[2],
                                        values[3]);
            }

            int iIndex = FindIndexToInsertSortedly(iKeyFrame);
            if ((iIndex < 0) || (iIndex >= this.List.Count))
            {
                return this.Add(data);
            }
            else
            {
                this.Insert(iIndex, data);
                return iIndex;
            }
        }


        /// <summary>
        /// Add a key frame to the list.
        /// </summary>
        /// <param name="iKeyFrame">The key frame number.</param>
        /// <param name="src">The source key frame.</param>
        /// <returns>The index to the added key frame.</returns>
        public int AddKeyFrame( int iKeyFrame,
                                KeyFrameData src )
        {
            if ( src.Count!=m_iNumFields )
                return -1;

            if ( m_iNumFields==1 )
                return AddKeyFrame( iKeyFrame, src[0] );
            else if ( m_iNumFields==2 )
                return AddKeyFrame( iKeyFrame, src[0], src[1] );
            else if ( m_iNumFields==3 )
                return AddKeyFrame( iKeyFrame, src[0], src[1], src[2] );
            else
                return AddKeyFrame( iKeyFrame, src[0], src[1], src[2], src[3] );
        }


        /// <summary>
        /// Add a key frame to the list.
        /// </summary>
        /// <param name="iKeyFrame">The key frame number.</param>
        /// <param name="value1">The value.</param>
        /// <returns>The index to the added key frame.</returns>
        public int AddKeyFrame( int iKeyFrame,
                                T value1 )
        {
            if ( m_iNumFields!=1 )
                return -1;

            return this.Add( new KeyFrameData(this,
                                              iKeyFrame,
                                              value1) );
        }


        /// <summary>
        /// Add a key frame to the list.
        /// </summary>
        /// <param name="iKeyFrame">The key frame number.</param>
        /// <param name="value1">The value.</param>
        /// <param name="value2">The value.</param>
        /// <returns>The index to the added key frame.</returns>
        public int AddKeyFrame( int iKeyFrame,
                                T value1,
                                T value2 )
        {
            if ( m_iNumFields==1 )
                return AddKeyFrame( iKeyFrame, value1 );
            else if ( m_iNumFields==2 )
                return this.Add( new KeyFrameData( this,
                                                  iKeyFrame,
                                                  value1,
                                                  value2 ) );
            else
                return -1;
        }


        /// <summary>
        /// Add a key frame to the list.
        /// </summary>
        /// <param name="iKeyFrame">The key frame number.</param>
        /// <param name="value1">The value.</param>
        /// <param name="value2">The value.</param>
        /// <param name="value3">The value.</param>
        /// <param name="value4">The value.</param>
        /// <returns>The index to the added key frame.</returns>
        public int AddKeyFrame( int iKeyFrame,
                                T value1,
                                T value2,
                                T value3 )
        {
            if ( m_iNumFields==1 )
                return AddKeyFrame( iKeyFrame, value1 );
            else if ( m_iNumFields==2 )
                return AddKeyFrame( iKeyFrame, value1, value2 );
            else if ( m_iNumFields==3 )
                return this.Add( new KeyFrameData(this,
                                                  iKeyFrame,
                                                  value1,
                                                  value2,
                                                  value3) );
            else
                return -1;
        }


        /// <summary>
        /// Add a key frame to the list.
        /// </summary>
        /// <param name="iKeyFrame">The key frame number.</param>
        /// <param name="value1">The value.</param>
        /// <param name="value2">The value.</param>
        /// <param name="value3">The value.</param>
        /// <param name="value4">The value.</param>
        /// <returns>The index to the added key frame.</returns>
        public int AddKeyFrame( int iKeyFrame,
                                T value1,
                                T value2,
                                T value3,
                                T value4 )
        {
            if ( m_iNumFields==1 )
                return AddKeyFrame( iKeyFrame, value1 );
            else if ( m_iNumFields==2 )
                return AddKeyFrame( iKeyFrame, value1, value2 );
            else if ( m_iNumFields==3 )
                return AddKeyFrame( iKeyFrame, value1, value2, value3 );
            else
                return this.Add( new KeyFrameData(this,
                                                  iKeyFrame,
                                                  value1,
                                                  value2,
                                                  value3,
                                                  value4) );
        }

        #endregion

        #region Field information

        /// <summary>
        /// Set the display name on the data grid for the specified key frame value field.
        /// </summary>
        /// <param name="iFieldIndex">The index to the key frame value field.</param>
        /// <param name="name">The display name.</param>
        /// <param name="target">The applied target of the animation table field.</param>
        public override void SetupField( int iFieldIndex,
                                         string name,
                                         AnimTableTargetTypes target )
        {
            if ( iFieldIndex<0 || iFieldIndex>=m_iNumFields )
                return;

            m_fieldNames[iFieldIndex] = name;
            m_srcTargets[iFieldIndex] = target;
        }


        /// <summary>
        /// Get the display name on the data grid for the specified key frame value field.
        /// </summary>
        /// <param name="iFieldIndex">The index to the key frame value field.</param>
        public override string GetFieldName( int iFieldIndex )
        {
            if ( iFieldIndex<0 || iFieldIndex>=m_iNumFields )
                return string.Empty;

            return m_fieldNames[iFieldIndex];
        }


        /// <summary>
        /// Get the source data name of the specified key frame value field.
        /// </summary>
        /// <param name="iFieldIndex">The index to the key frame value field.</param>
        /// <returns>The applied target of the animation table field.</returns>
        public override AnimTableTargetTypes GetFieldTarget( int iFieldIndex )
        {
            if ( iFieldIndex<0 || iFieldIndex>=m_iNumFields )
                return AnimTableTargetTypes.unknown;

            return m_srcTargets[iFieldIndex];
        }

        #endregion

        #region Utility methods

        /// <summary>
        /// Determine if a new key frame can be inserted after the given row.
        ///
        /// For the given row index smaller than zero or greater than the last row,
        /// the method will process it like adding a new key frame in the end of the
        /// list.
        /// </summary>
        /// <param name="iRowIndex">The index of the row.</param>
        /// <param name="iMaxFrameNumber">The maximum key frame number.</param>
        /// <returns>True if insert is possible.</returns>
        public override InsertKeyFrameValidateResult CanInsertKeyFrameAfter( int iRowIndex,
                                                                             int iMaxFrameNumber )
        {
            if ( this.List.Count<=0 )
                return InsertKeyFrameValidateResult.Success;

            if ( this.List.Count>=this.MaxNumKeyFrames )
                return InsertKeyFrameValidateResult.ExceedMaxKeyFrameCount;

            IKeyFrameData keyFrame = null;
            if ( iRowIndex<0 || iRowIndex>=(this.List.Count-1) )
                keyFrame = this[this.List.Count-1];
            else
                keyFrame = this[iRowIndex];

            if ( keyFrame.KeyFrame>=iMaxFrameNumber )
                return InsertKeyFrameValidateResult.ExceedMaxKeyFrameNumber;

            if ( iRowIndex<0 || iRowIndex>=(this.List.Count-1) )
                return InsertKeyFrameValidateResult.Success;

            if ( this[iRowIndex+1].KeyFrame>(this[iRowIndex].KeyFrame + 1) )
                return InsertKeyFrameValidateResult.Success;

            return InsertKeyFrameValidateResult.InvalidKeyFrameNumber;
        }


        /// <summary>
        /// Call this method for finding out the key frame number to use
        /// for creating a new key frame data.
        /// For example, if there are already key frame data for frame 1, 5, 8,
        /// then this method returns 9 ( next frame of the latest frame number ).
        /// </summary>
        /// <returns>The next key frame number.</returns>
        private int FindNextKeyFrame()
        {
            int iCount = this.List.Count;
            if ( iCount<=0 )
                return 0;

            for ( int i=iCount-1;i>=0;--i )
            {
                IKeyFrameData data = this[iCount-1];
                if ( data!=null )
                    return data.KeyFrame + 1;
            }

            return 0;
        }


        /// <summary>
        /// Find the index to insert the key frame with the key frame number
        /// in the correctly order.
        /// </summary>
        /// <param name="iKeyFrameNumber">The key frame number to insert.</param>
        /// <returns>The index.</returns>
        public int FindIndexToInsertSortedly(int iKeyFrameNumber)
        {
            // Do trivial check
            int iSize = this.List.Count;
            if ((iSize==0) ||
                (((IKeyFrameData)this.List[iSize - 1]).KeyFrame < iKeyFrameNumber))
            {
                return iSize;
            }
            else if (((IKeyFrameData)this.List[0]).KeyFrame > iKeyFrameNumber)
            {
                return 0;
            }

            int iLeft  = 0;
            int iRight = iSize - 1;
            int iMid   = (iLeft + iRight) / 2;

            // Search the index to insert the key frame with binary search.
            while (iLeft < iRight)
            {
                // Compare updater versions
                var key = this.List[iMid] as IKeyFrameData;
                if (key.KeyFrame > iKeyFrameNumber)
                {
                    iRight = iMid;
                }
                else if (key.KeyFrame < iKeyFrameNumber)
                {
                    iLeft = iMid + 1;
                }
                else
                {
                    return iMid + 1;
                }

                // Compute middle index
                iMid = (iLeft + iRight) / 2;
            }

            return iLeft;
        }


        /// <summary>
        /// Check if the given object has same values as this key frame list.
        /// </summary>
        /// <param name="obj">The object to compare with.</param>
        /// <returns>True if equal.</returns>
        public override bool Equals( object obj )
        {
            KeyFrameList<T> another = obj as KeyFrameList<T>;
            if ( another==null )
                return false;

            return this.IsEqual( another );
        }


        /// <summary>
        /// Get hash code.
        /// Use default method for the hash code.
        /// The method is overridden just to shut the compiler up.
        /// </summary>
        /// <returns>The hash code.</returns>
        public override int GetHashCode()
        {
            return base.GetHashCode();
        }


        /// <summary>
        /// Clone this instance. Implemented for the ICloneable interface.
        /// </summary>
        /// <returns>The copy of this instance.</returns>
        public override object Clone()
        {
            return new KeyFrameList<T>( this );
        }

        #endregion

        #region Member variables

        private int             m_iNumFields   = 1;
        private List<string>    m_fieldNames   = new List<string>(4);

        private List<AnimTableTargetTypes> m_srcTargets = new List<AnimTableTargetTypes>( 4 );

        private KeyFrameData    m_defaultData  = null;

        #endregion
    }

    #endregion
}
