﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;

namespace App.Data
{
    /// <summary>
    /// Interface of the data model updaters.
    /// </summary>
    public interface IDataModelUpdater
    {
        /// <summary>Get the data model version this updater converts from.</summary>
        Version PreviousVersion { get; }


        /// <summary>Get the data model version this updater converts to.</summary>
        Version CurrentVersion { get; }


        /// <summary>
        /// Check if the updater can deserialize the data model with
        /// the given binary version.
        /// </summary>
        /// <param name="binaryVersion">The binary version.</param>
        /// <returns>True if the updater can deserialize the file.</returns>
        bool CanDeserialize( Version binaryVersion );


        /// <summary>
        /// Check if the data model can be updated with this updater.
        /// </summary>
        /// <param name="dataModel">The data model to be updated.</param>
        /// <returns>True if the data model can be updated by this updater.</returns>
        bool CanUpdate( object dataModel );


        /// <summary>
        /// Deserialize the data in the given memory stream and return the
        /// deserialized data model.
        /// </summary>
        /// <param name="binaryVersion">The binary version of the data.</param>
        /// <param name="stream">The memory stream contains the data to be deserialized.</param>
        /// <returns>The deserialized data model.</returns>
        object Deserialize( Version binaryVersion,
                            Stream stream );


        /// <summary>
        /// Update the given data model to a later version.
        /// </summary>
        /// <param name="src">The source data model.</param>
        /// <returns>The updated data model.</returns>
        object Update( object src );
    }
}
