﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Collections.Generic;

using App.IO;

using NWCore.DataModel;
using NWCore.Serializer;

namespace App.Data
{
    /// <summary>
    /// パーティクル・セット用ドキュメント
    /// </summary>
    public sealed class PreviewModelDocument : BaseProjectPanelDocument,
                                               IPreviewModelDocument,
                                               IDataModelProxyOwner
    {
        #region Animation list update event handelr

        public delegate void AnimationListUpdateEventHandler( PreviewModelDocument doc );

        #endregion

        #region Memeber Variables

        private readonly PreviewDocument  m_previewDoc = null;    // parent/owner
        private readonly PreviewModelData m_modelData  = null;
        private          int              m_iIndex     = -1;
        private          string           m_modelName  = string.Empty;
        private          string           m_modelFile  = string.Empty;

        private NWCore.Utility.Path    m_modelNameDataPath      = null;
        private NWCore.Utility.Path    m_modelFnmDataPath       = null;
        private NWCore.Utility.Path    m_animFolderDataPath     = null;
        private NWCore.Utility.Path    m_animNameDataPath       = null;
        private NWCore.Utility.Path    m_animStartFrameDataPath = null;
        private NWCore.Utility.Path    m_animEndFrameDataPath   = null;

        private DataModelProxy         m_dataModelProxy = null;
        private List<UserFunctionInfo> m_userFuncList   = new List<UserFunctionInfo>();

        private Dictionary<uint, string> m_animationMap = new Dictionary<uint, string>();

        #endregion

        #region Properties

        /// <summary>
        /// 名前
        /// </summary>
        public override string Name
        {
            get
            {
                if (m_modelData == null)
                {
                    // モデル未設定
                    return res.Strings.PREVIEW_MODEL_DEFAULT_NAME;
                }
                else
                {
                    // ファイルパスが設定されていないときは未設定
                    if (String.IsNullOrEmpty(m_modelData.ModelFilePath) == true)
                    {
                        // モデル未設定
                        return res.Strings.PREVIEW_MODEL_DEFAULT_NAME;
                    }
                    // ファイルパスが設定されていて、モデル名がないときは未取得
                    else if (String.IsNullOrEmpty(m_modelData.ModelName) == true
                        && String.IsNullOrEmpty(m_modelData.ModelFilePath) == false)
                    {
                        // モデル未取得
                        return Path.GetFileName(m_modelData.ModelFilePath);
                    }
                    // 表示名と違っていて、ボーン数＝０ならば未取得
                    else if (String.IsNullOrEmpty(m_modelData.ModelName) == false
                        && String.IsNullOrEmpty(m_modelData.ModelFilePath) == false
                        && m_modelName.Equals(m_modelData.ModelName) == false
                        && m_modelData.BoneCount == 0)
                    {
                        // モデル未取得
                        return Path.GetFileName(m_modelData.ModelFilePath);
                    }

                }

                return m_modelName;
            }
            set
            {
            }
        }

        /// <summary>
        /// ビューア表示On/Off
        /// </summary>
        public bool IsShow
        {
            get { return this.PreviewModelData.DisplayModel; }
            set { this.PreviewModelData.DisplayModel = value; }
        }

        /// <summary>
        /// Get the flag indicating whether the model name is received.
        /// </summary>
        public bool IsModelNameReceived
        {
            get
            {
                if ( m_modelData!=null )
                {
                    // ファイルパスが設定されていて、モデル名がないときは未取得
                    if ( String.IsNullOrEmpty(m_modelData.ModelName)==true &&
                         String.IsNullOrEmpty(m_modelData.ModelFilePath)==false )
                    {
                        // モデル未取得
                        return false;
                    }
                    // 表示名と違っていて、ボーン数＝０ならば未取得
                    else if ( String.IsNullOrEmpty(m_modelData.ModelName)==false &&
                              String.IsNullOrEmpty(m_modelData.ModelFilePath)==false &&
                              m_modelName.Equals(m_modelData.ModelName)==false &&
                              m_modelData.BoneCount==0 )
                    {
                        // モデル未取得
                        return false;
                    }
                }

                return true;
            }
        }

        /// <summary>
        /// オーナードキュメント。
        /// </summary>
        public override IDocument OwnerDocument
        {
            get { return this.m_previewDoc; }
        }

        /// <summary>
        /// Preview document that this child belongs to
        /// </summary>
        public IPreviewDocument PreviewDocument
        {
            get { return this.m_previewDoc; }
        }

        /// <summary>
        /// データモデル：プレビュー設定
        /// </summary>
        public PreviewModelData PreviewModelData
        {
            get { return this.m_modelData; }
        }

        /// <summary>
        /// data source path without data model
        /// </summary>
        public override string RelativeDataScrPath
        {
            get
            {
                return "PreviewModelData";
            }
        }

        /// <summary>
        /// Document index.
        /// </summary>
        public int DocumentIndex
        {
            get
            {
                return m_iIndex;
            }
        }

        /// <summary>
        /// path of the data source
        /// </summary>
        public override string DataScrPath
        {
            get
            {
                // Because the path nodes use ID==0 to determine if ID is being used,
                // so we need to kinda hack it by adding 1 to the index.
                return this.PreviewDocument.DataScrPath +
                       ".PreviewModelDocuments@" +
                       (m_iIndex+1).ToString( "X" );
            }
        }


        /// <summary>
        /// Get the data model proxy.
        /// </summary>
        public DataModelProxy DataModelProxy
        {
            get { return m_dataModelProxy; }
        }

        /// <summary>
        /// Internal flag indicating if the modify flag should be notified to parent document.
        /// </summary>
        protected override bool ShouldNotifyParentDocModified
        {
            get { return false; }
        }

        /// <summary>
        /// Get the dictionary for the animation file paths and their ID.
        /// </summary>
        public Dictionary<uint, string> AnimationFileMap
        {
            get { return m_animationMap; }
        }

        #endregion

        #region Constructors

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public PreviewModelDocument( PreviewDocument parentDoc,
                                     PreviewModelData modelData,
                                     int iIndex ) :
            base( res.Strings.PREVIEW_MODEL_DEFAULT_NAME )
        {
            this.m_previewDoc = parentDoc;
            this.m_modelData  = modelData;
            this.m_iIndex     = iIndex;

            UpdateModelName();
            InitDataModelProxy();

            // Compose the source data paths.
            string modelNameDataPath =
                this.DataScrPath + "." + this.RelativeDataScrPath + ".ModelName";
            string modelFnmDataPath =
                this.DataScrPath + "." + this.RelativeDataScrPath + ".ModelFilePath";
            string animFolderDataPath =
                this.DataScrPath + "." + this.RelativeDataScrPath + ".AnimationFolder";
            string animNameDataPath =
                this.DataScrPath + "." + this.RelativeDataScrPath + ".AnimationName";
            string animStartFrameDataPath =
                this.DataScrPath + "." + this.RelativeDataScrPath + ".AnimFileStartFrame";
            string animEndFrameDataPath =
                this.DataScrPath + "." + this.RelativeDataScrPath + ".AnimFileEndFrame";

            m_modelNameDataPath      = new NWCore.Utility.Path(modelNameDataPath);
            m_modelFnmDataPath       = new NWCore.Utility.Path(modelFnmDataPath);
            m_animFolderDataPath     = new NWCore.Utility.Path(animFolderDataPath);
            m_animNameDataPath       = new NWCore.Utility.Path(animNameDataPath);
            m_animStartFrameDataPath = new NWCore.Utility.Path(animStartFrameDataPath);
            m_animEndFrameDataPath   = new NWCore.Utility.Path(animEndFrameDataPath);


            TheApp.ConfigChanged     += OnAppConfigChanged;
            Document.PropertyChanged += property_Changed;
        }


        #endregion

        #region オブジェクト

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override GuiObjectID ObjectID
        {
            get { return GuiObjectID.PreviewModel; }
        }

        #endregion

        #region User functions

        /// <summary>
        /// Get the number of user functions.
        /// </summary>
        /// <returns>The number of user functions.</returns>
        public int GetNumUserFunctions()
        {
            return m_userFuncList.Count;
        }


        /// <summary>
        /// Get the user function information with the given index.
        /// </summary>
        /// <param name="iIndex">The index to the user function info.</param>
        /// <returns>The user function info.</returns>
        public UserFunctionInfo GetUserFunctionInfo( int iIndex )
        {
            if ( iIndex<0 || iIndex>=m_userFuncList.Count )
                return null;

            return m_userFuncList[iIndex];
        }


        /// <summary>
        /// Enable / disable the specified user function.
        /// </summary>
        /// <param name="iUserFuncID">User function ID.</param>
        /// <param name="bEnable">True to enable the user function.</param>
        public bool EnableUserFunction( uint iUserFuncID,
                                        bool bEnable )
        {
            var hasChanged = false;

            foreach ( UserFunctionInfo info in m_userFuncList )
            {
                if ( info!=null &&
                     info.UserFuncID==iUserFuncID )
                {
                    if ( info.Enabled!=bEnable )
                    {
                        info.Enabled = bEnable;
                        if ( m_dataModelProxy!=null )
                        {
                            DataModelInfo dataModel =
                                m_dataModelProxy.GetDataModel( info.UserFuncName );

                            if (dataModel != null)
                            {
                                dataModel.Enabled = bEnable;
                                hasChanged = true;
                            }
                        }

                        // Trigger event.
                        if ( UserFunctionEnableChanged!=null )
                        {
                            UserFunctionEnableChanged( this, System.EventArgs.Empty );
                        }
                    }
                    return hasChanged;
                }
            }

            return false;
        }


        /// <summary>
        /// Check if the specified user function is enabled.
        /// </summary>
        /// <param name="iUserFuncID">User function ID.</param>
        /// <returns>True if the user function is enabled.</returns>
        public bool IsDataModelEnabled( uint iUserFuncID )
        {
            foreach ( UserFunctionInfo info in m_userFuncList )
            {
                if ( info!=null &&
                     info.UserFuncID==iUserFuncID )
                {
                    return info.Enabled;
                }
            }

            return false;
        }

        #endregion

        #region Data model proxy

        /// <summary>
        /// Initialize the data model proxy.
        /// </summary>
        private void InitDataModelProxy()
        {
            ////UIManager UIMgr = TheApp.UIManager;
            ////string    strPageTitle;
            ////string    strUserFuncName;
            ////uint      iUserFuncID;

            ////int iNumUserFuncs =
            ////    UIMgr.GetNumUserDefinedPropertyPages( PropertyPanelID.PreviewModelPropertyPanel );
            ////for ( int i=0;i<iNumUserFuncs;++i )
            ////{
            ////    UIMgr.GetUserFunctionInfo( PropertyPanelID.PreviewModelPropertyPanel,
            ////                               i,
            ////                               out strPageTitle,
            ////                               out strUserFuncName,
            ////                               out iUserFuncID );

            ////    UserFunctionInfo info = new UserFunctionInfo( strPageTitle,
            ////                                                  strUserFuncName,
            ////                                                  iUserFuncID );

            ////    m_userFuncList.Add( info );
            ////}

            // Create and register data model proxy
            m_dataModelProxy = new DataModelProxy( this );
        }

        #endregion

        #region Utilities

        /// <summary>
        /// プロパティ更新時
        /// </summary>
        private void property_Changed(object sender, DocumentPropertyChangedEventArgs args)
        {
            // 名前の更新
            UpdateModelName();
        }

        /// <summary>
        /// Utility method for updating the model name from model data.
        /// </summary>
        private void UpdateModelName()
        {
            if ( m_modelData==null )
                return;

            bool bFileNameChanged = false;

            // Has the model file already been set?
            if ( m_modelFile!=m_modelData.ModelFilePath )
            {
                m_modelFile = string.Copy( m_modelData.ModelFilePath );
                bFileNameChanged = true;
            }

            // Have we received the model name?
            if ( string.IsNullOrEmpty(m_modelData.ModelName)==true )
            {
                // Is the model file name changed?
                if ( m_modelName!=res.Strings.PREVIEW_MODEL_DEFAULT_NAME )
                    bFileNameChanged = true;

                // Set the model name to default.
                m_modelName = res.Strings.PREVIEW_MODEL_DEFAULT_NAME;
            }
            else
            {
                // Get file name from the full path.
                string fileName = System.IO.Path.GetFileName(m_modelData.ModelName);

                // Is the model file name changed?
                if ( fileName!=m_modelName )
                    bFileNameChanged = true;

                // Remember the file name.
                m_modelName = fileName;
            }

            // Update our project tree if the file name has changed.
            if ( bFileNameChanged==true &&
                 m_previewDoc!=null &&
                 m_previewDoc.Project!=null )
            {
                m_previewDoc.Project.BuildTreeNodes();
            }
        }

        #endregion

        #region Load document

        /// <summary>
        /// Load the contents from the deserializer.
        /// </summary>
        /// <param name="deserializer">The deserializer.</param>
        /// <returns>True on success.</returns>
        public bool LoadFromDeserializer( EnvConfigDataXml deserializer )
        {
            // First disable all the user functions
            foreach ( UserFunctionInfo info in m_userFuncList )
            {
                if ( info.Enabled==true )
                {
                    info.Enabled = false;
                    DataModelInfo dataModel =
                        m_dataModelProxy.GetDataModel( info.UserFuncName );

                    if ( dataModel!=null )
                        dataModel.Enabled = false;
                }
            }

            PreviewModelData data =
                deserializer.ConfigData.PreviewConfigData.GetModelData( m_iIndex );
            if ( data==null )
                return false;

            // Load the user functions.
            foreach ( UserFunctionXML userFunc in data.UserFunctionList )
            {
                userFunc.PostDeserialize();

                this.DataModelProxy.LoadUserFunction( userFunc );

                // Enable the loaded user function
                foreach ( UserFunctionInfo info in m_userFuncList )
                {
                    if ( info.UserFuncID==userFunc.UserFuncID )
                    {
                        info.Enabled = true;
                        DataModelInfo dataModel =
                            m_dataModelProxy.GetDataModel( info.UserFuncName );

                        if ( dataModel!=null )
                            dataModel.Enabled = true;
                    }
                }
            }

            // Trigger event to notify the user function has been enabled/disabled.
            if ( UserFunctionEnableChanged!=null )
            {
                UserFunctionEnableChanged( this, System.EventArgs.Empty );
            }

            return true;
        }

        #endregion

        #region Events

        /// <summary>
        /// Triggers when enable/disable any of the user functions.
        /// </summary>
        public event EventHandler UserFunctionEnableChanged;

        /// <summary>
        /// Static event triggers when the animation list has changed.
        /// </summary>
        public static event AnimationListUpdateEventHandler AnimationListUpdated;

        #endregion

        #region Event handlers

        /// <summary>
        /// Handle ConfigChanged event from the application.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnAppConfigChanged( object sender,
                                         EventArgs e )
        {
            RegisterModelFileMonitors();
            RegisterAnimFolderMonitors();
        }


        /// <summary>
        /// Notify this document that it was added to its parent.
        /// </summary>
        public override void NotifyAddedToParent()
        {
            base.NotifyAddedToParent();

            UpdateAnimation();

            RegisterModelFileMonitors();
            RegisterAnimFolderMonitors();

            DocumentManager.RegisterDocumentPath( this, this.DataScrPath );
            DocumentManager.RegisterDataModelProxy( m_dataModelProxy );
        }


        /// <summary>
        /// Notify this document that it was removed from its parent.
        /// </summary>
        public override void NotifyRemovedFromParent()
        {
            DocumentManager.UnregisterDataModelProxy( m_dataModelProxy );
            DocumentManager.UnregisterDocumentPath( this, this.DataScrPath );

            base.NotifyRemovedFromParent();
        }

        #endregion

        #region File monitors

        /// <summary>
        /// Register file system watcher for the model file.
        /// </summary>
        private void RegisterModelFileMonitors()
        {

            string primitiveFilePath = this.PreviewModelData.ModelFilePath;
            if ( string.IsNullOrEmpty(primitiveFilePath)==true ||
                 Path.IsPathRooted(primitiveFilePath)==false )
            {
                return;
            }

            string dirPath = Path.GetDirectoryName( primitiveFilePath );
            if ( string.IsNullOrEmpty(dirPath)==true ||
                 Path.IsPathRooted(dirPath)==false ||
                 Directory.Exists(dirPath)==false )
            {
                return;
            }
        }

        /// <summary>
        /// Register file system watcher for the animation folder.
        /// </summary>
        private void RegisterAnimFolderMonitors()
        {

            string animationFolderPath = this.PreviewModelData.AnimationFolder;
            if ( string.IsNullOrEmpty(animationFolderPath)==true ||
                 Path.IsPathRooted(animationFolderPath)==false ||
                 Directory.Exists(animationFolderPath)==false )
            {
                return;
            }
        }


        /// <summary>
        /// Reload model file.
        /// </summary>
        private void ReloadModel()
        {
            // Source data path to EmitterData.BillboardPrimitiveFileSource
            string srcPath = this.DataScrPath + "." +
                             this.RelativeDataScrPath +
                             ".ModelFilePath";

            this.PreviewModelData.ModelName = string.Empty;
        }


        /// <summary>
        /// Handle event from the file system watcher.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event argument.</param>
        private void OnModelFileChanged( object sender,
                                         System.IO.FileSystemEventArgs e )
        {
            if ( Config.Data.Option.AutoPrimitiveReload==false )
                return;

            string primitiveFilePath = this.PreviewModelData.ModelFilePath;
            if ( e.FullPath!=primitiveFilePath )
                return;

            //MainFrame.SynchronizationContext.Post( delegate { this.ReloadModel(); } );
        }


        /// <summary>
        /// Handle Rename event from the file system watcher.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event argument.</param>
        private void OnModelFileRenamed( object sender,
                                         System.IO.RenamedEventArgs e )
        {
            if ( Config.Data.Option.AutoPrimitiveReload==false )
                return;

            string primitiveFilePath = this.PreviewModelData.ModelFilePath;
            if ( e.FullPath!=primitiveFilePath && e.OldFullPath!=primitiveFilePath )
                return;

            ////MainFrame.SynchronizationContext.Post( delegate { this.ReloadModel(); } );
        }


        /// <summary>
        /// Called when animation folder contents is changed,
        /// check if the selected animation file still exists or if it's been modified.
        /// </summary>
        /// <param name="overwrittenFolderPath">The animation folder path to use.</param>
        public void UpdateAnimation( string overwrittenFolderPath = null )
        {
            // Get the animation folder path.
            string folderPath = this.PreviewModelData.AnimationFolder;
            if ( overwrittenFolderPath!=null )
                folderPath = overwrittenFolderPath;

            // Update the animation list.
            if ( string.IsNullOrEmpty(folderPath)==true ||
                 Directory.Exists(folderPath)==false )
            {
                m_animationMap.Clear();
                m_animationMap.Add( 0, res.Strings.UI_PREVMODEL_ANIM_DO_NOT_USE_ANIMATION );
            }
            else
            {
                m_animationMap.Clear();
                m_animationMap.Add( 0, res.Strings.UI_PREVMODEL_ANIM_DO_NOT_USE_ANIMATION );

                string filter = "*" + DocumentConstants.GetNativeAnimationFileExt();

                DirectoryInfo rootDirInfo = new DirectoryInfo( folderPath );
                foreach ( DirectoryInfo dirInfo in DocumentConstants.EnumerateSubFolders(rootDirInfo) )
                {
                    foreach ( FileInfo fileInfo in dirInfo.EnumerateFiles(filter) )
                    {
                        uint iPathCRC =
                            NWCore.NWKernel.CRC32Helper.ComputeCRC32Str( fileInfo.FullName.ToLower() );
                        m_animationMap.Add( iPathCRC, fileInfo.FullName );
                    }
                }
            }

            // Notify that the animation list has changed.
            if ( PreviewModelDocument.AnimationListUpdated!=null )
            {
                PreviewModelDocument.AnimationListUpdated( this );
            }
        }


        /// <summary>
        /// Handle event from the file system watcher.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event argument.</param>
        private void OnAnimFolderChanged( object sender,
                                          System.IO.FileSystemEventArgs e )
        {
            if ( Config.Data.Option.AutoPrimitiveReload==false )
                return;

            ////MainFrame.SynchronizationContext.Post( delegate { this.UpdateAnimation(); } );
        }


        /// <summary>
        /// Handle Rename event from the file system watcher.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event argument.</param>
        private void OnAnimFolderRenamed( object sender,
                                          System.IO.RenamedEventArgs e )
        {
            if ( Config.Data.Option.AutoPrimitiveReload==false )
                return;

            ////MainFrame.SynchronizationContext.Post( delegate { this.UpdateAnimation(); } );
        }

        #endregion
    }
}
