﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Windows.Forms;
using App.IO;
using NintendoWare.ToolDevelopmentKit.Xml;
using NWCore.DataModel;
using NWCore.Serializer;

namespace App.Data
{
    /// <summary>
    /// エフェクト用プロジェクト・ドキュメントです
    /// </summary>
    public class EffectProjectDocument : BaseProjectPanelDocument, IProjectDocument
    {
        #region Member Variables

        private readonly List<IEmitterSetDocument>
            m_emitterSetList = new List<IEmitterSetDocument>();
        private readonly List<TreeNode>
            m_treeNode = new List<TreeNode>();
        private PreviewDocument m_previewDocument;
        private readonly InitSettingsDocument m_initDocument;

        private readonly EnvConfigData m_envConfigData = new EnvConfigData();
        private IDocument m_selectedDocument;
        private WorkSpaceDocument m_workSpaceDoc = new WorkSpaceDocument();

        private readonly List<IRequireDocument> m_requireDocs = new List<IRequireDocument>();
        private readonly Queue<RequireDocData> m_queueRequire = new Queue<RequireDocData>();

        #endregion

        #region Constructors

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public EffectProjectDocument(string name)
            : base(name)
        {
            m_envConfigData.Initialize();

            DocumentManager.RegisterDocumentPath( this, this.DataScrPath );

            this.EmitterSetListChanged += emitterSet_ListChanged;

            this.EnableBuildTreeNodes = false;
            m_previewDocument = new PreviewDocument(this, m_envConfigData.PreviewConfigData);
            m_initDocument = new InitSettingsDocument(this);
            this.EnableBuildTreeNodes = true;
        }

        #endregion

        #region Properties

        /// <summary>
        /// オーナードキュメント
        /// </summary>
        public override IDocument OwnerDocument
        {
            get { return null; }
        }

        /// <summary>
        /// 編集状態かどうか
        /// </summary>
        public override bool Modified
        {
            get
            {
                // Effect project is going to be removed, and the command
                // manager no longer has effect to projects.
                // So for now we just always return true.
                return true;
            }
        }

        /// <summary>
        /// オーバーライド
        /// </summary>
        public override string FileExt
        {
            get { return DocumentConstants.FEnv; }
        }

        /// <summary>
        /// ファイルフィルタ。
        /// </summary>
        public override string FileFilter
        {
            get { return res.Strings.IO_FILEFILTER_CAFE_FENV; }
        }

        /// <summary>
        /// path of the data source
        /// </summary>
        public override string DataScrPath
        {
            get
            {
                return "Project_uid" + this.DocumentID.ToString();
            }
        }


        /// <summary>
        /// Overridden data path for display, project document is not visible for
        /// user, so just return an empty string.
        /// </summary>
        public override string DisplayDataPath
        {
            get
            {
                return string.Empty;
            }
        }


        /// <summary>
        /// data source path without data model
        /// </summary>
        public override string RelativeDataScrPath
        {
            get
            {
                return "Project_uid" + this.DocumentID.ToString();
            }
        }


        /// <summary>
        /// Work space document.
        /// </summary>
        public WorkSpaceDocument WorkSpaceDocument
        {
            get { return m_workSpaceDoc; }
        }


        /// <summary>
        /// Preview document.
        /// </summary>
        public IPreviewDocument PreviewDocument
        {
            get { return m_previewDocument; }
        }

        /// <summary>
        /// Initial setttings document.
        /// </summary>
        public InitSettingsDocument InitSettingsDocument
        {
            get { return m_initDocument; }
        }

        /// <summary>
        /// エミッタセット数
        /// </summary>
        public override int Count
        {
            get { return this.m_emitterSetList.Count; }
        }

        /// <summary>
        /// Emitter set count.
        /// </summary>
        public int NumEmitterSetDocuments
        {
            get { return this.m_emitterSetList.Count; }
        }

        /// <summary>
        /// 保持しているエフェクトドキュメントのリストです
        /// </summary>
        public IEnumerable<IEmitterSetDocument> EmitterSetDocuments
        {
            get
            {
                foreach (EmitterSetDocument emitterSet in this.m_emitterSetList)
                {
                    yield return emitterSet;
                }
            }
        }

        /// <summary>
        /// 要求ドキュメントを列挙
        /// </summary>
        public IEnumerable<IRequireDocument> RequireDocuments
        {
            get
            {
                foreach (IRequireDocument require in this.m_requireDocs)
                {
                    yield return require;
                }
            }
        }

        /// <summary>
        /// 子供のドキュメントを纏めて取得します
        /// </summary>
        public override IEnumerable<IDocument> SubDocuments
        {
            get { return this.EmitterSetDocuments; }
        }

        /// <summary>
        /// 選択されている（アクティブな）ドキュメント
        /// </summary>
        public IDocument ActiveDocument
        {
            get { return m_selectedDocument; }
            set
            {
                IDocument oldDoc = m_selectedDocument;
                // アクティブドキュメントの切り替え
                m_selectedDocument = value;
                // 切り替えを通知します。
                NotifyActiveDocumentChanged(oldDoc);
            }
        }

        /// <summary>
        /// 環境設定データ
        /// </summary>
        public EnvConfigData EnvConfigData
        {
            get { return m_envConfigData; }
        }

        /// <summary>
        /// ツリーノードの更新を有効にします。（DEF:=true）
        /// </summary>
        public bool EnableBuildTreeNodes
        {
            get;
            set;
        }

        #endregion

        #region Utility Functions

        /// <summary>
        /// Reset the project document to the default values.
        /// </summary>
        public void Reset()
        {
            // All other documents had already been taken care of,
            // just remove the old preview document and create a new one.
            if ( m_previewDocument!=null )
            {
                m_previewDocument.NotifyRemovedFromParent();

                m_envConfigData.PreviewConfigData.Initialize();

                m_previewDocument = new PreviewDocument( this, m_envConfigData.PreviewConfigData );

                m_previewDocument.NotifyAddedToParent();

                DocumentManager.ClearDataModifyFlag( this.DataScrPath );
            }

            BuildTreeNodes();
        }

        /// <summary>
        /// Find the index to the given emitter set document.
        /// </summary>
        /// <param name="doc">The emitter set document to find.</param>
        /// <returns>The index to the given emitter set document.</returns>
        public int FindEmitterSetDocumentIndex( IEmitterSetDocument doc )
        {
            return m_emitterSetList.IndexOf( doc );
        }

        /// <summary>
        /// Move the given emitter set document to the location the given index points to.
        /// </summary>
        /// <param name="doc">The emitter set document to move.</param>
        /// <param name="iIndex">The index to move to.</param>
        /// <returns>True on success.</returns>
        public bool MoveEmitterSetDocument( IEmitterSetDocument doc,
                                            int iIndex )
        {
            int iOrigIndex = m_emitterSetList.IndexOf( doc );
            if ( iOrigIndex<0 || iOrigIndex>=m_emitterSetList.Count )
                return false;

            if ( iOrigIndex==iIndex )
                return true;

            if ( iIndex<0 || iIndex>=m_emitterSetList.Count )
            {
                // Add the document to the end of the list.
                m_emitterSetList.Add( doc );
            }
            else
            {
                if ( iIndex<iOrigIndex )
                    ++iOrigIndex;

                // Insert the document back to the desired location.
                m_emitterSetList.Insert( iIndex, doc );
            }

            // Remove the document from the original location.
            m_emitterSetList.RemoveAt( iOrigIndex );

            return true;
        }

        /// <summary>
        /// find child document by name
        /// </summary>
        /// <param name="name">name of the child document</param>
        /// <returns></returns>
        public override IDocument FindChild(string name)
        {
            return FindByName(name);
        }

        /// <summary>
        /// EmitterSetDocumentを追加します
        /// </summary>
        /// <param name="index">挿入する位置のインデックスです</param>
        /// <param name="document">追加するEmitterSetDocumentです</param>
        public void AddEmitterSetDocument(int index, IEmitterSetDocument document)
        {
            if (index >= 0 && this.m_emitterSetList.Count > index)
            {
                this.m_emitterSetList.Insert(index, document);
            }
            else
            {
                this.m_emitterSetList.Add(document);
            }

            document.NotifyAddedToParent();
        }

        /// <summary>
        /// EmitterSetDocumentを削除します
        /// </summary>
        /// <param name="document">削除するEmitterSetDocumentです</param>
        /// <returns>挿入されていたインデックスを返します(=-1..存在しないとき)</returns>
        public int RemoveEmitterSetDocument(IEmitterSetDocument document)
        {
            // リストから削除します
            int index = this.m_emitterSetList.IndexOf(document);
            if (index >= 0)
            {
                this.m_emitterSetList.RemoveAt(index);

                // Notify document removed
                document.NotifyRemovedFromParent();
            }
            return index;
        }

        /// <summary>
        /// RequireDocumentを削除します。
        /// </summary>
        /// <param name="document"></param>
        public int RemoveRequireDocument(IRequireDocument document)
        {
            // リストから削除します
            int index = this.m_requireDocs.IndexOf(document);
            if (index >= 0)
            {
                this.m_requireDocs.RemoveAt(index);

                // Notify document removed
                document.NotifyRemovedFromParent();
            }
            return index;
        }

        /// <summary>
        ///
        /// </summary>
        /// <param name="document"></param>
        public IEmitterSetDocument GetEmitterSetDocumentByIndex(int index)
        {
            if (index >= 0 && index < this.Count)
                return this.m_emitterSetList[index];

            return null;
        }

        /// <summary>
        /// エフェクトドキュメントを既に持っているかチェックします
        /// </summary>
        /// <param name="findItem">検査するエフェクトです</param>
        /// <returns>=true..持っている /=false..持っていない</returns>
        public bool Contains(IEmitterSetDocument findItem)
        {
            foreach (IEmitterSetDocument effect in this.EmitterSetDocuments)
            {
                if (effect == findItem)
                {
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        /// エフェクトドキュメントを検索します（名前）
        /// </summary>
        /// <param name="effectName">エフェクト名です</param>
        /// <returns>見つかったエフェクトドキュメントです</returns>
        public EmitterSetDocument FindByName(string effectName)
        {
            foreach (EmitterSetDocument effect in this.EmitterSetDocuments)
            {
                if (effect.Name == effectName)
                {
                    return effect;
                }
            }
            return null;
        }


        /// <summary>
        /// エフェクトドキュメントを検索します（パス）
        /// </summary>
        /// <param name="filePath">エフェクトパス</param>
        /// <returns>見つかったエフェクトドキュメントです</returns>
        public EmitterSetDocument FindByPath(string filePath)
        {
            if (String.IsNullOrEmpty(filePath))
            {
                return null;
            }
            string fullPath = Path.GetFullPath(filePath);
            foreach (EmitterSetDocument effect in this.EmitterSetDocuments)
            {
                if (Path.GetFullPath(effect.FilePath) == fullPath)
                {
                    return effect;
                }
            }
            return null;
        }


        /// <summary>
        /// Notify this document that it was added to its parent.
        /// </summary>
        public override void NotifyAddedToParent()
        {
            base.NotifyAddedToParent();

            DocumentManager.RegisterDocumentPath( this, this.DataScrPath );

            if ( m_previewDocument!=null )
                m_previewDocument.NotifyAddedToParent();

            foreach ( EmitterSetDocument emitterSetDoc in this.EmitterSetDocuments )
            {
                // Notify the document that it has been added
                if ( emitterSetDoc!=null )
                    emitterSetDoc.NotifyAddedToParent();
            }
        }


        /// <summary>
        /// Notify this document that it was removed from its parent.
        /// </summary>
        public override void NotifyRemovedFromParent()
        {
            foreach ( EmitterSetDocument emitterSetDoc in this.EmitterSetDocuments )
            {
                // Notify the document that it's going to been removed
                if ( emitterSetDoc!=null )
                    emitterSetDoc.NotifyRemovedFromParent();
            }

            if ( m_previewDocument!=null )
                m_previewDocument.NotifyRemovedFromParent();

            DocumentManager.UnregisterDocumentPath( this, this.DataScrPath );

            base.NotifyRemovedFromParent();
        }


        /// <summary>
        /// 要求項目を追加します。
        /// </summary>
        public void PushRequireItem(RequireDocData item)
        {
            lock (m_queueRequire)
            {
                m_queueRequire.Enqueue(item);
            }
        }


        /// <summary>
        /// Check if the given document can be a child of this document.
        /// E.q. Emitter can be emitter set's child, but cannot be a child
        /// of an effect project document.
        /// </summary>
        /// <param name="doc">The child document.</param>
        /// <returns>True if the given document is a valid child document type.</returns>
        public override bool CanAddAsChildDoc( IDocument doc )
        {
            if ( doc is PreviewDocument )
                return true;
            else if ( doc is WorkSpaceDocument )
                return true;
            else if ( doc is InitSettingsDocument )
                return true;

            return false;
        }

        #endregion

        #region Updates

        /// <summary>
        /// プロジェクトウィンドウ更新
        /// </summary>
        public void UpdateTreeNode()
        {
        }

        /// <summary>
        /// プロジェクトウィンドウ用のツリー更新
        /// </summary>
        public void BuildTreeNodes()
        {
            if (this.EnableBuildTreeNodes == false)
            {
                // 更新を無効にします。
                return;
            }


            // Tree生成
            this.Nodes.Clear();

            // add emitter set nodes
            TreeNode rootEffectNode = CreateTreeNode(m_workSpaceDoc);
            this.Nodes.Add(rootEffectNode);
            foreach (EmitterSetDocument emitterSet in this.EmitterSetDocuments)
            {
                var nodeEmitterSet = CreateTreeNode(emitterSet);
                rootEffectNode.Nodes.Add(nodeEmitterSet);

                // add emitter nodes
                nodeEmitterSet.Nodes.Clear();
                foreach (EmitterDocument emitter in emitterSet.EmitterDocuments)
                {
                    var nodeEmitter = CreateTreeNode(emitter);
                    nodeEmitterSet.Nodes.Add(nodeEmitter);

                    // Child/Field/ゆらぎ
                    if (emitter.EmitterType == EmitterType.Complex)
                    {
                        if (emitter.FieldDocument != null)
                        {
                            nodeEmitter.Nodes.Add(CreateTreeNode(emitter.FieldDocument));
                        }
                        if (emitter.FluctuationDocument != null)
                        {
                            nodeEmitter.Nodes.Add(CreateTreeNode(emitter.FluctuationDocument));
                        }
                        if (emitter.ChildDocument != null)
                        {
                            nodeEmitter.Nodes.Add(CreateTreeNode(emitter.ChildDocument));
                        }
                    }

                }

                // Add game settings document node.
                foreach (GameSettingsDocument gameSettingsDoc in emitterSet.GameSettingsDocuments)
                {
                    nodeEmitterSet.Nodes.Add(CreateTreeNode(gameSettingsDoc));
                }
            }

            // Create tree node for preview document and its children
            TreeNode previewDocNode = null;
            if (m_previewDocument != null)
            {
                // Tree node for preview document
                previewDocNode = CreateTreeNode(m_previewDocument);

                // Add the tree node
                this.Nodes.Add(previewDocNode);

                // Loop through the children documents and create the tree nodes
                foreach (PreviewModelDocument childDoc in m_previewDocument.PreviewModelDocuments)
                {
                    // Tree node for preview model document
                    TreeNode childDocNode = CreateTreeNode(childDoc);
                    if ( childDoc.IsModelNameReceived==false )
                        childDocNode.ForeColor = System.Drawing.Color.Red;

                    // Add the tree node
                    previewDocNode.Nodes.Add(childDocNode);
                }
            }

            // RequireNode生成
            BuildRequireNodes(rootEffectNode, previewDocNode);

            this.NotifyEmitterSetListChanged();
        }

        private TreeNode CreateTreeNode(IDocument document)
        {
            var node = new TreeNode(document.Name);
            node.ImageKey = document.ImageKey;
            node.SelectedImageKey = document.ObjectID.ToString();
            node.Tag = document;
            node.ForeColor = System.Drawing.Color.Black;

            return node;
        }

        private void BuildRequireNodes(TreeNode rootEffectNode, TreeNode previewDocNode)
        {
            // ドキュメント生成
            var list = new List<RequireDocData>();
            lock (m_queueRequire)
            {
                while (m_queueRequire.Count > 0)
                {
                    list.Add(m_queueRequire.Dequeue());
                }
            }
            foreach (var item in list)
            {
                var foundRequire = m_requireDocs.Find(
                    delegate(IRequireDocument require)
                    {
                        return require.DataModel.Name.Equals(item.Name);
                    }
                );
                var foundEset = m_emitterSetList.Find(
                    delegate(IEmitterSetDocument eset)
                    {
                        return eset.Name.Equals(item.Name);
                    }
                );
                if (foundRequire == null && foundEset == null)
                {
                    m_requireDocs.Add(new RequireDocument(this, item));
                }
            }

            // ノード生成
            foreach (var node in m_requireDocs)
            {
                if (rootEffectNode != null && node.DataModel.BinaryType == NWCore.Protocols.BinaryDataType.Ptcl)
                {
                    // エフェクトノードに追加
                    rootEffectNode.Nodes.Add(CreateTreeNode(node));
                }
                else if (previewDocNode != null && node.DataModel.BinaryType == NWCore.Protocols.BinaryDataType.Model)
                {
                    // ビューアノードに追加
                    previewDocNode.Nodes.Add(CreateTreeNode(node));
                }
                else
                {
                    // ルートに追加
                    this.Nodes.Add(CreateTreeNode(node));
                }
            }
        }

        #endregion

        #region Events

        /// <summary>
        /// Event triggered when effect list of the project has changed.
        /// </summary>
        public event EventHandler EmitterSetListChanged = null;


        /// <summary>
        /// Notify effect list changed.
        /// </summary>
        public void NotifyEmitterSetListChanged()
        {
            // Trigger event.
            if (EmitterSetListChanged != null)
            {
                EmitterSetListChanged(this, new EventArgs());
            }
        }

        /// <summary>
        /// Event triggered when assets availability have changed.
        /// </summary>
        public event EventHandler AssetAvailabilityChanged = null;

        /// <summary>
        /// notify the availability of some assets have changed.
        /// </summary>
        /// <param name="effect">the EmitterSet document whose assets changed.</param>
        public void NotifyEmitterSetAssetAvailabilityChanged(EmitterSetDocument emitterSet)
        {
            // Trigger event.
            if (emitterSet != null && AssetAvailabilityChanged != null)
            {
                AssetAvailabilityChanged(emitterSet, new EventArgs());
            }
        }

        /// <summary>
        /// Event triggered when effect name has been changed.
        /// </summary>
        public event EventHandler EffectNameChanged = null;

        /// <summary>
        /// notify effect name had been changed.
        /// </summary>
        /// <param name="effect">the EmitterSet document whose name is changed.</param>
        public void NotifyEmitterSetNameChanged(EmitterSetDocument emitterSet)
        {
            // Trigger event.
            if (emitterSet != null && EffectNameChanged != null)
            {
                EffectNameChanged(emitterSet, new EventArgs());
            }
        }

        /// <summary>
        /// アクティブドキュメントが変更されたときにコールされます。
        /// </summary>
        public event DocumentChangedEventHandler ActiveDocumentChanged;

        /// <summary>
        /// アクティブドキュメントが変更されたことを通知します。
        /// </summary>
        private void NotifyActiveDocumentChanged(IDocument oldDocument)
        {
            if (this.ActiveDocumentChanged != null)
            {
                // ログ
                if (this.ActiveDocument != null)
                {
                    DebugConsole.WriteLine("selected document:[{0}]", this.ActiveDocument.Name);
                }
                else
                {
                    DebugConsole.WriteLine("selected document:[null]");
                }

                // イベント発行
                this.ActiveDocumentChanged(
                    this, new DocumentChangedEventArgs(oldDocument, this.ActiveDocument));
            }
        }

        /// <summary>
        /// notify effectset lisd changed.
        /// </summary>
        private void emitterSet_ListChanged(object sender, EventArgs e)
        {
            EmitterDocument emitterDoc = ProjectManager.ActiveDocument as EmitterDocument;
            if ( emitterDoc!=null &&
                 emitterDoc.IsRemoved==false )
            {
                // ノード変更をビューアへ通知
                ////Viewer.Message.SendUtility.SendEmitterSetBinary(emitterDoc.EmitterSetDocument);
                return;
            }

            GameSettingsDocument previewDoc = ProjectManager.ActiveDocument as GameSettingsDocument;
            if (previewDoc != null &&
                previewDoc.IsRemoved == false)
            {
                // ノード変更をビューアへ通知
                ////Viewer.Message.SendUtility.SendEmitterSetBinary(previewDoc.EmitterSetDocument);
            }

        }

        #endregion

        #region アトリビュート

        /// <summary>
        /// ObjectID
        /// </summary>
        public override GuiObjectID ObjectID
        {
            get { return GuiObjectID.EffectProject; }
        }

        /// <summary>
        /// TreeNode
        /// </summary>
        public IList<TreeNode> Nodes
        {
            get { return this.m_treeNode; }
        }

        #endregion

        #region シリアライズ関係

        /// <summary>
        /// ストリームにドキュメントの書き込みします。
        /// </summary>
        /// <param name="stream">保存先のストリーム。</param>
        /// <param name="bClearModifyFlag">
        /// Trueを指定したら、ドキュメントの変更のフラグを削除します。
        /// </param>
        /// <param name="bNormalizePath">
        /// Trueを指定場合、ドキュメントは記載するファイルパスを相対パスに転換する。
        /// </param>
        /// <returns>成功するとtrueを戻します。</returns>
        public override bool SaveDocumentToStream( Stream stream,
                                                   bool bClearModifyFlag = true,
                                                   bool bNormalizePath = true )
        {
            // Create serializer
            NWCore.Serializer.EnvConfigDataXml serializer =
                new NWCore.Serializer.EnvConfigDataXml( this.EnvConfigData );

            #region User functions of the preview model documents

            for ( int i=0;i<m_previewDocument.PreviewModelDocuments.Length;++i )
            {
                PreviewModelDocument doc =
                    m_previewDocument.PreviewModelDocuments[i] as PreviewModelDocument;
                if ( doc==null )
                    continue;

                PreviewModelData data =
                    serializer.ConfigData.PreviewConfigData.GetModelData(i);
                if ( data==null )
                    continue;

                data.UserFunctionList.Clear();

                IList<IDataModelSerializeList> dataModelList =
                    doc.DataModelProxy.GetAllDataModelsForSerialize();
                foreach ( DataModelSerializeList dataModel in dataModelList )
                {
                    UserFunctionXML userFunc =
                        new UserFunctionXML( dataModel.Name, dataModel.ID );

                    // Fields in the user function
                    foreach ( DataModelSerializeItem item in dataModel )
                    {
                        uint iNameCRC = TheApp.CRC32Helper.ComputeCRC32Str( item.Name );
                        userFunc.Add( item.Name, iNameCRC, item.Value );
                    }

                    data.UserFunctionList.Add( userFunc );

                    userFunc.PreSerialize();
                }
            }

            #endregion

            #region Save emitter set configs & game settings documents

            foreach ( IEmitterSetDocument doc in m_emitterSetList )
            {
                EmitterSetDocument emitterSetDoc = doc as EmitterSetDocument;
                if ( emitterSetDoc==null )
                    continue;

                // Add the emitter set configs.
                serializer.EmitterSetConfigs.Add(new EnvConfigDataXml.EmitterSetConfigXml(doc));

                foreach ( GameSettingsDocument child in emitterSetDoc.GameSettingsDocuments )
                {
                    if ( child==null )
                        continue;

                    GameConfigDataXml prevData;
                    child.SerializeDocument( out prevData, true );

                    if ( bClearModifyFlag==true )
                        child.ClearModifiedFlag( true );

                    serializer.PrevDataList.Add( prevData );
                }
            }

            #endregion

            // serialize into xml
            XmlUtility.SerializeToStream( serializer, ref stream );

            return true;
        }

        #endregion

        #region Drag & drop

        /// <summary>
        /// Move the specified child document to the location
        /// before another child document.
        /// </summary>
        /// <param name="doc">The child document to move.</param>
        /// <param name="beforeDoc">
        /// The child document which "doc" will be moved before,
        /// null to move "doc" to the last child document.
        /// </param>
        /// <returns>The document after moved.</returns>
        public override IDocument MoveChildDocument( IDocument doc,
                                                     IDocument beforeDoc )
        {
            // Same location?
            if ( doc==beforeDoc )
                return doc;

            if ( doc is EmitterSetDocument )
            {
                EmitterSetDocument child = doc as EmitterSetDocument;
                if ( child.Project!=this )
                    return null;

                int iOrigIndex = m_emitterSetList.IndexOf( child );
                if ( iOrigIndex<0 || iOrigIndex>=m_emitterSetList.Count )
                    return null;

                int iDestIndex = -1;
                if ( beforeDoc!=null )
                {
                    EmitterSetDocument beforeChild = beforeDoc as EmitterSetDocument;
                    if ( beforeChild==null || beforeChild.Project!=this )
                        return null;

                    iDestIndex = m_emitterSetList.IndexOf( beforeChild );
                    if ( iDestIndex<0 || iDestIndex>=m_emitterSetList.Count )
                        return null;
                    else if ( iOrigIndex==(iDestIndex-1) )
                        return doc;
                }
                else if ( iOrigIndex==m_emitterSetList.Count-1 )
                {
                    // Same location
                    return doc;
                }
            }
            else
            {
                return null;
            }

            this.BuildTreeNodes();

            return doc;
        }

        #endregion
    }
}

